/************************************************************************/
/*                                                                      */
/*                      RC5 Remote Receiver                             */
/*                                                                      */
/*              Author: Peter Dannegger                                 */
/*                      danni@specs.de                                  */
/*                                                                      */
/************************************************************************/

/*
Modified by Malte Marwedel

VUSB need to get interrupts within 25cycles, so nested interrupts must be
allowed.
VUSB interrupt routine will use ~100µs. So this hopefully does not conflict with
the interrupt every 120µs.
*/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>

#include "rc5decoder.h"

#define RC5TIME 	1.778e-3		// 1.778msec
//values with original 0.4, 0.8, 1.2 factors:
//512 cycle: 22, 44, 67
//2048 cycle: 6, 11, 17
//3072 cycle: 4, 7, 11: did work 50% of the times
//4096 cycle: 3, 6, 8: did not work most of the times
#define PULSE_MIN	(uchar)(F_CPU / 1496 * RC5TIME * 0.3 + 0.5)
#define PULSE_1_2	(uchar)(F_CPU / 1496 * RC5TIME * 0.7 + 0.5)
#define PULSE_MAX	(uchar)(F_CPU / 1496 * RC5TIME * 1.3 + 0.5)


uchar	rc5_bit;				// bit value
uchar	rc5_time;				// count bit time
uint	rc5_tmp;				// shift bits in

volatile uint16_t rc5_data;

ISR(SIG_OUTPUT_COMPARE0A, ISR_NOBLOCK) {
  uint tmp = rc5_tmp;				// for faster access
  if( ++rc5_time > PULSE_MAX ){			// count pulse time
    if( !(tmp & 0x4000) && tmp & 0x2000 )	{// only if 14 bits received
       rc5_data = tmp;           //store  recived value
		}
    tmp = 0;
  }

  if( (rc5_bit ^ xRC5_IN) & 1<<xRC5 ){		// change detect
    rc5_bit = ~rc5_bit;				// 0x00 -> 0xFF -> 0x00

    if( rc5_time < PULSE_MIN )			// too short
      tmp = 0;

    if( !tmp || rc5_time > PULSE_1_2 ){		// start or long pulse time
      if( !(tmp & 0x4000) )			// not too many bits
        tmp <<= 1;				// shift
      if( !(rc5_bit & 1<<xRC5) )		// inverted bit
        tmp |= 1;				// insert new bit
      rc5_time = 0;				// count next pulse time
    }
  }
  rc5_tmp = tmp;
}

void rc5Init(void) {
	TCCR0A = (1<<WGM01);  //CTC mode
	TCNT0 = 0;
	OCR0A = 187; //8*187 -> 1496
	TCCR0B = (1<<CS01);   //diviede by 8 -> 1,5Mhz
	TIMSK0 |= 1<<OCIE0A;  //enable timer0 compare interrupt
}

uint16_t rc5getdata(void) {
	uint16_t val;
	cli();
	val = rc5_data;
	rc5_data = 0;
	sei();
	return val;
}
