/* Infplay
  (c) 2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <inttypes.h>
#include <stdio.h>
#include <string.h>

#include "writemp3.h"
#include "stringhelper.h"

//#define WRITEMP3_DEBUG

#ifdef TEST_PC

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "pctests/avrmapper.h"

#define RECORDDIR "records"


uint32_t NutGetTickCount(void) {
	return rand();
}

void NutThreadYield(void) {
}

int main(int argc, char ** argv) {
	printf("Writes an empty mp3 with meta information into the directory '"RECORDDIR"'.\n");
	printf("Usage: [TITLE] [ARTIST] [ALBUM] [SOURCE]\n");
	char * title = NULL, * artist = NULL, * album = NULL, * source = NULL;
	if (argc > 1) {
		title = argv[1];
		srand(*((uint32_t *)argv[1]));
	}
	if (argc > 2) {
		artist = argv[2];
	}
	if (argc > 3) {
		album = argv[3];
	}
	if (argc > 4) {
		source = argv[4];
	}
	FILE * f = writemp3_start(title, artist, album, source);
	if (f) {
		printf("Writing successful\n");
		fclose(f);
	} else {
		printf("Writing failed\n");
	}
	return 0;
}


#else

#include <fs/phatvol.h>
#include <sys/stat.h>
#include <compiler.h>
#include <sys/timer.h>


#include "stringhelper.h"
#include "infplay.h"

#define RECORDDIR FSDEV_ROOT"records"

#endif

/* Maximum filename length:
  10 char title or source
  3 char " - "
  10 char artist or 5 char number
  4 char counting number
  4 char ".mp3"
  1 char null termination
  sum: 32 (40 just to be sure)

  Maximum with pathname
  strlen(RECORDDIR)+ "/" + maximum filename length
  sum: 47
*/

#define WRITEMP3_MAXNAME 40
#define WRITEMP3_MAXPATH (strlen(RECORDDIR) + 1 + WRITEMP3_MAXNAME)

//how many free filenames should be tested
#define WRITEMP3_MAXFSEEK 1000

static void writemp3_appendname(char * dest, char * source, uint8_t * idx) {
	uint8_t j;
	for (j = 0; j < 10; j++) {
		if (source[j] == '\0') {
			break;
		}
		if (((source[j] >= 'a') && (source[j] <= 'z')) ||
		    ((source[j] >= 'A') && (source[j] <= 'Z'))) {
			dest[*idx] = source[j];
			(*idx)++;
		}
	}
}

static void writemp3_frame(FILE * f, prog_char * tag, char * data) {
	uint8_t buff[11];
	memcpy_P(buff, tag, 4);
	uint32_t len = strlen(data)+2;
	buff[4] = (uint8_t)(len >> 24);
	buff[5] = (uint8_t)(len >> 16);
	buff[6] = (uint8_t)len >> 8;
	buff[7] = (uint8_t)len;
	buff[8] = 0;
	buff[9] = 0;
	buff[10] = 0;
	fwrite(buff, 1, 11, f);
	fwrite(data, 1, strlen(data)+1, f);
}

static prog_char writemp3_header[6] = {'I', 'D', '3', 3, 0, 0};

FILE * writemp3_start(char * title, char * artist, char * album, char * datasource) {
	//create directory if not present
	struct stat st;
	if (stat(RECORDDIR, &st) == 0) {
		if (!S_ISDIR(st.st_mode)) {
#ifdef WRITEMP3_DEBUG
			puts_P(PSTR("Error: non directory"));
#endif
			return NULL;
		}
	} else { //not there, try to create
    if (mkdir(RECORDDIR, 0777)) {
#ifdef WRITEMP3_DEBUG
			puts_P(PSTR("Error: create dir failed"));
#endif
			return NULL;
		}
	}
	//create filename
	char filepathname[WRITEMP3_MAXPATH];
	char filename[WRITEMP3_MAXNAME];
	uint8_t idx = 0;
	if (artist) {
		writemp3_appendname(filename, artist, &idx);
	} else if (datasource) {
		writemp3_appendname(filename, datasource, &idx);
	}
	if (idx > 0) {
		filename[idx++] = ' ';
		filename[idx++] = '-';
		filename[idx++] = ' ';
	}
	if (title) {
		writemp3_appendname(filename, title, &idx);
	} else { //add a number if no title name was given
		sprintf_P(filename+idx, PSTR("%u"), (uint16_t)NutGetTickCount());
		idx = strlen(filename); //the length may be unknown otherwise
	}
	//append a counting number if the filename exists
	uint16_t cnt;
	for (cnt = 0; cnt < WRITEMP3_MAXFSEEK; cnt++) {
		uint8_t idx2 = idx;
		if (cnt != 0) {
			sprintf_P(filename+idx2, PSTR("%u"), cnt);
			idx2 = s_strlen(filename);
		}
		filename[idx2++] = '.';
		filename[idx2++] = 'm';
		filename[idx2++] = 'p';
		filename[idx2++] = '3';
		filename[idx2++] = '\0';
		strcpy(filepathname, RECORDDIR);
		path_append(filepathname, filename, WRITEMP3_MAXPATH, 0);
		struct stat stf;
		if (stat(filepathname, &stf)) {
			break; //file name not already used
		}
		NutThreadYield();
#ifdef WRITEMP3_DEBUG
		printf_P(PSTR("Filename '%s' already in use\n"), filepathname);
#endif
	}
	if (cnt >= WRITEMP3_MAXFSEEK) {
		return NULL;
	}
#ifdef WRITEMP3_DEBUG
	printf_P(PSTR("File name will be '%s'\n"), filepathname);
#endif
	//create file
	FILE * f = fopen(filepathname, "wb");
	if (f == NULL) {
#ifdef WRITEMP3_DEBUG
		puts_P(PSTR("Create file failed"));
#endif
		return NULL;
	}
	//append mp3 header v2.3
	if ((title) || (artist) || (datasource)) {
		uint16_t headerlen = 0;
		/*a frame is 10 bytes in size + 1 byte determining the encoding
			+ length of the string + 1 byte for null termination.
		*/
		if (title) {
			headerlen += strlen(title)+11+1;
		}
		if (artist) {
			headerlen += strlen(artist)+11+1;
		}
		if (datasource) {
			headerlen += strlen(datasource)+11+1;
		}
		uint8_t buff[10]; //the id3 header is always 10 bytes in size
		memcpy_P(buff, writemp3_header, 6);
		uint8_t i;
		uint16_t tmplen = headerlen;
		for (i = 0; i < 4; i++) {
			buff[9-i] = tmplen & 0x7F;
			tmplen >>= 7;
		}
		fwrite(buff, sizeof(uint8_t), 10, f);
		if (title) {
			writemp3_frame(f, PSTR("TIT2"), title);
		}
		if (artist) {
			writemp3_frame(f, PSTR("TPE1"), artist);
		}
		if (album) {
			writemp3_frame(f, PSTR("TALB"), album);
		}
		if (datasource) {
			writemp3_frame(f, PSTR("TRSN"), datasource);
		}
	}
	return f;
}

