/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* s_: Secure string functions.
Can handle if there is accidentally a NULL pointer.
And its guranteed, that no NULL pointer will be returned.
 */


#include <string.h>
#include <stdio.h>

#ifdef TEST_PC

#include "pctests/avrmapper.h"

#else

#include "memmapper.h"
#include "infplay.h"
#include "error.h"

#endif

size_t s_strlen(const char *s) {
	if (s == NULL) {
		puts_P(PSTR("NULL for strlen"));
		ThreadTerminate();
	}
	return strlen(s);
}

char * s_strndup(const char *s, size_t n) {
	if (s == NULL) {
		return s_cmalloc(sizeof(char));
	}
	size_t len = s_strlen(s);
	if (len > n) {
		len = n;
	}
	char * newbuf = s_cmalloc((len+1)*sizeof(char));
	//the \0 is generated by calloc
	memcpy(newbuf, s, len);
	return newbuf;
}

char * s_strdup(const char *s) {
	size_t len;
	if (s == NULL) {
		len = 0;
	} else {
		len = s_strlen(s);
	}
	return s_strndup(s, len);
}

char * extractfilename(char * pathname) {
	char * filename = NULL;
	if (pathname != NULL) {
		filename = strrchr(pathname, '/');
		if (filename == NULL) {
			filename = pathname;
		} else
			filename++; //go after the slash
	}
	return filename;
}

char * extractfilepathdup(char * pathname) {
	char * path = s_strdup(pathname);
	char * en = strrchr(path, '/');
	if (en) {
		*en = '\0';
	}
	return path;
}

void path_append(char * buff, char * toappend, size_t buffsize, uint8_t isdir) {
	size_t buffused = s_strlen(buff);
	size_t appendsize = s_strlen(toappend);
	if ((buffused > 0) && (buffused < buffsize-1)) {
		if (buff[buffused-1] != '/') {
			buff[buffused] = '/';
			buffused++;
			buff[buffused] = '\0';
		}
	}
	if ((buffsize - buffused) > (appendsize)) {
		strcpy(buff+buffused, toappend);
		buffused += appendsize;
	} else {
		error_general(10);
	}
	if ((isdir) && (buffused > 0) && (buffused < buffsize-1)) {
		if (buff[buffused-1] != '/') {
			buff[buffused] = '/';
			buffused++;
			buff[buffused] = '\0';
		}
	}
}

uint8_t endswith(char * text, char * ending) {
	uint16_t tlen = s_strlen(text);
	uint16_t elen = s_strlen(ending);
	if (tlen < elen)
		return 0;
	if (!strcmp((text)+tlen-elen, ending))
		return 1;
	return 0;
}

uint16_t utf8toiso8859_subs(char * text, uint16_t length, uint16_t rpoffs) {
	uint16_t rp = rpoffs, wp = 0;
	while (rp < length) {
		uint8_t t1 = text[rp];
		if ((t1 >= 0x20) && (t1 < 0x80)) { //filter all normal ascii chars
			text[wp] = t1;
			wp++;
		} else if (rp+1 < length) { //check if ISO-8859-1 subset
			rp++;
			uint8_t t2 = text[rp];
			/* useful chars: t1=0xC2: t2 = 0xA0 ... 0xBF
			                 t1=0xC3: t2 = 0x80 ... 0xBF */
			if ((t1 == 0xC2) && ((t2 & 0xE0) == 0xA0)) {
				text[wp] = t2;
				wp++;
			} else
			if ((t1 == 0xC3) && ((t2 & 0xC0) == 0x80)) { //2 upper bits fixed
				text[wp] = t2+0x40;
				wp++;
			}
		}
		rp++;
	}
	return wp;
}

void utf8toiso8859(char * text) {
	uint16_t en = utf8toiso8859_subs(text, s_strlen(text), 0);
	text[en] = '\0';
}

void * iso8859toutf8(char * text) {
	//count chars which are not iso
	uint16_t chars = 0;
	uint16_t rp = 0;
	while (text[rp] != '\0') {
		rp++;
		chars++;
		if ((uint8_t)text[rp] > 127) {
			chars++;
		}
	}
	//generate new string
	char * utf = s_cmalloc(sizeof(char)*(chars+1));
	//convert
	rp = 0;
	uint16_t wp = 0;
	while (text[rp] != '\0') {
		uint8_t in = text[rp];
		if (in < 128) {
			utf[wp] = in;
			wp++;
		} else {
			if (in < 0xC0) {
				utf[wp] = 0xC2;
				utf[wp+1] = in;
			} else {
				utf[wp] = 0xC3;
				utf[wp+1] = in-0x40;
			}
			wp += 2;
		}
		rp++;
	}
	return utf;
}

static uint8_t mustencoded(char c) {
	if (((c >= '0') && (c <= '9')) ||
	    ((c >= 'a') && (c <= 'z')) ||
	    ((c >= 'A') && (c <= 'Z'))) {
		return 0;
	}
	return 1;
}

char hexadecimal(uint8_t val) {
	char c = (val & 0x0F) + '0';
	if (c > '9') {
		c += 'A'-'9'-1;
	}
	return c;
}

char * urlencode(char * text) {
	uint16_t sourcelen = s_strlen(text);
	uint16_t targetlen = sourcelen;
	uint16_t i;
	//count lenght of target string
	for (i = 0; i < sourcelen; i++) {
		if (mustencoded(text[i])) {
			targetlen += 2;
		}
	}
	char * encodedtext = s_cmalloc((targetlen+1)*sizeof(char));
	//encode
	int j = 0;
	for (i = 0; i < sourcelen; i++) {
		char c = text[i];
		if (mustencoded(c)) {
			encodedtext[j++] = '%';
			encodedtext[j++] = hexadecimal(c >> 4);
			encodedtext[j++] = hexadecimal(c);
		} else {
			encodedtext[j++] = c;
		}
	}
	return encodedtext;
}

uint8_t isutf8(char * text) {
	while (*text != '\0') {
		if (((uint8_t)(*text) == 0xC2) || ((uint8_t)(*text) == 0xC3)) {
			return 1;
		}
		text++;
	}
	return 0;
}

//do not get confused with multiple main functions
#ifdef TEST_PC_STRINGHELPER

#include <stdlib.h>

int main(int argc, char ** argv) {
	if (argc == 3) {
		if (strcmp(argv[1], "e") == 0) {
			char * encoded = urlencode(argv[2]);
			printf("Encoded url: '%s'\n", encoded);
			return 0;
		}
		if (strcmp(argv[1], "i") == 0) {
			char * str = s_strdup(argv[2]);
			utf8toiso8859(str);
			printf("Utf to iso: '%s'\n", str);
			return 0;
		}
		if (strcmp(argv[1], "u") == 0) {
			char * str = s_strdup(argv[2]);
			char * res = iso8859toutf8(str);
			printf("Iso to utf-8: '%s'\n", res);
			return 0;
		}
		if (strcmp(argv[1], "h") == 0) {
			int res = isutf8(argv[2]);
			if (res) {
				printf("Heuristic guess: utf-8\n");
			} else {
				printf("Heuristic guess: iso8859\n");
			}
			return 0;
		}
	}
	if (argc == 4) {
		if (strcmp(argv[1], "c") == 0) {
			if (endswith(argv[2], argv[3])) {
				printf("Ending matches\n");
			} else
				printf("Ending does not match\n");
			return 0;
		}
	}
	if (argc == 5) {
		if (strcmp(argv[1], "a") == 0) {
			int len = strlen(argv[2])+strlen(argv[2])+10;
			char * target = s_cmalloc(len*sizeof(char));
			strcpy(target, argv[2]);
			path_append(target, argv[3], len, atoi(argv[4]));
			printf("Appended: %s\n", target);
			return 0;
		}
	}
	printf("Usage:\ne TEXT: Encode text\ni TEXT: Convert utf-8 to iso8859\n");
	printf("u TEXT: Convert iso8859 to utf-8\n");
	printf("h TEXT: Heuristic checking if the text is iso8859 or utf-8\n");
	printf("c TEXT ENDING: Test if text ends with the ending\n");
	printf("a PATH APPEND ISDIR(0/1): Append paths together\n");
	return 1;
}

#endif
