/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* This logic handles the mp3 data copying form the network buffer or sd file
	to the mp3 decoder. This is realized as a state machine.
	possible transitions:
	STREAM_STOPPED -> STREAM_PLAY_FILE_START
	STREAM_PLAY_FILE_START -> STREAM_PLAY_FILE
	STREAM_PLAY_FILE -> STREAM_FILE_END
	STREAM_STOP -> STREAM_STOPPED
	STREAM_FILE_END -> STREAM_PLAY_FILE_START
	STREAM_STOPPED -> STREAM_PLAY_NETWORK_START
	STREAM_PLAY_NETWORK_START -> STREAM_PLAY_NETWORK
	STREAM_PLAY_NETWORK -> STREAM_NETWORK_BUFFER_EMPTY
	STREAM_PLAY_LASTFM_START -> STREAM_PLAY_LASTFM_NEXT
	STREAM_PLAY_LASTFM_NEXT -> STREAM_PLAY_LASTFM
	STREAM_PLAY_LASTFM -> STREAM_PLAY_LASTFM_NEXT
	it can always go back to STREAM_STOPPED
	if a transition is not supported or some error occurs, it goes back to STREAM_STOPPED too.
 */

#include <compiler.h>
#include <avr/pgmspace.h>
#include <inttypes.h>
#include <stdio.h>
#include <string.h>
#include <sys/timer.h>
#include <fs/phatvol.h>

#include "statestorage.h"
#include "hardware/vs1002drv.h"
#include "bigbuff.h"
#include "id3.h"
#include "connurl.h"
#include "filemanager.h"
#include "infplay.h"
#include "icy200.h"
#include "error.h"
#include "action.h"
#include "metainfoqueue.h"
#include "writemp3.h"
#include "spi.h"
#include "info.h"
#include "lastfm.h"
#include "stringhelper.h"

#define STREAM_FILSESWTICHBUF 24000
#define STREAM_BLOCKSIZE 4096
#define STREAM_CONNECTION_RETRIES 5
#define STREAM_CONNECTION_RETRY_WAIT 1000
#define STREAM_NETWORK_FIRSTBUFFFILL (BUFF_MAXRAM/2)
#define STREAM_FILE_FIRSTBUFFFILL STREAM_FILSESWTICHBUF
//music bytes to append to the closing record to prevent a missing track ending
#define STREAM_RECORDINGAPPEND 170000L

//minimum free buffer for last fm to allow user next select
#define STREAM_BUFF_MINFREE_NEXT (BUFF_MAXRAM/3)

/*reconnect if the buffer is empty and there were no data within
 the last x seconds */
#define STREAM_TIMEOUT1 15
/*reconnect if there were no data within the last x seconds, regardless how full
  the buffer is */
#define STREAM_TIMEOUT2 50

/*Timeout for lastfm.*/
#define STREAM_TIMEOUT3 15

/*Timeout for lastfm, if the file seems to be finished*/
#define STREAM_TIMEOUT4 3

FILE * stream_file;
uint32_t stream_datalen; //in bytes
uint32_t stream_datastart; //in bytes (data start or copied lastfm data

uint32_t stream_toread;
uint32_t stream_lastdatatime;
char * stream_connectnext; //used for redirects and lastfm playlist

FILE * record_file;

void stream_set_length(uint32_t length) {
	stream_datalen = length;
}

static void stream_record_close(void) {
	buff_copy_tofile(record_file, STREAM_RECORDINGAPPEND);
	puts_P(PSTR("Stop recording"));
	buff_set_dual_rp_mode(0);
	if (record_file) {
		fclose(record_file);
		record_file = NULL;
	}
	stream_state.recording = STREAM_RECORDING_STOP;
	visual_recordbutton_upd();
}

static void stream_record_start(void) {
	puts_P(PSTR("Start recording"));
	char * title = s_strdup(state_id3_get(ID3_TITLE));
	char * artist = s_strdup(state_id3_get(ID3_ARTIST));
	char * datasource = s_strdup(state_id3_get(ID3_SOURCE));
	char * album = s_strdup(state_id3_get(ID3_ALBUM));
	buff_set_dual_rp_mode(1);
	record_file = writemp3_start(title, artist, album, datasource);
	free(title);
	free(artist);
	free(datasource);
}

static void stream_handle_recording(void) {
	if ((stream_state.nextstate == STREAM_PLAY_NETWORK) ||
	    (((stream_state.nextstate == STREAM_PLAY_LASTFM) ||
	      (stream_state.nextstate == STREAM_PLAY_LASTFM_NEXT)) &&
	     (state_lastfm_recordallow_get()))) {
		uint8_t oldstate = stream_state.recording;
		uint8_t newstate = stream_state.recordingnext;
		if (newstate != STREAM_RECORDING_UNCHANGED) {
			if ((oldstate == STREAM_RECORDING_STOP) &&
			    (newstate != STREAM_RECORDING_STOP)) { //open new file
				stream_record_start();
			} else if ((oldstate == STREAM_RECORDING_TITLE) &&
			           (newstate == STREAM_RECORDING_TITLE)) { //close and open file
				stream_record_close();
				stream_record_start();
			} else if ((oldstate == STREAM_RECORDING_TITLE) &&
			           (newstate == STREAM_RECORDING_CONTINUOUS)) { //change to state 2
				stream_state.recording = STREAM_RECORDING_CONTINUOUS;
			} else if ((oldstate != STREAM_RECORDING_STOP) &&
			           (newstate == STREAM_RECORDING_STOP)) { //close file
				stream_record_close();
			}
			if ((record_file) || (newstate == STREAM_RECORDING_STOP)) {
				stream_state.recording = newstate;
			}
			stream_state.recordingnext = STREAM_RECORDING_UNCHANGED;
			visual_recordbutton_upd();
		}
		if (stream_state.recording != STREAM_RECORDING_STOP) {
			buff_copy_tofile_follow(record_file);
			if (STREAM_RECORDING_MINMEMFREE >= spi_mmc_kbfree()) {
				stream_record_close();
			}
		}
	} else {
		if (record_file) {
			stream_record_close();
		}
	}
}

/*
Mode: 0: next, 1: folderplay, 2: random, 3: near random, 4: previous

*/
void stream_dbfileselect(uint8_t mode) {
	static uint8_t mutexlock = 0; //only works with cooperative schedulers
	while(mutexlock) {
		NutThreadYield();
	}
	mutexlock = 1;
	char * source = stream_state.source;
	if (stream_state.sourcetype == STREAM_TYPE_FILE) {
		if (!manager_buildup_busy_get()) {
			uint16_t nextid;
			switch(mode) {
			  case 0: nextid = manager_nextfileid(source); break;
			  case 1: nextid = manager_nextfolderfileid(source); break;
			  case 2: nextid = manager_randomfileid(); break;
			  case 3: nextid = manager_randomfilenearid(source); break;
			  case 4: nextid = manager_prevfileid(source); break;
			  default: nextid = 0;
			}
			char * filename = manager_filenameget(nextid, FSDEV_ROOT);
			if (filename != NULL) {
				state_mp3_fileplay(filename);
				free(filename);
			} else {
				error_general(16);
			}
		} else {
			info_message_P(PSTR("Database busy"));
			/*if database is busy, replay current file
			 (and wait 1,5 seconds to allow the user other actions) */
			NutSleep(1500);
			state_mp3_fileplay(source);
		}
	}
	mutexlock = 0;
}

void stream_cleandecoder(void) {
	mp3dev_softarereset();
	mp3dev_playtimereset();
	state_bassboost_update();
	state_mp3_volume_update();
	meta_queueclear();
}

//calls from icy200...
void stream_error(void) {
	state_mp3_stop();
	error_message_P(PSTR("Invalid data from URL"));
}

//calls from icy200...
void stream_redirect(char * url) {
	printf_P(PSTR("Stream redirect to: '%s'\n"), url);
	if (stream_connectnext != NULL) {
		free(stream_connectnext);
	}
	stream_connectnext = s_strdup(url);
}

uint8_t stream_openfile(void) {
	if (stream_file != NULL) {
		fclose(stream_file);
		stream_file = NULL;
	}
	connurl_terminate(); //just to be sure
	stream_file = fopen(stream_state.source, "rb");
	if (stream_file != NULL) {
		if (BUFF_MAXRAM - buff_free() > STREAM_FILSESWTICHBUF) { //wait until buffer is mostly empty
			buff_init(); //stop current playing, otherwise let old playing end normally
			NutSleep(1); //to let the timer push out his 32 byte micro buffer
			stream_cleandecoder();
		}
		mp3dev_playtimereset();
		meta_queueclear();
		state_mp3_seclen(0); //unknown at the moment
		state_id3_clear();
		state_bassboost_update();
		id3_extract(); //needs an opened stream_file
		//_filelength() only works on fd. and nut/os prevents getting this from a FILE *...
		stream_datastart = ftell(stream_file);
		fseek(stream_file, 0, SEEK_END);
		uint32_t e = ftell(stream_file);
		fseek(stream_file, stream_datastart, SEEK_SET);
		stream_datalen = (e-stream_datastart);
		state_filenameonemptytitle();
		state_pause_off();
		return STREAM_PLAY_FILE;
	}
	error_message_P(PSTR("Open file failed"));
	return STREAM_STOPPED; //could not open file
}

uint8_t stream_copyfiledata(void) {
	static uint8_t reamupd;
	uint32_t avail = buff_free();
	if (avail == BUFF_MAXRAM) {
		mp3dev_pause1(0xFF);
	}
	//wait for at least 24k buffer filling
	if ((BUFF_MAXRAM - avail) > STREAM_FILE_FIRSTBUFFFILL) {
		mp3dev_pause1(0);
	}
	if (avail >= STREAM_BLOCKSIZE) {
		size_t cpd = buff_filecopy(stream_file, STREAM_BLOCKSIZE);
		if (cpd != STREAM_BLOCKSIZE) {
			return STREAM_FILE_END;
		}
	}
	//calc datarate -> guess play length
	if (!reamupd) { //update every 32th time
		uint16_t played = mp3dev_playtimeget();
		uint32_t bplay = ftell(stream_file) - stream_datastart - (BUFF_MAXRAM - buff_free());
		uint16_t rate = 0;
		if (played > 0) {
			rate = (uint32_t)(bplay/played) + 500;
			rate -= rate % 1000; //round to next kb/s
		}
		if ((rate < 40000) && (rate > 0)) {
			//printf("rate:%u\n", rate);
			state_mp3_seclen(stream_datalen/rate);
		}
	}
	reamupd++;
	reamupd &= 0x1F;
	return STREAM_PLAY_FILE;
}

size_t stream_icy_copyfunct(uint8_t * buffer, size_t size) {
	size_t cop = 0;
	uint8_t i = 0;
	while (i < 200) {
		cop += fread(buffer+cop, sizeof(uint8_t), size-cop, conn_stream);
		if (cop < size) {
			NutSleep(10);
			i++;
		} else
			break;
	}
	state_stats_traffic_add(cop);
	return cop;
}

uint8_t stream_copynet(uint8_t lastfm) {
	uint32_t avail = buff_free();
	if (avail == BUFF_MAXRAM) {
		mp3dev_pause1(0xFF);
	}
	//wait for at least ~224k buffer filling
	if ((BUFF_MAXRAM - avail) > STREAM_NETWORK_FIRSTBUFFFILL) {
		mp3dev_pause1(0);
	}
	if (avail >= STREAM_BLOCKSIZE) {
		if (stream_toread == 0) {
			stream_toread = icy200_decode();
			if (stream_connectnext) { //can get set by icy200_decode()
				if (lastfm) {
					return STREAM_PLAY_LASTFM_NEXT;
				} else {
					return STREAM_PLAY_NETWORK_START;
				}
			}
		}
		uint16_t readnum = STREAM_BLOCKSIZE;
		if (stream_toread < readnum) {
			readnum = stream_toread;
		}
		size_t cpd = buff_filecopy(conn_stream, readnum);
		stream_toread -= cpd;
		if (lastfm) {
			stream_datastart += cpd;
		}
		state_stats_traffic_add(cpd);
		//watch if stream gets data
		if (cpd) {
			stream_lastdatatime = NutGetSeconds();
		} else {
			if (!lastfm) {
				if ((stream_lastdatatime+STREAM_TIMEOUT1 < NutGetSeconds()) &&
				    ((avail == BUFF_MAXRAM) || (stream_lastdatatime+STREAM_TIMEOUT2 < NutGetSeconds()))) {
					puts_P(PSTR("Network data timeout. Reconnecting"));
					return STREAM_PLAY_NETWORK_START;
				}
			} else {
				if ((avail > (BUFF_MAXRAM/2)) &&
				    (stream_datastart >= stream_datalen) && (stream_datalen > 0) &&
				    (stream_lastdatatime+STREAM_TIMEOUT4 < NutGetSeconds())) {
					return STREAM_PLAY_LASTFM_NEXT;
				}
				if ((avail >= (BUFF_MAXRAM/2)) &&
				    (stream_lastdatatime+STREAM_TIMEOUT3 < NutGetSeconds())) {
					printf_P(PSTR("Missing lastfm data. Got %lu, expected %lu bytes\n"), (long unsigned int)stream_datastart, (long unsigned int)stream_datalen);
					return STREAM_PLAY_LASTFM_NEXT;
				}
			}
		}
	} else {
		state_pause_off(); //use the networks as buffer is bad
	}
	if (lastfm) {
		return STREAM_PLAY_LASTFM;
	} else {
		return STREAM_PLAY_NETWORK;
	}
}

uint8_t stream_stop(void) {
	connurl_terminate();
	buff_init();
	stream_cleandecoder();
	return STREAM_STOPPED;
}

uint8_t stream_nextfile(void) {
	if ((BUFF_MAXRAM - buff_free() > STREAM_FILSESWTICHBUF) &&
	    (mp3dev_pause1_get() == 0)) { //wait until buffer is mostly empty
		return STREAM_FILE_END;
	}
	stream_dbfileselect(state_fileselectmode_get());
	return STREAM_FILE_END;
}

static void stream_gui_upd(void) {
	//update buffer state
	state_bufffill_set(100-(buff_free()*100)/BUFF_MAXRAM);
	state_mp3_secplayed(mp3dev_playtimeget());
}

static void stream_waitforfree(void) {
	/*Disconnect old streams. This is important to release the memory,
	  otherwise serveral threads crash because there is no heap left.
	  An extra delay is added, because the user might have pressed stop
	  shortly before the next play.
	*/
	connurl_terminate();
	NutSleep(1000);
	uint8_t delay = 50;
	while (delay--) {
		NutSleep(200);
		if (NutHeapAvailable() > 12000) {
			break;
		}
		stream_gui_upd();
	}
}

uint8_t stream_network_start(void) {
	uint8_t res;
	uint8_t retry;
	state_mp3_seclen(0); //infinty in theory, may be days in practice
	state_id3_clear();
	state_filenameonemptytitle();
	stream_waitforfree();
	icy200_reset();
	stream_toread = 0;
	for (retry = 0; retry < STREAM_CONNECTION_RETRIES; retry++) {
		if (stream_connectnext) {
			char * url = s_strdup(stream_connectnext);
			res = connurl_connect(url);
			free(url);
		} else {
			res = connurl_retrieve(stream_state.source);
		}
		if ((res) || (stream_state.nextstate != STREAM_INVALID)) {
			/*We check nextstate here, because otherwise it could take very long for
			  the thread to react again (connurl_retrive takes time too) and the user
			  might be confused if he aborts and wants to play a file and the
			  system does not react. */
			break;
		}
		stream_gui_upd();
		NutSleep(STREAM_CONNECTION_RETRY_WAIT);
	}
	if (stream_connectnext) {
		free(stream_connectnext);
		stream_connectnext = NULL;
	}
	buff_init();
	stream_cleandecoder();
	stream_lastdatatime = NutGetSeconds();
	if (res) {
		return STREAM_PLAY_NETWORK;
	}
	error_message_P(PSTR("Open URL failed"));
	return STREAM_STOPPED;
}

uint8_t stream_lastfm_start(void) {
	uint8_t res;
	uint8_t retry;
	char * stationname = state_lastfm_stationname_get();
	if (stationname == NULL) {
		error_message_P(PSTR("No station name"));
		return STREAM_STOPPED;
	}
	char * stationnameutf;
	if (!isutf8(stationname)) {
		stationnameutf = iso8859toutf8(stationname);
		free(stationname);
	} else {
		stationnameutf = stationname;
	}
	char * stationnameenc = urlencode(stationnameutf);
	free(stationnameutf);
	stream_waitforfree();
	state_id3_clear(); //may be needed before for the station name
	state_filenameonemptytitle();
	stream_toread = 0;
	for (retry = 0; retry < STREAM_CONNECTION_RETRIES; retry++) {
		res = lastfm_init(state_lastfm_username_get(), state_lastfm_md5pwd_get(), stationnameenc);
		if ((res) || (stream_state.nextstate != STREAM_INVALID)) {
			/*We check nextstate here, because otherwise it could take very long for
			  the thread to react again (connurl_retrive takes time too) and the user
			  might be confused if he aborts and wants to play a file and the
			  system does not react. */
			break;
		}
		stream_gui_upd();
		NutSleep(STREAM_CONNECTION_RETRY_WAIT);
	}
	free(stationnameenc);
	buff_init();
	stream_cleandecoder();
	if (res) {
		return STREAM_PLAY_LASTFM_NEXT;
	}
	error_message_P(PSTR("Lastfm init failed"));
	return STREAM_STOPPED;
}

uint8_t stream_lastfm_next(void) {
	uint8_t res;
	uint8_t retry;
	connurl_terminate();
	//wait until buffer is more empty
	while (buff_free() < STREAM_BUFF_MINFREE_NEXT) {
		if (stream_state.nextstate != STREAM_INVALID) {
			return STREAM_PLAY_LASTFM_NEXT;
		}
		NutSleep(200);
		stream_gui_upd();
	}
	state_mp3_seclen(0); //not used, even if lastfm provides it
	icy200_reset();
	stream_toread = 0;
	for (retry = 0; retry < STREAM_CONNECTION_RETRIES; retry++) {
		if (!stream_connectnext) {
			stream_connectnext = lastfm_nextrack(state_id3_get(ID3_ARTIST));
		}
		if (stream_connectnext) {
			res = connurl_connect(stream_connectnext);
			free(stream_connectnext);
			stream_connectnext = NULL;
		} else {
			puts_P(PSTR("Get next lastfm track failed"));
		}
		if ((res) || (stream_state.nextstate != STREAM_INVALID)) {
			/*We check nextstate here, because otherwise it could take very long for
			  the thread to react again (connurl_retrive takes time too) and the user
			  might be confused if he aborts and wants to play a file and the
			  system does not react. */
			break;
		}
		stream_gui_upd();
		NutSleep(STREAM_CONNECTION_RETRY_WAIT);
	}
	stream_lastdatatime = NutGetSeconds();
	stream_datastart = 0;
	if (res) {
		return STREAM_PLAY_LASTFM;
	}
	error_message_P(PSTR("Lastfm open failed"));
	return STREAM_STOPPED;
}

void stream_handle(void) {
	uint8_t ev = stream_state.nextstate;
	stream_state.laststate = ev;
	stream_state.nextstate = STREAM_INVALID; //detects external changes
	uint8_t ne = STREAM_STOPPED;
	visual_pauseplay_upd();
	//the part for file operations
	if (ev == STREAM_PLAY_FILE_START) {
		ne = stream_openfile();
	}
	if (ev == STREAM_PLAY_FILE) {
		ne = stream_copyfiledata();
	}
	if (ev == STREAM_FILE_END) {
		ne = stream_nextfile();
	}
	//the part for stream operation
	if (ev == STREAM_PLAY_NETWORK_START) {
		ne = stream_network_start();
	}
	if (ev == STREAM_PLAY_NETWORK) {
		ne = stream_copynet(0);
	}
	//the part for lastfm operation
	if (ev == STREAM_PLAY_LASTFM_START) {
		ne = stream_lastfm_start();
	}
	if (ev == STREAM_PLAY_LASTFM_NEXT) {
		ne = stream_lastfm_next();
	}
	if (ev == STREAM_PLAY_LASTFM) {
		ne = stream_copynet(1);
	}
	//general
	if (ev == STREAM_STOP) {
		ne = stream_stop();
	}
	if ((stream_state.nextstate == STREAM_INVALID) || (stream_state.nextstate == stream_state.laststate)) {
		//only overwrite, if no user changes (and remove double-requests)
		stream_state.nextstate = ne;
	}
	//update playing time, has to be done in various states -> simply do in all.
	stream_gui_upd();
	meta_queuecheck();
	stream_handle_recording();
}
