/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <compiler.h>
#include <inttypes.h>
#include <string.h>
#include <avr/interrupt.h>
#include <dev/nvmem.h>

#include "infplay.h"
#include "statestorage.h"
#include "hardware/ks0108drv.h"
#include "hardware/vs1002drv.h"
#include "hardware/lcdbacklight.h"
#include "id3.h"
#include "memmapper.h"
#include "menu-interpreter.h"
#include "action.h"
#include "stringhelper.h"
#include "error.h"
#include "recorder.h"
#include "info.h"
#include "spi.h"
#include "lastfm.h"

#define EVENTQUEUE_SIZE 32

typedef struct {
	uint8_t events[EVENTQUEUE_SIZE];
	uint8_t rp;
	uint8_t wp;
} eventqueue_t;

eventqueue_t equeue;

streamstate_t stream_state;

typedef struct {
uint8_t devinits[INIT_STEPS];
} misc_settings_t;

uint8_t permas_has_changed = 0;

typedef struct {
uint8_t dhcp;
uint16_t remotecodes[REMOTECODES];
uint8_t lcdbacklight;
uint8_t playtimedisplay;
uint8_t sortfilenames;
uint8_t bassboost;
uint8_t autorestore;
uint32_t myip;
uint32_t mygate;
uint32_t mydns;
} permanent_settings_t;

misc_settings_t miscs;
permanent_settings_t permas;

uint8_t state_player_stopped;

typedef struct {
uint8_t validconfig;
uint8_t recordallow;
char username[LASTFM_USERNAME_LEN]; //19 char maximum + \0
char md5passwd[LASTFM_MD5PWD_LEN]; //32 char md5 + \0
} lastfm_auth_t;

lastfm_auth_t state_lastfm_auth;

//rc5_states
uint8_t learnmode;
uint8_t learnidx;

volatile uint16_t lastmess;
volatile uint8_t rc5_repeated;

//stats
uint64_t mp3traffic; //traffic in byte since last reboot

permastats_t state_stats;

void state_backlight_upd(void) {
	backlight_set(permas.lcdbacklight);
}

void state_bassboost_update(void) {
	if ((state_devinit_get(INIT_MP3_DONE)) && state_devinit_get(INIT_BLKDEV_DONE)) {
		mp3dev_bass(permas.bassboost);
	}
}

void state_bassboost_set(uint8_t value) {
	permas.bassboost = value;
	state_bassboost_update();
	permas_has_changed = 1;
}

void state_autorestore_set(uint8_t value) {
	permas.autorestore = value;
	permas_has_changed = 1;
}

uint8_t state_autorestore_get(void) {
	return permas.autorestore;
}

void state_sortfilenames_set(uint8_t value) {
	permas.sortfilenames = value;
	permas_has_changed = 1;
}

uint8_t state_sortfilenames_get(void) {
	return permas.sortfilenames;
}

uint32_t state_stats_checksum(permastats_t * stats) {
	return ((stats->mp3totaltraffic ^ stats->reboots ^
	       stats->operatingminutes) + 42);
}

void state_init(void) {
	NutNvMemLoad(EEPROM_SETTINGS, &permas, sizeof(permas));
	NutNvMemLoad(EEPROM_STATS, &state_stats, sizeof(state_stats));
	NutNvMemLoad(EEPROM_LASTFM, &state_lastfm_auth, sizeof(state_lastfm_auth));
	//validate settings
	if (permas.dhcp > 1) {
		permas.dhcp = 1;
	}
	if (permas.lcdbacklight > 2) {
		permas.lcdbacklight = 0;
	}
	if (permas.playtimedisplay > 1) {
		permas.playtimedisplay = 0;
	}
	if (permas.bassboost > 1) {
		permas.bassboost = 0;
	}
	if (permas.sortfilenames > 1) {
		permas.sortfilenames = 0;
	}
	if (permas.autorestore > 1) {
		permas.autorestore = 0;
	}
	if (permas.myip == 0xFFFFFFFF) {
		permas.myip = INITIAL_IP;
	}
	if (permas.mygate == 0xFFFFFFFF) {
		permas.mygate = INITIAL_GATE;
	}
	if (permas.mydns == 0xFFFFFFFF) {
		permas.mydns = INITIAL_DNS;
	}
	if (state_stats_checksum(&state_stats) != state_stats.checksum) {
		state_stats.mp3totaltraffic = 0;
		state_stats.reboots = 0;
		state_stats.operatingminutes = 0;
		error_general(11);
	}
	state_stats.reboots++;
	state_backlight_upd();
	state_bassboost_set(permas.bassboost);
	menu_checkboxstate[MENU_CHECKBOX_bassboost] = permas.bassboost;
	menu_checkboxstate[MENU_CHECKBOX_sortfilenames] = permas.sortfilenames;
	menu_checkboxstate[MENU_CHECKBOX_autorestore] = permas.autorestore;
	stream_state.metaalbum = s_cmalloc(sizeof(char));
	stream_state.metasource = s_cmalloc(sizeof(char));
/* old default settings
	miscs.remotecodes[0] = 0x0016;
	miscs.remotecodes[1] = 0x0015;
	miscs.remotecodes[2] = 0x0010;
	miscs.remotecodes[3] = 0x0011;
	miscs.remotecodes[2] = 0x0010;
	miscs.remotecodes[3] = 0x0011;
	miscs.remotecodes[4] = 12320 & 0x13FF;
	miscs.remotecodes[5] = 14369 & 0x13FF;
	miscs.remotecodes[6] = 14352 & 0x13FF;
	miscs.remotecodes[7] = 12305 & 0x13FF;
	miscs.remotecodes[8] = 12305 & 0x13FF;
	miscs.remotecodes[9] = 10286 & 0x13FF;
	miscs.remotecodes[10] = 8237 & 0x13FF;
	miscs.remotecodes[11] = 14377 & 0x13FF;
	miscs.remotecodes[12] = 8236 & 0x13FF;
	miscs.remotecodes[13] = 10283 & 0x13FF;
	miscs.remotecodes[14] = 14379 & 0x13FF;

*/
	stream_state.volume = 25;
}

uint32_t state_ip_get(void) {
	return permas.myip;
}

void state_ip_set(uint32_t ip) {
	permas.myip = ip;
}

uint32_t state_gateway_get(void) {
	return permas.mygate;
}

void state_gateway_set(uint32_t gw) {
	permas.mygate = gw;
}

uint32_t state_dns_get(void) {
	return permas.mydns;
}

void state_dns_set(uint32_t dns) {
	permas.mydns = dns;
}

uint32_t state_totaltraffic_get(void) {
	return state_stats_traffic_get()/1000+state_stats.mp3totaltraffic;
}

uint32_t state_reboots_get(void) {
	return state_stats.reboots;
}

uint32_t state_operatingminutes_get(void) {
	return state_stats.operatingminutes + NutGetSeconds()/60;
}

void state_save_stats(void) {
	//update settings
	permastats_t * stats = s_cmalloc(sizeof(permastats_t));
	stats->mp3totaltraffic = state_totaltraffic_get();
	stats->reboots = state_reboots_get();
	stats->operatingminutes = state_operatingminutes_get();
	stats->checksum = state_stats_checksum(stats);
	NutNvMemSave(EEPROM_STATS, stats, sizeof(permastats_t));
	free(stats);
}

void state_save_settings(void) {
	if (permas_has_changed) {
		NutNvMemSave(EEPROM_SETTINGS, &permas, sizeof(permas));
		permas_has_changed = 0;
		info_message_P(PSTR("Saved"));
	}
}

void state_save_playingstate(void) {
	//important: sourcetype, state, fileselectmode, volume, source
	streamstate_nonvolatile_t eesave;
	eesave.sourcetype = stream_state.sourcetype;
	eesave.laststate = stream_state.laststate;
	//printf("save state %i\n", eesave.laststate);
	eesave.volume = stream_state.volume;
	eesave.recordandplaymode = state_fileselectmode_get() |
	                           state_recordmode_get() << 4;
	/*Uses duplicated strings, to make sure it's not changing between strlen and
	  saving. Moreover, s_strdup can handle NULL strings. */
	char * stationname = s_strdup(stream_state.radioname);
	NutNvMemSave(EEPROM_LASTSTATION, stationname, STATE_MAX_STATIONLEN);
	free(stationname);
	char * source = s_strdup(stream_state.source);
	eesave.sourcelength = s_strlen(source);
	if (eesave.sourcelength > STATE_MAX_PATHLEN) {
		eesave.sourcelength = STATE_MAX_PATHLEN;
	}
	NutNvMemSave(EEPROM_PLAYINGS, &eesave, sizeof(eesave));
	NutNvMemSave(EEPROM_SOURCENAME, source, eesave.sourcelength);
	free(source);
}

uint8_t state_load_playingstate(void) {
	streamstate_nonvolatile_t eeload;
	NutNvMemLoad(EEPROM_PLAYINGS, &eeload, sizeof(eeload));
	if (eeload.sourcelength <= STATE_MAX_PATHLEN) {
		char * lastartist = s_cmalloc(sizeof(char)*STATE_MAX_STATIONLEN+1);
		NutNvMemLoad(EEPROM_LASTSTATION, lastartist, STATE_MAX_STATIONLEN);
		if (stream_state.radioname) {
			free(stream_state.radioname);
		}
		stream_state.radioname = lastartist;
		char * source = s_cmalloc(sizeof(char)*(eeload.sourcelength+1));
		NutNvMemLoad(EEPROM_SOURCENAME, source, sizeof(char)*(eeload.sourcelength));
		state_mp3_filename(source);
		state_filenameonemptytitle();
		free(source);
		stream_state.recording = STREAM_RECORDING_STOP;
		//printf_P(PSTR("State %i restore '%s'\r\n"), eeload.laststate, source);
		if (eeload.volume <= STREAM_VOLUME_MAX) {
			stream_state.volume = eeload.volume;
			state_mp3_volume_update();
			visual_volume_upd();
		}
		state_fileselectmode_set(eeload.recordandplaymode & 0x0F); //does a range check
		state_recordmode_set(eeload.recordandplaymode >> 4); //does a range check
		if (eeload.sourcetype == STREAM_TYPE_FILE) {
			stream_state.sourcetype = STREAM_TYPE_FILE;
			state_event_put(EVENT_SHOW_FILEPLAYER);
		} else if (eeload.sourcetype == STREAM_TYPE_NETWORK) {
			stream_state.sourcetype = STREAM_TYPE_NETWORK;
			state_event_put(EVENT_SHOW_NETWORKPLAYER);
		} else {
			puts_P(PSTR("Source type invalid"));
			return 0;
		}
		if ((eeload.laststate == STREAM_STOPPED) || (eeload.laststate == STREAM_STOP)) {
			puts_P(PSTR("State restore stopped"));
			stream_state.nextstate = STREAM_STOPPED;
		} else {
			puts_P(PSTR("State restore playing"));
			state_mp3_replay();
		}
		return 1;
	}
	puts_P(PSTR("EEPROM playing state invalid"));
	return 0;
}

uint8_t state_event_get(void) {
	volatile uint8_t event = equeue.events[equeue.rp];
	if (event) {
		equeue.events[equeue.rp] = 0;
		equeue.rp = (equeue.rp+1) % EVENTQUEUE_SIZE;
	}
	return event;
}

uint8_t state_event_peek(void) {
	return equeue.events[equeue.rp];
}

void state_event_put(uint8_t event) {
	uint8_t sreg = SREG;
	cli();
	if (!equeue.events[equeue.wp]) {
		equeue.events[equeue.wp] = event;
		equeue.wp++;
		if (equeue.wp >= EVENTQUEUE_SIZE) {
			equeue.wp = 0;
		}
	} else {
		error_general(14);
	}
	SREG = sreg;
}

void state_devinit_set(uint8_t device) {
	if (!miscs.devinits[device]) {
		miscs.devinits[device] = 1;
		visual_init_upd();
		uint8_t i;
		uint8_t event = EVENT_ALLINIT;
		for (i = 0; i < INIT_STEPS; i++) {
			if (!miscs.devinits[i]) {
				event = EVENT_REDRAW;
				break;
			}
		}
		state_event_put(event);
	}
}

uint8_t state_devinit_get(uint8_t device) {
	return miscs.devinits[device];
}

void state_fileselectmode_set(uint8_t mode) {
	if (mode < 4) {
		menu_radiobuttonstate[MENU_RBUTTON_playmode] = mode;
	}
}

void state_recordmode_set(uint8_t mode) {
	if (mode < 2) {
		menu_radiobuttonstate[MENU_RBUTTON_recordmode] = mode;
	}
}

uint8_t state_fileselectmode_get(void) {
	return menu_radiobuttonstate[MENU_RBUTTON_playmode];
}

uint8_t state_recordmode_get(void) {
	return menu_radiobuttonstate[MENU_RBUTTON_recordmode];
}

void state_pause_off(void) {
	mp3dev_pause2(0x00);
	visual_pauseplay_upd();
}

void state_pause_toggle(void) {
	uint8_t nstate = 0xFF-mp3dev_pause2_get();
	mp3dev_pause2(nstate);
	visual_pauseplay_upd();
}

void state_backlight_dec(void) {
	if (permas.lcdbacklight > 0) {
		permas.lcdbacklight--;
		permas_has_changed = 1;
	}
	state_backlight_upd();
}

void state_backlight_inc(void) {
	if (permas.lcdbacklight < 0x2) {
		permas.lcdbacklight++;
		permas_has_changed = 1;
	}
	state_backlight_upd();
}

uint8_t state_backlight_get(void) {
	return permas.lcdbacklight;
}

void state_mp3_volume_update(void) {
	mp3dev_volume(0xff - (stream_state.volume << 3));
}

void state_mp3_volume_dec(void) {
	if (stream_state.volume > 0)
		stream_state.volume--;
	state_mp3_volume_update();
}

void state_mp3_volume_inc(void) {
	if (stream_state.volume < STREAM_VOLUME_MAX)
		stream_state.volume++;
	state_mp3_volume_update();
}

uint8_t state_mp3_volume_get(void) {
	return stream_state.volume;
}

void state_mp3_filename(char * filename) {
	if (filename != stream_state.source) {
		if (stream_state.source != NULL) {
			free(stream_state.source);
		}
		stream_state.source = s_strdup(filename);
	}
}

void state_mp3_fileplay(char * filename) {
	state_mp3_filename(filename);
	stream_state.nextstate = STREAM_PLAY_FILE_START;
	stream_state.sourcetype = STREAM_TYPE_FILE;
	state_pause_off();
}

void state_mp3_urlplay(char * filename) {
	state_mp3_filename(filename);
	if (endswith(filename, ending_lfm)) {
		if (state_lastfm_validconfig_get() == 42) {
			stream_state.nextstate = STREAM_PLAY_LASTFM_START;
		} else {
			error_message_P(PSTR("Lastfm auth not set"));
			return;
		}
	} else {
		stream_state.nextstate = STREAM_PLAY_NETWORK_START;
	}
	stream_state.sourcetype = STREAM_TYPE_NETWORK;
	state_pause_off();
}

void state_mp3_replay(void) {
	if (stream_state.sourcetype == STREAM_TYPE_NETWORK) {
		if (endswith(stream_state.source, ending_lfm)) {
			if (state_lastfm_validconfig_get() == 42) {
				stream_state.nextstate = STREAM_PLAY_LASTFM_START;
			} else {
				error_message_P(PSTR("Lastfm auth not set"));
				return;
			}
		} else {
			stream_state.nextstate = STREAM_PLAY_NETWORK_START;
		}
	} else if (stream_state.sourcetype == STREAM_TYPE_FILE) {
		stream_state.nextstate = STREAM_PLAY_FILE_START;
	} else {
		error_message_P(PSTR("No source to play"));
	}
	state_pause_off();
}

void state_mp3_lastfmskip(void) {
	if (stream_state.laststate == STREAM_PLAY_LASTFM) {
		stream_state.nextstate = STREAM_PLAY_LASTFM_NEXT;
	}
}

void state_mp3_secupdate(void) {
	uint16_t secs = stream_state.played;
	char dir = ' ';
	if (permas.playtimedisplay && stream_state.length && (stream_state.length >= stream_state.played)) {
		secs = stream_state.length - secs;
		dir = '-';
	}
	uint16_t min = secs/60;
	secs %= 60;
	//initialized memory with INITIAL_TEXT_LENGTH
	sprintf_P((char *)(menu_strings[MENU_TEXT_mp3time]), PSTR("%c%i:%02i"), dir, min, secs);
	state_event_put(EVENT_REDRAW);
}

void state_mp3_seclen(uint16_t seconds) {
	if (stream_state.length != seconds) {
		stream_state.length = seconds;
		state_mp3_secupdate();
	}
}

uint16_t state_mp3_seclen_get(void) {
	return stream_state.length;
}

void state_mp3_secplayed(uint16_t seconds) {
	if (stream_state.played != seconds) {
		stream_state.played = seconds;
		state_mp3_secupdate();
	}
}

void state_mp3_playtimetoggle(void) {
	permas.playtimedisplay = 1-permas.playtimedisplay;
	permas_has_changed = 1;
	state_mp3_secupdate();
}

void state_mp3_stop(void) {
	stream_state.nextstate = STREAM_STOP;
}

uint8_t state_buff_perc = 255; //invalid starting value forces a redraw
void state_bufffill_set(uint8_t percent) {
	if (state_buff_perc != percent) {
		state_buff_perc = percent;
		sprintf_P((char *)(menu_strings[MENU_TEXT_mp3buf]), PSTR("%2i%%"),percent);
		state_event_put(EVENT_REDRAW);
	}
}

static uint8_t state_recording_permitted(void) {
	if ((stream_state.laststate == STREAM_PLAY_NETWORK) ||
	    ((state_lastfm_recordallow_get()) &&
	     ((stream_state.laststate == STREAM_PLAY_LASTFM) ||
	      (stream_state.laststate == STREAM_PLAY_LASTFM_NEXT)))) {
		return 1;
	}
	return 0;
}

static void state_checkautorecord(void) {
	if (state_recording_permitted()) {
		if (recorder_check(state_id3_get(ID3_ARTIST), state_id3_get(ID3_TITLE))) {
			if (spi_mmc_kbfree() >= STREAM_RECORDING_MINMEMSTARTFREE) {
				recorder_remove(state_id3_get(ID3_ARTIST), state_id3_get(ID3_TITLE));
				state_record_change(STREAM_RECORDING_TITLE);
			}
		} else if (stream_state.recording == STREAM_RECORDING_TITLE) {
			state_record_change(STREAM_RECORDING_STOP);
		}
	}
}

void state_id3_set(char * text, uint8_t type) {
	/*
	must gurantee, that no menu_strings[x] is accidentally set to NULL or other
	uninitialized RAM*/
	//printf("%s\n", text);
	uint16_t len = s_strlen(text);
	unsigned char * ntext = s_cmalloc(sizeof(char)*(len+1));
	unsigned char * otext = NULL;
	strcpy((char *)ntext, text);
	uint8_t titleartistchanged = 0;
	if (type == ID3_TITLE) {
		otext = menu_strings[MENU_TEXT_mp3title];
		menu_strings[MENU_TEXT_mp3title] = ntext;
		if (strcmp((char *)otext, (char *)ntext)) {
			titleartistchanged = 1;
		}
	} else if (type == ID3_ARTIST) {
		otext = menu_strings[MENU_TEXT_mp3artist];
		menu_strings[MENU_TEXT_mp3artist] = ntext;
		if (strcmp((char *)otext, (char *)ntext)) {
			titleartistchanged = 1;
		}
	} else if (type == ID3_ALBUM) {
		otext = stream_state.metaalbum;
		stream_state.metaalbum = ntext;
	} else if (type == ID3_SOURCE) {
		otext = stream_state.metasource;
		stream_state.metasource = ntext;
	} else { //invalid type!
		free(ntext);
	}
	free(otext);
	//switch between album and source
	unsigned char * source = stream_state.metasource;
	if (s_strlen((char *)stream_state.metaalbum) > 0) {
		source = stream_state.metaalbum;
	}
	free(menu_strings[MENU_TEXT_mp3source]);
	menu_strings[MENU_TEXT_mp3source] = (unsigned char *)s_strdup((char *)source);
	if (titleartistchanged) {
		state_checkautorecord();
	}
}

void state_id3_clear(void) {
	menu_strings[MENU_TEXT_mp3title][0] = '\0';
	menu_strings[MENU_TEXT_mp3artist][0] = '\0';
	menu_strings[MENU_TEXT_mp3source][0] = '\0';
	stream_state.metasource[0] = '\0';
	stream_state.metaalbum[0] = '\0';
}

void state_filenameonemptytitle(void) {
	if (strlen(state_id3_get(ID3_TITLE)) < 1) {
		char * filename = strrchr(stream_state.source, '/'); //+1: exclude the found '/'
		if (filename == NULL) { //if no '/' was found (very very unlikely)
			filename = stream_state.source;
		} else
			filename++;
		state_id3_set(filename, ID3_TITLE);
	}
}

char * state_id3_get(uint8_t type) {
	if (type == ID3_TITLE) {
		return (char *)menu_strings[MENU_TEXT_mp3title];
	} else if (type == ID3_ARTIST) {
		return (char *)menu_strings[MENU_TEXT_mp3artist];
	} else if (type == ID3_ALBUM) {
		return (char *)stream_state.metaalbum;
	} else if (type == ID3_SOURCE) {
		return (char *)stream_state.metasource;
	} else { //invalid type!
		return NULL;
	}
}

uint8_t state_usedhcp(void) {
	return permas.dhcp;
}

void state_dhcpset(uint8_t value) {
	permas.dhcp = value;
	permas_has_changed = 1;
}

void state_rc5_learn_start(uint8_t idx) {
	learnidx = idx;
	learnmode = 1;
}

void state_rc5_learn_stop(void) {
	learnmode = 0;
}

uint16_t state_rc5_code_get(uint8_t idx) {
	return permas.remotecodes[idx];
}

void state_rc5_rec(uint16_t data) {
	uint8_t i;
	if (data == lastmess) {
		if (rc5_repeated < RC5REPEATS_PRESS) {
			rc5_repeated++;
			return;
		}
	} else
		rc5_repeated = 0;
	lastmess = data;
	if (learnmode) {
		if (learnidx < REMOTECODES) {
			permas.remotecodes[learnidx] = lastmess & 0x13FF;
			permas_has_changed = 1;
		}
		learnmode = 0;
		state_event_put(EVENT_LEAVELEARN); //leave learn screen
	} else {
		data &= 0x13FF;
		for (i = 0; i < REMOTECODES; i++) {
			if (permas.remotecodes[i] == data) {
				state_event_put(i+1);
				break;
			}
		}
	}
}

//stats
/*add traffic in bytes */
void state_stats_traffic_add(uint16_t value) {
	mp3traffic += value;
}

uint32_t state_stats_traffic_get(void) {
	return mp3traffic/1000;
}

void state_record_trychange(uint8_t newstate) {
	if (state_recording_permitted()) {
		if (spi_mmc_kbfree() >= STREAM_RECORDING_MINMEMSTARTFREE) {
			state_record_change(newstate);
		} else {
			//change text, if you change STREAM_RECORDING_MINMEMSTARTFREE
			error_message_P(PSTR("SD/MMC < 5MB free"));
		}
	} else {
		error_message_P(PSTR("Not permitted"));
	}
}

void state_record_change(uint8_t newstate) {
	stream_state.recordingnext = newstate;
	visual_recordbutton_upd();
}

char * state_lastfm_md5pwd_get(void) {
	return state_lastfm_auth.md5passwd;
}

char * state_lastfm_username_get(void) {
	return state_lastfm_auth.username;
}

uint8_t state_lastfm_recordallow_get(void) {
	return state_lastfm_auth.recordallow;
}

uint8_t state_lastfm_validconfig_get(void) {
	return state_lastfm_auth.validconfig;
}

void state_lastfm_config_set(uint8_t valid, uint8_t recordallow, char * username, char * md5pwd) {
	state_lastfm_auth.validconfig = valid;
	state_lastfm_auth.recordallow = recordallow;
	strncpy(state_lastfm_auth.username, username, LASTFM_USERNAME_LEN-1);
	state_lastfm_auth.username[LASTFM_USERNAME_LEN-1] = '\0';
	strncpy(state_lastfm_auth.md5passwd, md5pwd, LASTFM_MD5PWD_LEN-1);
	state_lastfm_auth.md5passwd[LASTFM_MD5PWD_LEN-1] = '\0';
	NutNvMemSave(EEPROM_LASTFM, &state_lastfm_auth, sizeof(lastfm_auth_t));
}

char * state_lastfm_stationname_get(void) {
	FILE * f = fopen(stream_state.source, "r");
	if (f == NULL) {
		puts_P(PSTR("Could not open file"));
		return NULL; //no need to free() here
	}
	char * stationname = s_cmalloc((STATE_MAX_STATIONLEN+1)*sizeof(char)); //is already null terminated
	fread(stationname, sizeof(char), STATE_MAX_STATIONLEN, f);
	fclose(f);
	uint16_t i;
	for (i = 0; i < STATE_MAX_STATIONLEN; i++) {
		if (stationname[i] == '\n') {
			stationname[i] = '\0';
			break;
		}
		if (stationname[i] == '\r') {
			stationname[i] = '\0';
			break;
		}
	}
	if (strcmp_P(stationname, PSTR("USECURRENTARTIST")) == 0) {
		char * artist = state_id3_get(ID3_ARTIST);
		if (s_strlen(artist) > 1) {
			strncpy(stationname, artist, STATE_MAX_STATIONLEN);
			if (stream_state.radioname) {
				free(stream_state.radioname);
			}
			stream_state.radioname = strdup(artist);
		} else {
			if (s_strlen(stream_state.radioname) > 0) {
				strncpy(stationname, stream_state.radioname, STATE_MAX_STATIONLEN);
			} else {
				puts_P(PSTR("No current artist"));
				return NULL;
			}
		}
	}
	return stationname;
}

