/* Infplay
  (c) 2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* A hashed element has the following format:
10 bytes per song.
the bytes mean the following:
1: 1. title name char
2: 2. title name char
3: middle title name char
4: last title name char
5: 1. artist name char
6: 2. artist name char
7: middle artist char
8: last artist char
9: upper byte hashsum
10: lower byte hashsum
If title or artist is less than two chars long, its ignored and \0 are saved.
*/
#include <stdio.h>
#include <inttypes.h>
#include <stdlib.h>
#include <string.h>

#include "recorder.h"
#include "stringhelper.h"

#define RECORDER_ELEMNUM 100
#define RECORDER_ELEMSIZE 10
#define RECORDER_MAXSIZE (RECORDER_ELEMNUM*RECORDER_ELEMSIZE)

uint8_t * recorder_hasharray;
size_t recorder_size;

#ifdef TEST_PC

#define RECORDER_FILENAME "torecord.bin"

#include "pctests/avrmapper.h"

int main(void) {
	printf("Tests the record later feature. Uses the filename '"RECORDER_FILENAME"'\n");
	recorder_init();
	printf("Record size: %i\n", recorder_size);
	recorder_add("ABC", "def");
	recorder_add("ABC", "foobar");
	recorder_add("asdfgh", "barfood");
	if (recorder_check("ABC", "de")) {
		puts("Error 1: On list, but should not");
		return 1;
	}
	if (recorder_check("ABC", "foobar") == 0) {
		puts("Error 2: Missing something in the list");
		return 1;
	}
	recorder_remove("ABC", "foobar");
	if (recorder_check("ABC", "foobar")) {
		puts("Error 3: Removing from list failed");
		return 1;
	}
	recorder_add("hohoho", "xmas");
	while (recorder_size < RECORDER_MAXSIZE-1) {
		char tmp[10];
		sprintf(tmp, "%i", recorder_size);
		recorder_add("dummy", tmp);
	}
	printf("Record size max filled: %i\n", recorder_size);
	while (recorder_size > 0) {
		char tmp[10];
		sprintf(tmp, "%i", recorder_size-RECORDER_ELEMSIZE);
		if (recorder_check("dummy", tmp) == 0) {
			break;
		}
		recorder_remove("dummy", tmp);
	}
	printf("Finished. Record size: %i\n", recorder_size);
	return 0;
}

#else

#include "memmapper.h"
#include "error.h"
#include "info.h"
#include "infplay.h"

#define RECORDER_FILENAME FSDEV_ROOT"torecord.bin"

#endif


void recorder_init(void) {
	if (recorder_hasharray)
		free(recorder_hasharray);
	recorder_hasharray = s_cmalloc(RECORDER_MAXSIZE);
	FILE * f = fopen(RECORDER_FILENAME, "rb");
	if (f) {
		recorder_size = fread(recorder_hasharray, 1, RECORDER_MAXSIZE, f);
		recorder_size -= recorder_size%RECORDER_ELEMSIZE;
		fclose(f);
	} else
		recorder_size = 0;
}

static void recorder_calcelem(uint8_t * buff, char * artist, char * title) {
	size_t artl = s_strlen(artist);
	size_t titlel = s_strlen(title);
	uint16_t hash = 0;
	size_t i;
	if (titlel > 1) {
		buff[0] = title[0];
		buff[1] = title[1];
		buff[2] = title[titlel/2];
		buff[3] = title[titlel-1];
	}
	if (artl > 1) {
		buff[4] = artist[0];
		buff[5] = artist[1];
		buff[6] = artist[artl/2];
		buff[7] = artist[artl-1];
	}
	for (i = 0; i < titlel; i++) {
		hash = hash ^ (hash << 1);
		hash += title[i];
	}
	for (i = 0; i < artl; i++) {
		hash = hash ^ (hash << 1);
		hash += artist[i];
	}
	buff[8] = hash >> 8;
	buff[9] = hash & 0xFF;
}

static int16_t recorder_listpos(uint8_t * elem) {
	size_t i;
	for (i = 0; i < recorder_size; i += RECORDER_ELEMSIZE) {
		if (memcmp(elem, recorder_hasharray+i, RECORDER_ELEMSIZE) == 0) {
			return i;
		}
	}
	return -1;
}

static void recorder_writefile(void) {
	FILE * f = fopen(RECORDER_FILENAME, "wb");
	if (f) {
		fwrite(recorder_hasharray, 1, recorder_size, f);
		fclose(f);
	}
}

uint8_t recorder_check(char * artist, char * title) {
	if (recorder_hasharray == NULL) {
		return 0;
	}
	uint8_t elem[RECORDER_ELEMSIZE];
	recorder_calcelem(elem, artist, title);
	int16_t idx = recorder_listpos(elem);
	if (idx >= 0) {
		return 1;
	} else
		return 0;
}

void recorder_remove(char * artist, char * title) {
	if (recorder_hasharray == NULL) {
		return;
	}
	uint8_t elem[RECORDER_ELEMSIZE];
	recorder_calcelem(elem, artist, title);
	int16_t idx = recorder_listpos(elem);
	if (idx >= 0) { //remove from list and save
		uint16_t i = idx;
		//memmove would use a second array internally - bad if low RAM
		recorder_size -= RECORDER_ELEMSIZE;
		while (i < (recorder_size)) {
			recorder_hasharray[i] = recorder_hasharray[i+RECORDER_ELEMSIZE];
			i++;
		}
		recorder_writefile();
	}
}

void recorder_add(char * artist, char * title) {
	if (recorder_hasharray == NULL) {
		return;
	}
	uint8_t elem[RECORDER_ELEMSIZE];
	recorder_calcelem(elem, artist, title);
	int16_t idx = recorder_listpos(elem);
	if (idx < 0) { //not already on list
		if ((recorder_size + RECORDER_ELEMSIZE) <= RECORDER_MAXSIZE) {
			memcpy(recorder_hasharray+recorder_size, elem, RECORDER_ELEMSIZE);
			recorder_size += RECORDER_ELEMSIZE;
			recorder_writefile();
			info_message_P(PSTR("Added to list"));
		} else
			error_message_P(PSTR("List full"));
	} else
		info_message_P(PSTR("Already on list"));
}
