/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This file is based on the some example source code from Nut/OS!

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/confnet.h>
#include <net/route.h>
#include <netdb.h>
#include <avr/pgmspace.h>
#include <string.h>

#include "infplay.h"
#include "statestorage.h"
#include "ftpserver.h"
#include "action.h"
#include "filemanager.h"
#include "lastfm.h"
#include "hardware/spimmcdrv.h"

uint32_t net_ip_addr;
uint32_t net_ip_gate;

uint8_t net_abortrequest;

int net_update_config(void) {
	u_char mac[6] = MY_MAC;
	u_long ip_mask = inet_addr(MY_IPMASK);
	return NutNetIfConfig2(DEV_ETHER_NAME, mac, net_ip_addr, ip_mask, net_ip_gate);
}

void net_setip(uint32_t ip) {
	net_ip_addr = ip;
	net_update_config();
}

void net_setgateway(uint32_t gw) {
	net_ip_gate = gw;
	net_update_config();
}

void net_setdns(uint32_t dns) {
	NutDnsConfig2(0, 0, dns, dns);
}

/*
 * Setup the ethernet device. Try DHCP first. If this is
 * the first time boot with empty EEPROM and no DHCP server
 * was found, use hardcoded values.
 */
int InitEthernetDevice(void) {
	net_ip_addr = state_ip_get();
	net_ip_gate = state_gateway_get();
	u_char mac[6] = MY_MAC;

	if (NutRegisterDevice(&DEV_ETHER, 0x8300, 5)) {
		puts_P(PSTR("No Ethernet Device"));
		return -1;
	}
	state_devinit_set(INIT_NET_DONE);
	if (state_usedhcp()) {
		if (NutDhcpIfConfig(DEV_ETHER_NAME, 0, 60000) == 0) {
			printf_P(PSTR("Ethernet %s configured\n"), DEV_ETHER_NAME);
			net_ip_addr = confnet.cdn_ip_addr;
			NutIpRouteQuery(0, &net_ip_gate);
			return 0;
		}
		puts_P(PSTR("Initial boot..."));
		if (NutDhcpIfConfig(DEV_ETHER_NAME, mac, 60000) == 0) {
			net_ip_addr = confnet.cdn_ip_addr;
			NutIpRouteQuery(0, &net_ip_gate);
			printf_P(PSTR("Ethernet %s configured\n"), DEV_ETHER_NAME);
			return 0;
		}
	}
	net_update_config();
	net_setdns(state_dns_get());
	puts_P(PSTR("Configured without DHCP"));
	return 0;
}

/*
 * Query a time server and optionally update the hardware clock.
 */
static int QueryTimeServer(void)
{
	int rc = -1;
	time_t now;
  uint32_t timeserver = inet_addr(MYTIMED);
  /* Query network time service and set the system time. */
	uint8_t try;
	for (try = 0; try < 4; try++) {
		if(NutSNTPGetTime(&timeserver, &now) == 0) {
			printf_P(PSTR("Got time from %s\n"), MYTIMED);
			rc = 0;
			stime(&now);
			break;
		} else {
			printf_P(PSTR("Got time from %s failed\n"), MYTIMED);
		}
		NutSleep(500);
	}
	return rc;
}

/*
 * Try to get initial date and time from the hardware clock or a time server.
 */
static int InitTimeAndDate(void)
{
    int rc = -1;

    /* Set the local time zone. */
    _timezone = MYTZ * 60L * 60L;

    /* No hardware RTC, query the time server if available. */
    rc = QueryTimeServer();
    return rc;
}

THREAD(NetInit, arg) {
	uint32_t ipgate;
	/* Initialize the Ethernet device and print our IP address. */
	if (InitEthernetDevice()) {
		ThreadTerminate();
	}
	state_devinit_set(INIT_NETCONFIG_DONE);
	visual_network_upd();
	state_event_put(EVENT_REDRAW);
	printf_P(PSTR("IP Addr: %s\n"), my_inet_ntoa(confnet.cdn_ip_addr));
	printf_P(PSTR("IP Mask: %s\n"), my_inet_ntoa(confnet.cdn_ip_mask));
	NutIpRouteQuery(0, &ipgate);
	printf_P(PSTR("IP Gate: %s\n"), my_inet_ntoa(ipgate));

	/* Initialize system clock and calendar. */
	if (InitTimeAndDate() == 0) {
		time_t now = time(0);
		//Nut/OS vs glibc API incompatibility...
#ifdef __NUT_EMULATION__
		struct tm *lot;
#else
		tm *lot; //or: struct _tm * lot;
#endif
		lot = localtime(&now);
		state_devinit_set(INIT_CLOCK_DONE);
		printf_P(PSTR("Date: %02u.%02u.%u\n"), lot->tm_mday, lot->tm_mon + 1, 1900 + lot->tm_year);
		printf_P(PSTR("Time: %02u:%02u:%02u\n"), lot->tm_hour, lot->tm_min, lot->tm_sec);
		srandom(now); //now is a long according to the Nut/OS doc
	}
	NutThreadSetPriority(65); //less important than default priority
	while(!state_devinit_get(INIT_FTPROOT_DONE)) {
		NutSleep(500);
	}
	uint8_t firstbuild = 1;
	for (;;) {
		if (Spi_haswritten() || firstbuild) {
			lastfm_config_tryupdate();
			Spi_writeclear(); //clears the flag that something changed on the SD card
			manager_buildup(FSDEV_ROOT); //rebuild once and after every ftp closing
			if (state_sortfilenames_get() && (!net_abortrequest)) {
				manager_sortnames(FSDEV_ROOT);
			}
			firstbuild = 0;
		}
		if (!net_abortrequest) {
			state_devinit_set(INIT_FTP_DONE);
			FtpService();
		}
		if (net_abortrequest) {
			net_abortrequest = 2;
			ThreadTerminate();
		}
	}
}

void net_terminate(void) {
	net_abortrequest = 1;
	manager_abort();
	uint8_t timeout = 75;
	while ((net_abortrequest < 2) && (timeout)) {
		NutSleep(10);
		timeout--;
	}
	if (net_abortrequest < 2) {
		NUTTHREADINFO *tdp;
		for (tdp = nutThreadList; tdp; tdp = tdp->td_next) {
			if (strcmp(tdp->td_name, "network") == 0) {
				NutThreadRemoveQueue(tdp, (NUTTHREADINFO **) & runQueue);
				break;
			}
		}
	}
}

FILE * net_connectandopen(TCPSOCKET ** sock, uint32_t ip, uint16_t port) {
	*sock = NutTcpCreateSocket();
	if (*sock == NULL) {
		puts_P(PSTR("socket create failed"));
		return NULL;
	}
	if (NutTcpConnect(*sock, ip, port)) {
		puts_P(PSTR("NutTcpConnect failed"));
		return NULL;
	}
	//open stream
	FILE * stream;
	if ((stream = _fdopen((int)*sock, "r+b")) == 0) {
		puts_P(PSTR("_fdopen failed"));
		NutTcpCloseSocket(*sock);
		return NULL;
	}
	return stream;
}

