/* Infplay
  (c) 2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
Documentation about the protocol can be found here:
http://code.google.com/p/thelastripper/wiki/LastFM12UnofficialDocumentation
*/

#define _GNU_SOURCE

#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>

#include "lastfm.h"
#include "stringhelper.h"


#define LASTFM_TIMEOUT 7

#define LASTFM_MAXLINELENGTH 256

#define LASTFM_DEBUG

/*Trys a better authentication method, where the password is send with salt
  unfortunately, with this method the station name can not be set.
  Remember to add the code for compiling the md5sum routine into the Makefile if
  you want to fix this.
 */
//#define LASTFM_EXPERIMENTAL

#ifdef  LASTFM_EXPERIMENTAL
#include "cryptlib/md5.h"
#endif

#ifdef TEST_PC

/*
Press Control+C to terminate the program (and to avoid useless traffic).
*/

#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>
#include <time.h>

#include "pctests/avrmapper.h"
#include "pctests/net_nuttounix.h"

#define LASTFM_CONFIGFILENAME "lastfm-auth.txt"

//order must fit with the id3 v1 order too
#ifndef ID3_TITLE
#define ID3_TITLE 0
#endif

#ifndef ID3_ARTIST
#define ID3_ARTIST 1
#endif

#ifndef ID3_ALBUM
#define ID3_ALBUM 2
#endif

#ifndef ID3_SOURCE
#define ID3_SOURCE 3
#endif

void lastfm_settext(char * text, uint8_t id) {
	if (id == ID3_TITLE) {
		printf("title: %s\n", text);
	}
	if (id == ID3_ALBUM) {
		printf("album: %s\n", text);
	}
	if (id == ID3_ARTIST) {
		printf("artist: %s\n", text);
	}
	if (id == ID3_SOURCE) {
		printf("source: %s\n", text);
	}
}

char * fm_username;
char * fm_password;
char * fm_station;

static void state_lastfm_config_set(uint8_t valid, uint8_t recordallow, char * username, char * password) {
	printf("Username: '%s'\n", username);
	printf("Password: '%s'\n", password);
	fm_username = strdup(username);
	fm_password = strdup(password);
	printf("Valid value: %i, recording allowed: %i\n", valid, recordallow);
}

int main(int argc, char **argv) {
	if ((argc != 4) && (argc != 3)) {
		printf("Usage: username md5ofpassword station\n");
		printf("or 'f' station: reads in the file '"LASTFM_CONFIGFILENAME"'. THE FILE WILL BE DELETED AFTER READING.\n");
		printf("The station must already be in the URL encoded format\n");
		return 1;
	}
	if (argc == 4) {
		fm_username = argv[1];
		fm_password = argv[2];
		fm_station = argv[3];
	} else if (strcmp(argv[1], "f") == 0) {
		lastfm_config_tryupdate();
		fm_station = argv[2];
		if (fm_username == NULL) {
			printf("Readind config file failed\n");
			return 2;
		}
	} else {
		printf("Invalid parameter\n");
		return 1;
	}
	if (lastfm_init(fm_username, fm_password, fm_station)) {
		char * url = lastfm_nextrack(NULL);
		printf("Play by entering: 'vlc %s'\n", url);
	} else {
		printf("Lastfm init failed\n");
		return 1;
	}
	return 0;
}

#else

#include <sys/socket.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <sys/timer.h>

#include "memmapper.h"
#include "infplay.h"
#include "statestorage.h"

#define LASTFM_CONFIGFILENAME FSDEV_ROOT"lastfm-auth.txt"

#endif

char * lastfm_sessionid = NULL;
uint32_t lastfm_ip;
const char * lastfm_staticdomain = "ws.audioscrobbler.com";
#ifdef LASTFM_EXPERIMENTAL
//const char * lastfm_staticdomain2 = "www.last.fm";
const char * lastfm_staticdomain2 = "post.audioscrobbler.com";
#endif

void lastfm_extracttag(char * input, PGM_P start, PGM_P end, char ** target) {
	char * st = strstr_P(input, start);
	char * ed = strstr_P(input, end);
	if ((st != NULL) && (ed != NULL) && (st < ed)) {
		int len = ed-st-strlen_P(start);
		if (*target) {
			free(*target);
		}
		*target = s_strndup(st+strlen_P(start), len);
	}
}

uint8_t lastfm_readline(char * target, FILE * source, uint16_t buffsize) {
	int i = 0;
	uint32_t st = NutGetSeconds();
	while(1) {
		//get a single line
		if (i >= buffsize) {
			i = 0;
		}
		int gb = fread(target+i, 1, 1, source);
		if (gb > 0) {
			if (target[i] == '\n') {
				target[i] = '\0';
				return 1;
			}
			if (target[i] != '\r') {
				i += gb;
			}
		} else if ((st+LASTFM_TIMEOUT) <  NutGetSeconds()) {
			target[i] = '\0';
			return 0;
		}
	}
}

#ifdef LASTFM_EXPERIMENTAL
static char * lastfm_calctoken(char * md5passwd, time_t timestamp) {
	md5_hash_t hash;
	size_t s = strlen(md5passwd)+33; //32 should be enough for a uint32_t
	char * tmp = s_cmalloc(s*sizeof(char));
#ifdef TEST_PC
	sprintf(tmp, "%s%u", md5passwd, (uint32_t)timestamp);
#else
	sprintf(tmp, "%s%lu", md5passwd, (uint32_t)timestamp);
#endif
	printf("to hash: '%s'\n", tmp);
	md5(&hash, tmp, strlen(tmp)*8);
	uint8_t i;
	for (i = 0; i < 16; i++) {
		tmp[i*2] = hexadecimal(hash[i]>>4);
		tmp[i*2+1] = hexadecimal(hash[i]);
	}
	tmp[32] = '\0';
	printf("hash: '%s'\n", tmp);
	return tmp;
}
#endif


uint8_t lastfm_init(char * username, char * md5passwd, char * artistname) {
	//clean up
	if (lastfm_sessionid) {
		free(lastfm_sessionid);
		lastfm_sessionid = NULL;
	}
	//resolve ip of lastfm domain to an ip
	lastfm_ip = NutDnsGetHostByName((unsigned char*)lastfm_staticdomain);
	if (!lastfm_ip) {
		puts_P(PSTR("could not resolve domain"));
		return 0;
	}
#ifndef LASTFM_EXPERIMENTAL
//old, insecure protocol with a fixed md5sum of the password
	//connect to ip, 1. request
	TCPSOCKET * sock1;
	FILE * stream1 = net_connectandopen(&sock1, lastfm_ip, 80);
	if (stream1 == NULL) {
		return 0;
	}
	fprintf_P(stream1, PSTR("GET /radio/handshake.php?username=%s&passwordmd5=%s HTTP/1.1\r\nHost: ws.audioscrobbler.com\r\n\r\n"), username, md5passwd);
	printf_P(PSTR("GET /radio/handshake.php?username=%s&passwordmd5=%s HTTP/1.1\r\nHost: ws.audioscrobbler.com\r\n\r\n"), username, md5passwd);
	fflush(stream1);
	//wait for return
	char * inpbuff = s_cmalloc(sizeof(char)*LASTFM_MAXLINELENGTH);
	uint8_t abort = 150;
	while ((lastfm_readline(inpbuff, stream1, LASTFM_MAXLINELENGTH)) && abort) {
		if (strncmp_P(inpbuff, PSTR("session="), 8) == 0) {
			lastfm_sessionid = strdup((char *)(inpbuff+8));
		}
		if (lastfm_sessionid) {
			fclose(stream1);
			NutTcpCloseSocket(sock1);
			break;
		}
		abort--;
	}
	if ((lastfm_sessionid == NULL) || (strcmp_P(lastfm_sessionid, PSTR("FAILED")) == 0)) {
		free(inpbuff);
		puts_P(PSTR("Lastfm: Login failed"));
		return 0;
	}
#ifdef LASTFM_DEBUG
	printf_P(PSTR("Session: '%s'\n"), lastfm_sessionid);
#endif
#else
//new more secure protocol
	//resolve ip of post lastfm domain to an ip
	uint32_t lastfm_postip = NutDnsGetHostByName((unsigned char*)lastfm_staticdomain2);
	if (!lastfm_postip) {
		puts_P(PSTR("could not resolve domain2"));
		return 0;
	}
	//connect to ip, 1. request
	TCPSOCKET * sock1;
	FILE * stream1 = net_connectandopen(&sock1, lastfm_postip, 80);
	if (stream1 == NULL) {
		return 0;
	}
	time_t timestamp;
	time(&timestamp);
	char * token = lastfm_calctoken(md5passwd, timestamp);
#ifdef LASTFM_DEBUG
	printf_P(PSTR("GET /?hs=true&p=1.2&c=ass&v=1.3.1.1&u=%s&t=%lu&a=%s HTTP/1.1\r\nHost: %s\r\n\r\n"), username, timestamp, token, lastfm_staticdomain2);
	//printf_P(PSTR("\n\n\n%s/?hs=true&p=1.2&c=ass&v=1.3.1.1&u=%s&t=%lu&a=%s\n"), lastfm_staticdomain2, username, timestamp, token);
#endif
	fprintf_P(stream1, PSTR("GET /?hs=true&p=1.2&c=ass&v=1.3.1.1&u=%s&t=%lu&a=%s HTTP/1.1\r\nHost: %s\r\n\r\n"), username, timestamp, token, lastfm_staticdomain2);
	free(token);
	fflush(stream1);
	//wait for return
	char * inpbuff = s_cmalloc(sizeof(char)*LASTFM_MAXLINELENGTH);
	uint8_t abort = 40;
	uint8_t nextissession = 0;
	while ((lastfm_readline(inpbuff, stream1, LASTFM_MAXLINELENGTH)) && abort) {
		if (nextissession) {
			lastfm_sessionid = strdup(inpbuff);
			break;
		}
		if (strcmp(inpbuff, "OK") == 0) {
			nextissession = 1;
		}
		abort--;
	}
	if (lastfm_sessionid == NULL) {
		free(inpbuff);
		puts_P(PSTR("Lastfm: Login failed"));
		return 0;
	}
#ifdef LASTFM_DEBUG
	printf_P(PSTR("Session: '%s'\n"), lastfm_sessionid);
#endif
#endif
	//send 2. request. change station
	TCPSOCKET * sock2;
	FILE * stream2 = net_connectandopen(&sock2, lastfm_ip, 80);
	if (stream2 == NULL) {
		free(inpbuff);
		return 0;
	}
	fprintf_P(stream2, PSTR("GET /radio/adjust.php?session=%s&url=lastfm://artist/%s/similarartists&debug=0 HTTP/1.1\r\nHost: %s\r\n\r\n"), lastfm_sessionid, artistname, lastfm_staticdomain);
	fflush(stream2);
	//wait for return
	abort = 150;
	uint8_t success = 0;
	while ((lastfm_readline(inpbuff, stream2, LASTFM_MAXLINELENGTH)) && abort) {
		if (strncmp_P(inpbuff, PSTR("response="), 9) == 0) {
			fclose(stream2);
			NutTcpCloseSocket(sock2);
			if (strcmp_P(inpbuff+9, PSTR("OK")) == 0) {
#ifdef LASTFM_DEBUG
				printf_P(PSTR("Radio station '%s' set\n"), artistname);
#endif
				success = 1;
				break;
			} else {
				printf_P(PSTR("Radio station '%s' not set: %s\n"), artistname, inpbuff);
				break;
			}
		}
		abort--;
	}
	free(inpbuff);
	return success;
}

static void lastfm_metaprepare(char * text, uint8_t type) {
	if (text) {
		utf8toiso8859(text);
		lastfm_settext(text, type);
		free(text);
	} else {
		lastfm_settext("", type);
	}
}

char * lastfm_nextrack(char * avoidauthor) {
	//send 3. request. request playlist
	TCPSOCKET * sock3;
	FILE * stream3 = net_connectandopen(&sock3, lastfm_ip, 80);
	if (stream3 == NULL) {
		return 0;
	}
	fprintf_P(stream3, PSTR("GET /radio/xspf.php?sk=%s&discovery=0&desktop=0 HTTP/1.1\r\nHost: %s\r\n\r\n"), lastfm_sessionid, lastfm_staticdomain);
	fflush(stream3);
#ifdef LASTFM_DEBUG
	printf_P(PSTR("Playlist: %s/radio/xspf.php?sk=%s&discovery=0&desktop=0\n"), lastfm_staticdomain, lastfm_sessionid);
#endif
	//wait for return
	char * url = NULL;
	char * title = NULL;
	char * author = NULL;
	char * album = NULL;
	char * source = NULL;
	uint8_t track = 0;
	char * recbuff = s_cmalloc(sizeof(char)*LASTFM_MAXLINELENGTH);
	uint8_t abort = 200;
	while (lastfm_readline(recbuff, stream3, LASTFM_MAXLINELENGTH)) {
		if (strstr_P(recbuff, PSTR("<track>"))) {
			track = 1;
		}
		if (track) {
			lastfm_extracttag(recbuff, PSTR("<location>"), PSTR("</location>"), &url);
			lastfm_extracttag(recbuff, PSTR("<title>"), PSTR("</title>"), &title);
			lastfm_extracttag(recbuff, PSTR("<creator>"), PSTR("</creator>"), &author);
			lastfm_extracttag(recbuff, PSTR("<album>"), PSTR("</album>"), &album);
			if (author) {
				if ((avoidauthor == NULL) || (strcmp(avoidauthor, author))) {
					break;
				} else {
					printf_P(PSTR("Avoiding author %s\n"), author);
				}
			}
		} else {
			lastfm_extracttag(recbuff, PSTR("<title>"), PSTR("</title>"), &source);
		}
		if (strstr_P(recbuff, PSTR("</track>"))) {
			track = 0;
			break;
		}
		abort--;
		if (!abort) {
			free(recbuff);
			puts_P(PSTR("Lastfm: Invalid data"));
			break;
		}
	}
	fclose(stream3);
	NutTcpCloseSocket(sock3);
	free(recbuff);
	if (url == NULL) {
		puts_P(PSTR("Lastfm: No URL"));
		return NULL;
	}
#ifdef LASTFM_DEBUG
	printf_P(PSTR("URL for music '%s'\n"), url);
#endif
	lastfm_metaprepare(title, ID3_TITLE);
	lastfm_metaprepare(author, ID3_ARTIST);
	lastfm_metaprepare(album, ID3_ALBUM);
	if (source) {
		utf8toiso8859(source);
		char * fmsource = s_cmalloc(s_strlen(source)+9*sizeof(char));
		sprintf_P(fmsource, PSTR("Lastfm: %s"), source);
		lastfm_settext(fmsource, ID3_SOURCE);
		free(source);
		free(fmsource);
	} else {
		lastfm_settext("", ID3_SOURCE);
	}

	return url;
}

void lastfm_config_tryupdate(void) {
	FILE * f = fopen(LASTFM_CONFIGFILENAME, "r");
	if (f) {
		char * valid = s_cmalloc(4*sizeof(char));
		char * recording = s_cmalloc(4*sizeof(char));
		char * username = s_cmalloc(LASTFM_USERNAME_LEN*sizeof(char));
		char * md5password = s_cmalloc(LASTFM_MD5PWD_LEN*sizeof(char));
		lastfm_readline(valid, f, 4);
		lastfm_readline(recording, f, 4);
		lastfm_readline(username, f, LASTFM_USERNAME_LEN);
		lastfm_readline(md5password, f, LASTFM_MD5PWD_LEN);
		fclose(f);
		state_lastfm_config_set(atoi(valid), atoi(recording), username, md5password);
		free(valid);
		free(recording);
		free(username);
		free(md5password);
		unlink(LASTFM_CONFIGFILENAME);
		puts_P(PSTR("Lastfm config updated"));
	}
}
