/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <avr/wdt.h>
#include <sys/heap.h>
#include <string.h>

#include "infplay.h"
#include "spi.h"
#include "network.h"
#include "ftpserver.h"
#include "hardware/ks0108drv.h"
#include "hardware/rotdecoder.h"
#include "hardware/rc5decoder.h"
#include "menudata-progmem.c"
#include "statestorage.h"
#include "menu-interpreter.h"
#include "action.h"
#include "bigbuff.h"
#include "error.h"
#include "hardware/rs232debug.h"
#include "memmapper.h"

//some dynamic graphics
//button has a size of 12x15
unsigned char gfx_pause[23] = {
 0x00, 0x03, 0x9C,
 0x39, 0xC3, 0x9C,
 0x39, 0xC3, 0x9C,
 0x39, 0xC3, 0x9C,
 0x39, 0xC3, 0x9C,
 0x39, 0xC3, 0x9C,
 0x39, 0xC3, 0x9C,
 0x00, 0x00};

unsigned char gfx_play[23] = {
 0x00, 0x01, 0x00,
 0x18, 0x01, 0xC0,
 0x1E, 0x01, 0xF0,
 0x1F, 0x81, 0xFC,
 0x1F, 0x81, 0xF0,
 0x1E, 0x01, 0xC0,
 0x18, 0x01, 0x00,
 0x00, 0x00
};

char * ending_mp3 = ".mp3";
char * ending_url = ".url";
char * ending_lfm = ".lfm";

uint8_t fat_count_abort;

void ThreadTerminate(void) {
	for(;;) {
		NutSleep(1000);
	}
}

/*
 * Assign stdout to the UART device.
 */
void InitDebugDevice(void) {
	u_long baud = DBG_BAUDRATE;
	NutRegisterDevice(&DEV_DEBUG, 0, 0);
//otherwise on the pc, the menu will end in the file too
#ifndef __NUT_EMULATION__
	freopen(DEV_DEBUG_NAME, "w", stdout);
#endif
	_ioctl(_fileno(stdout), UART_SETSPEED, &baud);
}

unsigned char menu_byte_get(MENUADDR addr) {
	if (addr >= MENU_DATASIZE) {
		return 0;
	}
	return pgm_read_byte(menudata+addr);
}

void menu_screen_set(unsigned char x, unsigned char y, unsigned char color) {
	ks0108_set_pixel(x, y, color);
}

void menu_screen_flush(void) {
	ks0108_flush();
}

void menu_screen_clear(void) {
	ks0108_clear();
}

void sched_menu_yield(void) {
	banking_displayrelease();
	NutThreadYield();
	banking_displayuse();
}

#ifndef __NUT_EMULATION__
/*
runningThread will be set to zero in .init4, but the pointer of
runningThread->td_name points most likely to the external ram, which is enabled
in .init5. So we save td_name before .init4 and use the value in .init6.
Note: Theoretically, it is not guranteed that *td_name has still the same
content, because .init5 already does some heap modifying stuff.
*/
char * inf_lastthread __attribute__ ((section (".noinit")));

void earliest_init(void) __attribute__((naked)) __attribute__((section(".init1")));
void earliest_init(void) {
	asm volatile ("clr __zero_reg__");
	wdt_enable(WDTO_2S); //includes a wdr reset
	rs232debug_printchar('B');
	rs232debug_printchar('O');
	rs232debug_printchar('O');
	rs232debug_printchar('T');
	if (MCUCSR & (1<<WDRF)) {
		inf_lastthread = runningThread->td_name; //td_name may point in the external ram
	} else {
		inf_lastthread = NULL;
	}
}

void i6(void) __attribute__((naked)) __attribute__((section(".init6")));
void i6(void) {
	if (inf_lastthread) {
		rs232debug_printstring(inf_lastthread);
	}
}

void i8(void) __attribute__((naked)) __attribute__((section(".init8")));
void i8(void) {
	wdt_reset();
}
#endif


THREAD(Watchdog, arg) {
	NutThreadSetPriority(5); //most important of all threads
	while(1) {
		wdt_reset();
		NutSleep(500);
	}
}

#ifdef NUTDEBUG_CHECK_STACK
extern HEAPNODE *heapAllocList;

void inf_print_heap(void) {
	HEAPNODE * x = heapFreeList;
	uint8_t secabort = 0xff;
	while(x && secabort) {
		printf_P(PSTR("f %u-%u\n"), x, x->hn_size);
		x = x->hn_next;
		wdt_reset();
		secabort--;
	}
#ifdef NUTDEBUG_HEAP
	x = heapAllocList;
	secabort = 0xff;
	while(x && secabort) {
		printf_P(PSTR("u %u-%u-%u-%.50s-%u\n"), x, x->hn_size, x->ht_size, x->ht_file, x->ht_line);
		x = x->ht_next;
		wdt_reset();
		secabort--;
	}
#endif
	printf_P(PSTR("Heap %i %i\n"), NutHeapAvailable(), NutHeapRegionAvailable());
}


#ifdef NUTMEM_GUARD

//defines copied from os/heap.c
#define NUTMEM_GUARD_PATTERN   ((int)(0xDEADBEEF))
#define NUTMEM_GUARD_BYTES     sizeof(NUTMEM_GUARD_PATTERN)
#define NUT_HEAP_OVERHEAD   (sizeof(HEAPNODE) - sizeof(HEAPNODE *))

static void inf_checkpattern(void) {
	HEAPNODE * x = heapAllocList;
	uint8_t secabort = 0xff;
	uint8_t corrupt = 0;
	while(x && secabort) {
		size_t off = (x->hn_size - NUT_HEAP_OVERHEAD) / sizeof(int) - 1;
		int *tp = (int *) (uintptr_t) &x->hn_next;
		if (*tp != NUTMEM_GUARD_PATTERN) {
			corrupt = 1;
			printf_P(PSTR("Guard corrupt before: %u, %i\n"), x, *tp);
			wdt_reset();
		}
		if (*(tp + off) != NUTMEM_GUARD_PATTERN) {
			corrupt = 1;
			printf_P(PSTR("Guard corrupt after: %u, %i\n"), x, *(tp+off));
			wdt_reset();
		}
		x = x->ht_next;
		secabort--;
	}
	if (corrupt) {
		inf_print_heap();
	}
}

#endif

THREAD(Memwatch, arg) {
	uint16_t minfreeheap = 0xFFFF;
	uint8_t countdown = 10;
	while(1) {
		uint16_t freeheap = NutHeapAvailable();
		if (minfreeheap > freeheap) {
			minfreeheap = freeheap;
		}
		if (countdown == 0) {
			NUTTHREADINFO *tdp;
			for (tdp = nutThreadList; tdp; tdp = tdp->td_next) {
				printf_P(PSTR("%s: %lu bytes stack never used\n"), tdp->td_name, (unsigned long)NutThreadStackAvailable(tdp->td_name));
				NutThreadYield(); //give the watchdog some time
			}
			printf_P(PSTR("Min free heap: %u bytes\n"), minfreeheap);
			//inf_print_heap();
		}
		countdown--;
		NutSleep(1000); //loop every second
#ifdef NUTMEM_GUARD
	inf_checkpattern();
#endif
	}
}
#endif

extern TCPSOCKET *tcpSocketList;
void inf_shutdown(void) {
	if (!state_player_stopped) {
		state_save_playingstate();
		state_save_stats();
		state_mp3_stop();
		//give the stream thread some time to close his own socket
		while (stream_state.laststate != STREAM_STOPPED) {
			NutSleep(20);
		}
#ifndef __NUT_EMULATION__
		//force a close of possible FTP connections
		TCPSOCKET *sp;
		for (sp = tcpSocketList; sp; sp = sp->so_next) {
			if (sp->so_state == TCPS_ESTABLISHED) {
				NutTcpCloseSocket(sp);
				//puts("Close");
			}
		}
#endif
		/*kill the network thread (otherwise the system crashes if the mmc is
		 suddently unmounted and sorting the filenames is in progress) */
		net_terminate();
		/*terminate the possibly still fat counting thread*/
		fat_count_abort = 1;
		NUTTHREADINFO *tdp;
		uint8_t found;
		do {
			found = 0;
			for (tdp = nutThreadList; tdp; tdp = tdp->td_next) {
				if (strcmp(tdp->td_name, "fatcount") == 0) {
					found = 1;
					NutSleep(10);
					break;
				}
			}
		} while (found);
		//unmount
		if (mmc_volid != -1) {
			_close(mmc_volid);
			mmc_volid = -1;
		}
		state_player_stopped = 1;
		puts_P(PSTR("Shutdown"));
	}
}

static prog_char head[] = "\n\nInfplay - Nut/OS %s - " CC_STRING "\n";
static prog_char resetsource[] = "Reset source: 0x%02x\n";

/*
 * Main application routine.
 *
 * Nut/OS automatically calls this entry after initialization.
 */
int main(void) {
	/* Initialize a debug output device and print a banner. */
	InitDebugDevice();
	printf_P(head, NutVersionString());
	wdt_enable(WDTO_2S); //includes a reset
	//wdt_disable();
	uint8_t rsource = MCUCSR;
	MCUCSR &= 0xE0; //clear reset cause flags
	if ((rsource & 0x1F) == 0) { //means an unexpected jump to 0000 had happened
		error_general(8); //hope this does not prevent calling error_panic()
		error_panic();
	}
	printf_P(resetsource, rsource);
#ifndef __NUT_EMULATION__
	printf_P(PSTR("Heap start: %p\n"), &__heap_start);
#endif
	printf_P(PSTR("Free Heap: %u\n"), NutHeapAvailable());
	/* the external RAM region 0x1500 to 0x2000 has 3 waitstates, it would
	   hurt performance if this gets allocated for the thread stacks.
	   Allocating this region while the threads spawn will prevent a use for
	   the heap. I think this is a better solution than starting without this
	   heap region and adding it later.
	   The allocated memory region must be at least 1280 byte and may not end up
     in the internal RAM.
 */
#ifndef __NUT_EMULATION__
	void * membulk = NULL;
	uint16_t blocky = 1280;
	do {
		if (membulk) {
			free(membulk);
		}
		membulk = s_cmalloc(blocky);
		blocky += 512;
	} while (membulk < (void *)NUTXMEM_START);
#ifdef NUTDEBUG_CHECK_STACK
	printf("bulk: %p\n", membulk);
#endif
#endif
	state_init(); //loads EEPROM settings
	/* Initialize everything for the display */
	ks0108_init(); //if display test, it takes too long for the watchdog
	menu_redraw();
	visual_init(); /*does not contain anything for the first screen,
	                must be called before the threads spawn! */
	/* spawn threads */
	NutThreadCreate("watchdog", Watchdog, 0, 172); //needs ~72 byte
	NutThreadCreate("spi", SpiInit, 0, 528); //needs ~428 byte
	NutThreadCreate("network", NetInit, 0, 700); //needs ~368 byte (may be more because of recursion)
#ifdef NUTDEBUG_CHECK_STACK
	NutThreadCreate("memwatch", Memwatch, 0, 204); //needs ~104 byte
#endif
	rc5Init();
	rotencoderInit();
	buff_init();
#ifndef __NUT_EMULATION__
	//release reserved memory
	free(membulk);
#endif
	//force memory corruption, to test if the code detect it
/*
	char * corr1 = s_cmalloc(10);
	corr1[10] = 0x42;
	char * corr2 = s_cmalloc(10);
	*(corr2 -1) = 0x43;
*/
	/* Main thread. */
	for(;;) {
		uint8_t key = state_event_get();
		if (key) {
			if (key == EVENT_REDRAW) {
				while (state_event_peek() == EVENT_REDRAW) {
					state_event_get();
				}
				if (visual_showstats) {
					visual_stats_upd();
				}
				banking_displayuse();
				menu_redraw();
				banking_displayrelease();
			} else {
				banking_displayuse();
				menu_keypress(key);
				banking_displayrelease();
			}
		} else if (visual_showstats) {
			state_event_put(EVENT_REDRAW);
			NutSleep(100);
		}
		NutSleep(50);
#ifdef BUFF_TESTMODE
		if (state_devinit_get(INIT_MOUNT_DONE)) {
			wdt_disable();
			buff_test();
		}
#endif
	}
	return 0; //never happens
}

