/* ID3 Reader Version 1.1 (c) 2009-2010 by Malte Marwedel
   http://www.marwedels.de/malte

   Terms of use: GPL Version 2 or later
   Example for reading id3 information from mp3 files

   Supports:
     Versions:
       ID3 V1.0
       ID3 V2.2
       ID3 V2.3
       ID3 V2.4
     Formtats:
       ISO-8859-1
       UTF-16 with BOM
       UTF-16 without BOM

   Partly supports:
     Flags:
       Jumps over external header
     Formats:
       UTF-8 is convertet to ISO-8859-1 (if chars can be represented there)

   Not supports:
     Unsynchronisation
     Extended Tag in ID3 V1.x

Should work on all emebdded systems, which support file operation similar to the
unix fread and fseek functions.
fseek must support: seek +x bytes from current position
                    go to position x (only if you want to use the same opened
                    file for mp3 playack afterwards)
                    go to position FILE_END - x (only if ID3 V1.0 support
                    is wished)
malloc and free are used too, but these could be replaced by static buffers.

Changelog:
  Version 1.1.1: Make sure, no uninitialized buffers are accessed if fread fails
  Version 1.1: Allowing umlauts in ISO-8859-1. Use version 1.0 if you only want
               ASCII in the output.

compile for pc: gcc -o id3 id3.c -Wall -Wextra -DTEST_PC
run on pc: ./id3 <your mp3 base dir>*.mp3
*/


#include <stdlib.h>
#include <inttypes.h>
#include <stdio.h>
#include <string.h>
#include "stringhelper.h"


#ifdef TEST_PC
#include <unistd.h>

#else
//add your avr specific includes here
#include "memmapper.h"
#include "statestorage.h"
#include <avr/pgmspace.h>
#endif

//order must fit with the id3 v1 order too
#ifndef ID3_TITLE
#define ID3_TITLE 0
#endif

#ifndef ID3_ARTIST
#define ID3_ARTIST 1
#endif

#ifndef ID3_ALBUM
#define ID3_ALBUM 2
#endif

#ifndef ID3_SOURCE
#define ID3_SOURCE 3
#endif

#ifdef TEST_PC

#include "pctests/avrmapper.h"

FILE * stream_file;

extern void id3_extract(void);


uint8_t pgm_read_byte(void * addr) {
	return *(uint8_t*)addr;
}

void state_id3_set(char * text, uint8_t id) {
	if (id == ID3_TITLE) {
		printf("title: %s\n", text);
	}
	if (id == ID3_ALBUM) {
		printf("album: %s\n", text);
	}
	if (id == ID3_ARTIST) {
		printf("artist: %s\n", text);
	}
}

int main(int argc, char ** argv) {
	if (argc < 2) {
		printf("Give one or more mp3 filenames as param\n");
		printf("Returns 0 if parsing done, 1: if no file could be openend, 2: if malloc failed\n");
		return 1;
	}
	int i;
	for (i = 1; i < argc; i++) {
		stream_file = fopen(argv[i], "rb");
		if (stream_file != NULL) {
			printf("%s:\n", argv[i]);
			id3_extract();
			fclose(stream_file);
		} else {
			printf("could not open file '%s'\n", argv[i]);
			return 1;
		}
	}
	return 0;
}

#endif

//==== the id3 reader =================
//you might want to put the #define ID3_ int an header file later


extern FILE * stream_file;

/*
See ID3 format at: http://www.id3.org/id3v2.3.0
*/

static prog_char id3_start[3] = {'I','D','3'};
static prog_char id3_tag[3] = {'T','A','G'};

//first half: id3.2 second half id3.3 and id3.4
#define ID3_COMPARES 8
//4 byte string, 1 byte ouptut index
static prog_char id3_table[ID3_COMPARES][5] = {
{'T','A','L',0, ID3_ALBUM},
{'T','T','2',0, ID3_TITLE},
{'T','P','1',0, ID3_ARTIST},
{'T','P','2',0, ID3_ARTIST},
{'T','A','L','B', ID3_ALBUM},
{'T','I','T','2', ID3_TITLE},
{'T','P','E','1', ID3_ARTIST},
{'T','P','E','2', ID3_ARTIST}};

static char * id3_v1text(void) {
	char * text = s_cmalloc(sizeof(char)*31); //the text is always 30 chars long
	fread(text, sizeof(char), 30, stream_file);
	uint8_t rp, wp = 0;
	for (rp = 0; rp <= 30; rp++) { //fix files with invalid chars
		uint8_t t1 = text[rp];
		if (t1 >= 0x20) {
			text[wp] = t1;
			wp++;
		}
		if (t1 == '\0')
			break;
	}
	text[wp] = '\0';
	return text;
}

static char * id3_text(uint16_t framesize) {
	//id 3v2 requires the framesize to be at least 1
	uint16_t l = 128;
	if (l > framesize)
		l = framesize;
	char * text = s_cmalloc(sizeof(char)*l);
	fread(text, sizeof(char), l, stream_file);
	/* text encoding defined by text[0]:
		$00 – ISO-8859-1 (ASCII).
		$01 – UTF-16 with BOM.
		$02 – UTF-16BE
		$03 – UTF-8: converted to ISO-8859-1
	*/
	uint16_t rp, wp = 0;
	if ((text[0] == 1) || (text[0] == 2)) { //utf16, convert
		//printf("utf-16\n");
		uint16_t rp = 1;
		uint8_t bom = 0; //byte order, useful first
		if (text[0] == 1) {
			rp = 3;
			if ((unsigned char)text[1] == 0xfe) {//may be either 0xfe or 0xff
				bom = 1; //byte order, useful last
				//printf("Info: Rare case with utf16 byte order inverted\n"); //untested, I never found such a file
			}
		}
		while (rp < l) {
			uint8_t t1 = text[rp+bom];
			uint8_t t2 = text[rp+1-bom];
			if ((t1 >= 0x20) && (!t2)) { //if useful char
				text[wp] = t1;
				wp++;
			}
			rp += 2;
			if ((!t1) & (!t2)) {
				break;
			}
		}
		text[wp] = '\0';
	} else if (text[0] == 3) { //2 indicate utf-8
		//printf("utf-8\n");
		wp = utf8toiso8859_subs(text, l, 1);
	} else { //0 indicate ISO-8859-1 (or fread failure)
		//printf("iso-8859-1\n");
		rp = 1;
		while (rp < l) {
			uint8_t t1 = text[rp];
			if (t1 >= 0x20) { //filter all useful chars
				text[wp] = t1;
				wp++;
			}
			rp++;
		}
	}
	if (wp >= l) //only happens if utf16 with wrong format occured
		wp = l-1;
	text[wp] = '\0'; //just make sure its \0 terminated
	if (framesize > l) {
		fseek(stream_file, framesize-l,  SEEK_CUR);
	}
	return text;
}

//supports only ID3 version 2.x
void id3_extract(void) {
	uint8_t id3temp[10];
	size_t readed = fread(id3temp, sizeof(uint8_t), 10, stream_file);
	if (readed < 10) {
		return;
	}
	uint8_t subversion = id3temp[3];
	if ((memcmp_P(id3temp, id3_start, 3) == 0) && (subversion <= 4)) {
		//printf("Info: ID3 version 2.%i\n", subversion);
		uint32_t headersize = 0;
		uint8_t i;
		for (i = 0; i < 4; i++) {
			headersize <<= 7;
			headersize += id3temp[i+6] & 0x7F;
		}
		if (id3temp[5] & 0x80) {
			//printf("Warning: Unsynchronisation must be used, not supported yet, could result in errors in very rare cases\n");
		}
		if (id3temp[5] & 0x40) { //only very few mp3s use this for a CRC check sum
			//printf("Warning: Extended header support is incomplete\n");
			readed = fread(id3temp, sizeof(uint8_t), 10, stream_file);
			if (readed != 10) {
				return;
			}
			fseek(stream_file, id3temp[3]-6, SEEK_CUR); //in version 2.3, the size is either 10 or 6 and nothing else
		}
		//printf("Header size: %u\n", headersize); //should be %lu on the AVR platform
		//decode information
		uint8_t frameheadersize = 10;
		uint8_t arrayoffset = ID3_COMPARES/2;
		uint8_t cmplen = 4;
		if (subversion < 3) {
			frameheadersize = 6;
			id3temp[9] = 0x00; //handle things as compression, group or discharged is 0 later
			arrayoffset = 0;
			cmplen = 3;
		}
		uint32_t parsed = 0;
		while (parsed < headersize) { //usually does one loop for every frame
			if (fread(id3temp, sizeof(uint8_t), frameheadersize, stream_file) != frameheadersize) {
				//printf("Error: File end!\n");
				break; //ouch! file end
			}
			if (id3temp[0] == 0) { //must be ascii to be valid
				//either wrong file format, or padding reached
				break;
			}
			//calculate frame size
			uint32_t framesize = 0;
			uint8_t i;
			for (i = 0; i < 4; i++) {
				if ((subversion <= 2) && (i < 3)) { //id3 v 2.2: has only 3 bytes frame size
					framesize <<= 8;
					framesize += id3temp[i+3];
				}
				if (subversion == 3) { //id3 v 2.3
					framesize <<= 8;
					framesize += id3temp[i+4];
				}
				if (subversion == 4) { //id3 v 2.4
					framesize <<= 7;
					framesize += id3temp[i+4] & 0x7F;
				}
				//printf("adding %i\n", id3temp[i+4]);
			}
			//printf("Framesize %i\n", framesize);
			//if compression, or group or discharged, or too large, must be larger than 0
			if ((id3temp[9] & 0xE0) || (framesize > 10000) || (!framesize)) {
				fseek(stream_file, framesize,  SEEK_CUR);
				//printf("Not fitting: %i\n", framesize);
			} else {
				char * text = NULL;
				for (i = 0; i < ID3_COMPARES/2; i++) {
					uint8_t type = pgm_read_byte(&id3_table[arrayoffset+i][4]);
					if (memcmp_P(id3temp, id3_table[arrayoffset+i], cmplen) == 0) {
						text = id3_text(framesize);
						state_id3_set(text, type);
						free(text);
						break;
					}
				}
				if (text == NULL) { //no fitting tag found
					fseek(stream_file, framesize,  SEEK_CUR);
				}
			}
			parsed += frameheadersize;
			parsed += framesize;
		}
		//make sure, mp3 playing starts at the right position, even if the frames did contain invalid data or padding was used
		fseek(stream_file, headersize+10,  SEEK_SET); //not needed if file is re-opened for playing anyway
	} else { //no id3 v2 header, look for id3 v1
		fseek(stream_file, -128, SEEK_END);
		readed = fread(id3temp, sizeof(uint8_t), 3, stream_file);
		if ((readed == 3) && (memcmp_P(id3temp, id3_tag, 3) == 0)) { //if there is an id3 tag
			//printf("Info: ID3 version 1\n");
			uint8_t i;
			for (i = 0; i < 3; i++) {
				char * text = id3_v1text();
				if (strlen(text) > 0)
					state_id3_set(text, i);
				free(text);
			}
		}
		//make sure, mp3 playing starts from the beginning
		fseek(stream_file, 0, SEEK_SET); //not needed if file is re-opened for playing anyway
	}
}
