/*
ICY200 meta info reader Version 2.2 (c) 2009-2010 by Malte Marwedel
   http://www.marwedels.de/malte

   Terms of use: GPL Version 2 or later

Implement:
1. A function icy200_settext() which gets the extracted informationen.
2. size_t icy200_getdata(uint8_t * buffer, size_t size) which reads size
   bytes from the stream into the buffer. The real number of read bytes
   should be returned (eg 0 if no data were available).
   Remember to close the connection if there were no data within the last
   several seconds.
3. void icy200_redirect(char * url) which gets called if the opened stream
   is a playlist with URLs or the opened stream was redirected.
4. icy200_stop(). This function gets called if something went wrong.
   Most likely you want to close the connection and inform the user.


Call:
3. icy200_reset() at the beginning of the stream
4. icy200_decode() directly after the beginning, and store the return value.
4b. Repeat calling icy200_decode() as soon as you have read exactly as much
   bytes as the previous icy200_decode() call told you. At the beginning
   and if there were no data to read, this can be immediately.

Possible bug: It could be a problem if the distance
  between meta data (icy-metaint:) is larger than sizeof(size_t)
  or sizeof(int).
  On 8 or 16 bit system, these types are most likely only 16 bit wide.
  However, I did not find streams using such large blocks.

*/



#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "icy200.h"

//#define ICY200_DEBUG

#ifdef TEST_PC

/*Enable to copy all read data to the file "mp3dump"
useful for later testing with the same input data".
Data will be appended, so delete the file before running the program. */
//#define dumpit

/*
Good source:
http://www.smackfu.com/stuff/programming/shoutcast.html

Example calls:
for http://at.ah.fm:9000
./icy200 87.98.160.187 9000 /

for http://streamer-dtc-aa04.somafm.com:80/stream/1018
./icy200 205.188.234.4 80 /stream/1018

for http://sc.slayradio.org:8000/
./icy200 81.186.251.7 8000 /

for http://scfire-dtc-aa05.stream.aol.com:80/stream/1048
./icy200 205.188.234.5 80 /stream/1048

Example for a playlist:
http://somafm.com/startstream=spacestation.pls
./icy200 66.98.132.51 80 /startstream=spacestation.pls

Example for a playlist (different format):
./icy200 www.slayradio.org 80 /tune_in.php/128kbps/listen.m3u

Example for invalid data:
./icy200 icy200

Press Control+C to terminate the program (and to avoid useless traffic).

*/

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <fcntl.h>
#include <netdb.h>

#include "pctests/avrmapper.h"

//order must fit with the id3 v1 order too
#ifndef ID3_TITLE
#define ID3_TITLE 0
#endif

#ifndef ID3_ARTIST
#define ID3_ARTIST 1
#endif

#ifndef ID3_ALBUM
#define ID3_ALBUM 2
#endif

#ifndef ID3_SOURCE
#define ID3_SOURCE 3
#endif

int sock;
int file;
int type;

void icy200_settext(char * text, uint8_t id) {
	if (id == ID3_TITLE) {
		printf("title: %s\n", text);
	}
	if (id == ID3_ALBUM) {
		printf("album: %s\n", text);
	}
	if (id == ID3_ARTIST) {
		printf("artist: %s\n", text);
	}
	if (id == ID3_SOURCE) {
		printf("source: %s\n", text);
	}
}

void icy200_redirect(char * url) {
	printf("Redirect to url: '%s'\n", url);
	exit(0);
}

void icy200_stop(void) {
	printf("The icy200 reader detected an invalid answer\n");
	exit(0);
}

void icy200_filesize(uint32_t contentlength) {
	printf("File size: %iKB\n", contentlength/1000);
}

size_t icy200_getdata(uint8_t * buffer, size_t size) {
	int gb = 0;
	if (type == 1) {
		gb = recv(sock, buffer, size, 0);
	}
	if (type == 2) {
		gb = read(file, buffer, size);
	}
	if (gb < 0) {
		gb = 0;
		usleep(200000);
	}
#ifdef dumpit
	FILE * f = fopen("mp3dump", "a");
	if (f) {
		fwrite(buffer, sizeof(uint8_t), gb, f);
		fclose(f);
	}
#endif
	return gb;
}

int main(int argc, char **argv) {
	if ((argc != 4) && (argc != 2)) {
		printf("Domain Port URL\n or FILE with recorded data\n");
		return 1;
	}
	if (argc == 4) {
		char * domain = argv[1];
		//resolve domain to an ip
		struct hostent * h = gethostbyname(domain);
		if ((h == NULL) || (!(h->h_addr_list[0]))) {
			printf("Resolving host failed\n");
			return 1;
		}
		char ip[20];
		int a = (unsigned char)(h->h_addr_list[0][0]);
		int b = (unsigned char)(h->h_addr_list[0][1]);
		int c = (unsigned char)(h->h_addr_list[0][2]);
		int d = (unsigned char)(h->h_addr_list[0][3]);
		sprintf(ip, "%i.%i.%i.%i", a, b, c, d);
		//printf("IP: %s\n", ip);
		int port = atoi(argv[2]);
		char * url = argv[3];
		struct sockaddr_in echoServAddr;
		if ((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
			return 1;
		memset(&echoServAddr, 0, sizeof(echoServAddr));
		echoServAddr.sin_family = AF_INET;
		echoServAddr.sin_addr.s_addr = inet_addr(ip);
		echoServAddr.sin_port = htons(port);
		if (connect(sock, (struct sockaddr *) &echoServAddr, sizeof(echoServAddr)) < 0) {
			printf("Connection failed\n");
			return 1;
		}
		fcntl(sock, F_SETFL, O_NONBLOCK); //for testing error recovery
		char str[1024];
		snprintf(str, 1024, "GET %s HTTP/1.0\r\nIcy-Metadata:1\r\nHost: %s\r\n\r\n", url, domain);
		//printf("'%s'\n", str);
		if (send(sock, str, strlen(str), 0) != (ssize_t) strlen(str)) {
			printf("Send failed\n");
			return 1;
		}
		type = 1;
	} else if (argc == 2) {
		file = open(argv[1], O_RDONLY);
		if (file < 0) {
			printf("Open file failed\n");
			return 1;
		}
		type = 2;
	}
	icy200_reset();
	uint8_t recbuff[1024];
	uint16_t secout = 10;
	while(secout) {
		int toread = icy200_decode();
		while ((toread) && (secout)) {
			int maxread = 1024;
			if (toread < maxread)
				maxread = toread;
			int gb = icy200_getdata(recbuff, maxread);
			if (gb == 0) {
				sleep(1);
				secout--;
			} else
				secout = 10;
			//printf("Got %i bytes mp3 data\n", gb);
			toread -= gb;
		}
	}
	if (secout == 0) {
		printf("No data...\n");
	}
	return 0;
}

#endif

#define ICY_UNDECIDED 0
#define ICY_MP3 1
#define ICY_MP3DATA 2
#define ICY_PLAYLIST 3
#define ICY_PLAYLISTB 4
#define ICY_ICYHEADER 5
#define ICY_ICYDATA 6
#define ICY_REDIRECT 7
#define ICY_INVALID 8

/*

icy200_metaint: bytes to read between next icy200 data (or file length if plain mp3)

icy200_buffer: buffer to read a whole line before interpreting the data
icy200_bufferpos: location in the buffer to write to
icy200_buffersize: maximum size of the buffer
icy200_abort: Counts down. Used for calling icy200_stop() if no proper
  data were found within some bytes
*/
uint8_t icy200_state;
uint32_t icy200_metaint;
uint16_t icy200_bufferpos;
uint16_t icy200_buffersize;
uint16_t icy200_abort;
char * icy200_buffer;

void icy200_reset(void) {
	icy200_state = ICY_UNDECIDED;
	icy200_metaint = 32768; //in the case the data were not send, use big blocks
	icy200_abort = 10000;
	free(icy200_buffer);
	icy200_buffer = NULL;
	icy200_filesize(0);
}

static void icy200_extract(char * buffer) {
	char * title = strstr_P(buffer, PSTR("StreamTitle='"));
	if (title) {
		title += 13; //+strlen("StreamTitle='")
		char * end = strchr(title, '\'');
		if (end) {
			*end = '\0';
			char * dash = strstr_P(title, PSTR(" - "));
			if (dash) {
				*dash = '\0';
				icy200_settext(dash+3, ID3_TITLE);
				icy200_settext(title, ID3_ARTIST);
			} else {
				icy200_settext(title, ID3_TITLE);
				icy200_settext("", ID3_ARTIST);
			}
		}
 	}
}

static char * icy200_getafter(char * buffer, PGM_P beginning) {
	char * param = NULL;
	size_t len = strlen_P(beginning);
	if (strncmp_P(buffer, beginning, len) == 0) {
		param = buffer + len;
		while (*param == ' ') {
			param++;
		}
	}
	return param;
}

//lastfm uses urls with ~154 chars length
#define ICY200_BUFFSIZE 256


/*
Initial seek for:
"icy-name"
"icy-metaint"
*/

static size_t icy200_decodeinitialmeta(void) {
	/* The initial meta data end with an empty line*/
	if (strlen(icy200_buffer) == 0) {
		icy200_state = ICY_ICYDATA;
		return icy200_metaint;
	}
	char * name = icy200_getafter(icy200_buffer, PSTR("icy-name:"));
	if (name) {
		icy200_settext(name, ID3_SOURCE);
	}
	char * metaint = icy200_getafter(icy200_buffer, PSTR("icy-metaint:"));
	if (metaint) {
		icy200_metaint = atol(metaint);
	}
	return 0;
}

static size_t icy200_decidetype(void) {
	if (strncmp_P(icy200_buffer, PSTR("icy-"), 4) == 0) {
#ifdef ICY200_DEBUG
			puts("ICY Data");
#endif
		icy200_state = ICY_ICYHEADER;
		return icy200_decodeinitialmeta();
	}
	if ((strncmp_P(icy200_buffer, PSTR("HTTP/"), 5) == 0) &&
		  (strstr_P(icy200_buffer, PSTR("302")))) { //redirect detected
#ifdef ICY200_DEBUG
		puts("Redirect");
#endif
		icy200_state = ICY_REDIRECT;
	}
	if ((strstr_P(icy200_buffer, PSTR("audio/x-scpls"))) ||
	    (strstr_P(icy200_buffer, PSTR("audio/mpegurl")))) { //must be a playlist or something invalid
#ifdef ICY200_DEBUG
		puts("Playlist");
#endif
		icy200_state = ICY_PLAYLIST;
	}
	if (icy200_state == ICY_UNDECIDED) {
		if (strstr_P(icy200_buffer, PSTR("audio/mpeg"))) {
#ifdef ICY200_DEBUG
			puts("Possible plain mp3");
#endif
			icy200_state = ICY_MP3;
		}
	}
	if ((icy200_state == ICY_MP3) && (strlen(icy200_buffer) == 0)) {
#ifdef ICY200_DEBUG
		puts("Plain mp3");
#endif
		icy200_state = ICY_MP3DATA;
	}
	char * clength = icy200_getafter(icy200_buffer, PSTR("Content-Length:"));
	if (clength) {
		icy200_filesize(atol(clength));
	}
	return 0;
}

static void icy200_decoderedirect(void) {
	char * location = icy200_getafter(icy200_buffer, PSTR("Location:"));
	if (location) {
		icy200_redirect(location);
	}
}

static void icy200_decodeplaylist(void) {
	char * location = icy200_getafter(icy200_buffer, PSTR("File1="));
	if (location) {
		icy200_redirect(location);
	}
	if ((icy200_state == ICY_PLAYLISTB) && (strncmp_P(icy200_buffer, PSTR("http://"), 7) == 0)) {
		icy200_redirect(icy200_buffer);
	}
	if (strcmp_P(icy200_buffer, PSTR("#EXTM3U")) == 0) {
		icy200_state = ICY_PLAYLISTB;
	}
}

static size_t icy200_decodemeta(void) {
	if (!icy200_buffer) {
		uint8_t c = 0;
		uint8_t readed = icy200_getdata(&c, 1);
		if (readed == 0) {
			return 0; //no 1. data byte -> retry
		}
		uint16_t maxlen = c*16;
		if (maxlen) {
			icy200_buffer = s_cmalloc(maxlen+1);
			icy200_buffersize = maxlen+1;
			icy200_bufferpos = 0;
		}
	}
	if (icy200_buffer) {
		uint16_t maxread = icy200_buffersize-1;
		while (icy200_bufferpos < maxread) {
			uint16_t nextread = maxread-icy200_bufferpos;
			if (nextread > 512) {
				nextread = 512;
			}
			size_t readed = icy200_getdata((uint8_t *)icy200_buffer+icy200_bufferpos, nextread);
			if (readed <= 0) {
				return 0; //no data byte -> retry
			}
			icy200_bufferpos += readed;
			if (icy200_bufferpos == maxread) { //we got everything, now analyze
				icy200_buffer[maxread] = '\0';
				icy200_extract(icy200_buffer);
				free(icy200_buffer);
				icy200_buffer = NULL;
				return icy200_metaint; //the normal way for a return
			}
		}
	}
	return icy200_metaint; //happens if there are no new (changed) data
}

size_t icy200_decode(void) {
	if (icy200_state == ICY_MP3DATA) { //no need to do something
		return 32700;
	}
	if (icy200_state == ICY_ICYDATA) { //update meta information of the stream
		return icy200_decodemeta();
	}
	if (icy200_state == ICY_INVALID) { //do not read further data
		return 0;
	}
	//try to read a line - needed for all other states
	if (!icy200_buffer) {
		icy200_buffer = s_cmalloc(ICY200_BUFFSIZE*sizeof(char));
		icy200_buffersize = ICY200_BUFFSIZE*sizeof(char);
		icy200_bufferpos = 0;
	}
	while (((icy200_bufferpos) < icy200_buffersize) && (icy200_abort)) {
		uint8_t c;
		size_t s = icy200_getdata(&c, 1);
		if (s != 1) {
			return 0; //no data, better next time
		}
		icy200_abort--;
		if (c == '\n') { //line is complete
			icy200_buffer[icy200_bufferpos] = '\0';
			size_t nextcall = 0;
			if ((icy200_state == ICY_UNDECIDED) || (icy200_state == ICY_MP3)) {
				nextcall = icy200_decidetype();
			} else if (icy200_state == ICY_ICYHEADER) {
				nextcall = icy200_decodeinitialmeta();
			} else if (icy200_state == ICY_REDIRECT) {
				icy200_decoderedirect();
			} else if ((icy200_state == ICY_PLAYLIST) || (icy200_state == ICY_PLAYLISTB)) {
				icy200_decodeplaylist();
			}
			free(icy200_buffer);
			icy200_buffer = NULL;
			return nextcall;
		} else if (c != '\r') {
			icy200_buffer[icy200_bufferpos++] = c;
		}
	}
	if (icy200_abort == 0) {
		icy200_state = ICY_INVALID;
		puts("Abort");
		icy200_stop();
	}
	if (icy200_buffer) {
		free(icy200_buffer);
		icy200_buffer = NULL;
	}
	return 0;
}
