/* Infplay
  (c) 2009 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <stdio.h>
#include <dev/board.h>
#include <compiler.h>
#include <sys/timer.h>
#include <avr/pgmspace.h>

#include "../bigbuff.h"
//for F_CPU:
#include "../infplay.h"
#include <util/delay.h>
#include "vs1002drv.h"
#include "../error.h"

//defined in dev/mmcard.c
extern HANDLE myspimutex;

/* Pins known to work:
	Vcc
	GND
	dreq
	Reset (active low)
	cs
	sck
	miso
	mosi
	dcs (previously this pin had problems)
*/

//bit 0: technical pause: buffer underum or similar
//bit 1: user pause
//bit 3: bigbuff write lock
volatile uint8_t mp3_pause;

void mp3dev_pause1(uint8_t val) {
	uint8_t p = mp3_pause;
	p &= 0xfe;
	p |= val & 1;
	mp3_pause = p;
}

uint8_t mp3dev_pause1_get(void) {
	return mp3_pause & 0x1;
}

void mp3dev_pause2(uint8_t val) {
	uint8_t p = mp3_pause;
	p &= 0xfd;
	p |= val & 2;
	mp3_pause = p;
}

uint8_t mp3dev_pause2_get(void) {
	return mp3_pause & 0x2;
}

// 0: continue, FF: pause
void mp3dev_pause3(uint8_t val) {
	uint8_t p = mp3_pause;
	p &= 0xfb;
	p |= val & 4;
	mp3_pause = p;
}

uint8_t volatile fastbuff[32];
uint8_t volatile fastbuffmax; //normally 32
uint8_t volatile fastbuffcnt;

void mp3dev_data_pause(void) {
	//stop everything
	cli();
	uint8_t datastate = TCCR3B;
	TCCR3B = 0;	     //stop timer
	uint8_t datastate2 = SPCR;
	cbi(SPCR, SPIE); //clear possible SPI interrupt
	sbi(ETIFR, TOV3); //clear possible waiting timer interrupt
	sei();

	if ((datastate == 0) && ((datastate2 & (1<< SPIE)) == 0)) {
		error_general(5); //means we were already in data_pause....
		//disable data select
		sbi(PORTB, DCS_PIN);
		return;
	}
	if ((datastate) && (datastate2 & (1<< SPIE))) {
		error_general(6);
	}
	/* It would be very difficult to determine if the SPI bus is still in use,
	   because, the hardware does not have a "busy" bit. And the settings in TCCR3B and
	   SPCR are not suitable for determining if a transmit is in progress. It turned out
	   that simply waiting is the best solution.
	*/
	_delay_us(4.5); //one byte is shifted in 4us @ 2Mhz
	/* Second: if SPSR is not read here, the next I/O routine may get a
		set SPIF flag, resulting in invalid data. */
	if (SPSR & (1<<SPIF)) {
		asm volatile("nop");
	}
	//disable data select
	sbi(PORTB, DCS_PIN);
}

void mp3dev_data_start(void) {
	//set up i/o ports
	*((volatile uint8_t *)(ARTHERCPLDSPI)) = 0xFF; /* Arthernet SPI */
	/*Set up correct spi settings
		according to the olimex example, they need more than 600kHz and less than
		1,2MHz, they used 1MHz, we had success with 2MHZ */
	outb(SPCR, (1 << SPE) | (1 << MSTR) | (1 << SPR0));
	outb(SPSR, (1 << SPI2X));
	cbi(PORTB, DCS_PIN);
	if (fastbuffmax <= fastbuffcnt) { //determine which state to restore
		outb(TCNT3, -1); //acts like a hardware interrupt
		outb(TCCR3B, 1<<CS30);
	} else {
		/* The interrupt blocking is needed, because there may be another interrupt between
			 these three lines, resulting in lost interrupts or simmilar or repetition of a byte,
			 because fastbuffcnt++ is not atomic.
		*/
		uint8_t sreg = SREG;
		cli();
		SPDR = fastbuff[fastbuffcnt]; //there may be an int between these two commands, re
		sbi(SPCR, SPIE);
		fastbuffcnt++;
		SREG = sreg;
	}
}

SIGNAL(SIG_OVERFLOW3) {
	uint8_t buff_block_rpl = buff_block_rp;
	uint16_t buff_byte_rpl = buff_byte_rp;
	uint8_t  buff_block_wpl = buff_block_wp;
	uint16_t buff_byte_wpl = buff_byte_wp;
	if (bit_is_set(PINB, 6) && (!mp3_pause) &&
	   ((buff_block_rpl != buff_block_wpl) || (buff_byte_rpl != buff_byte_wpl))) { //transfer mp3 data
		*((volatile uint8_t *)(ARTHERCPLDSTART)) = buff_block_rpl << 4;
		uint8_t i = 0;
		while (i < 32) {
			uint8_t data = *((uint8_t *)buff_byte_rpl);
			fastbuff[i] = data;
			i++;
			buff_byte_rpl++;
			if (!buff_byte_rpl) {
				buff_byte_rpl = NUTBANK_START;
				buff_block_rpl++;
				if (buff_block_rpl > BUFF_ENDBLOCK) {
					buff_block_rpl = BUFF_STARTBLOCK;
				}
				buff_block_rp = buff_block_rpl;
				*((volatile uint8_t *)(ARTHERCPLDSTART)) = buff_block_rpl << 4;
			}
			if ((buff_block_rpl == buff_block_wpl) && (buff_byte_rpl == buff_byte_wpl)) {
				break;
			}
		}
		buff_byte_rp = buff_byte_rpl;
		*((volatile uint8_t *)(ARTHERCPLDSTART)) = bank_c;
		sbi(SPCR, SPIE);
		SPDR = fastbuff[0];
		fastbuffcnt = 1;
		fastbuffmax = i;
		TCCR3B = 0;
	} else {
		TCNT3 = -8000; //retry in 8000 ticks
	}
}

uint8_t regbuff[4]; //faster than the external RAM
/* System clocks:
   prologue: 13
   code: 22
   epilogue: 13
   sum: 48
*/
//SIGNAL(SIG_SPI) __attribute__((naked));

ISR(SIG_SPI, ISR_NAKED) {
	asm volatile (
	//prologue
	"sts regbuff, r30""\n\t"
	"sts (regbuff)+1, r31""\n\t"
	"in r30, __SREG__""\n\t"
	"sts (regbuff)+2, r30""\n\t"
	"sts (regbuff)+3, r29""\n\t"
	//code
	"lds r29, fastbuffcnt""\n\t"  // if (fbct < fastbuffmax) {
	"lds r30, fastbuffmax""\n\t"
	"cp r29, r30""\n\t"
	"brsh 1f""\n\t"
	"ldi r30, lo8(fastbuff)""\n\t"// SPDR = fastbuff[fastbuffcnt++];
	"ldi r31, hi8(fastbuff)""\n\t"
	"add r30, r29""\n\t"
	"inc r29""\n\t"               //luckily this does not modify the carry
	"sts fastbuffcnt, r29""\n\t"
	"clr r29""\n\t"
	"adc r31, r29""\n\t"
	"ld r29, Z""\n\t"
	"out %0, r29""\n\t"
	"rjmp 2f""\n\t"               // }
	"1:""\n\t"                    // else {
	"ldi r30, lo8(-100)""\n\t"    // TCNT3 = -100;
	"ldi r31, hi8(-100)""\n\t"
	"sts %1, r31""\n\t"
	"sts %2, r30""\n\t"
	"ldi r30,lo8(1)""\n\t"        // TCCR3B = 1<<CS30;
	"sts %3, r30""\n\t"
	"cbi %4,7""\n\t"              // cbi(SPCR, SPIE);
	"2:""\n\t"                    // }
	//epilogue
	"lds r29, (regbuff)+3""\n\t"
	"lds r30, (regbuff)+2""\n\t"
	"out __SREG__, r30""\n\t"
	"lds r31, (regbuff)+1""\n\t"
	"lds r30, (regbuff)+0"
	: : "M" (_SFR_IO_ADDR(SPDR)), "M" (_SFR_MEM_ADDR(TCNT3H)),
             "M" (_SFR_MEM_ADDR(TCNT3L)), "M" (_SFR_MEM_ADDR(TCCR3B)),
             "M" (_SFR_IO_ADDR(SPCR)));
	reti();
}

/* a run takes approx: prologue: 39, calc: 17, epiloge: 39, sum: 95

possible source for the still annoying errors:
	SPDR = fastbuff[fbct];
	fbct++;
	fastbuffcnt = fbct;
	if (fbct >= fastbuffmax) {
		TCNT3 = -100; //overflow in 100 clocks -> current SPI must have finished
		TCCR3B = 1<<CS30;
		cbi(SPCR, SPIE);
	}
	Produces tons of errors, but this should not be the case in theory

 */
/*
SIGNAL(SIG_SPI) {
	uint8_t fbct = fastbuffcnt;
	if (fbct < fastbuffmax) {
		SPDR = fastbuff[fbct];
		fbct++;
		fastbuffcnt = fbct;
	} else {
		TCNT3 = -100; //overflow in 100 clocks -> current SPI must have finished
		TCCR3B = 1<<CS30;
		cbi(SPCR, SPIE);
	}
}
*/

uint8_t mp3dev_dreq_get(void) {
	if (bit_is_set(PINB, 6)) {
		return 1;
	} else {
		//puts("encf");
		return 0;
	}
}

void mp3dev_command_select(void) {
	NutEventWait(&myspimutex, NUT_WAIT_INFINITE);
	mp3dev_data_pause();
	sbi(PORTB, DCS_PIN);
	*((volatile uint8_t *)(ARTHERCPLDSPI)) = 0xFF; /* Arthernet SPI */
	/*Set up correct spi settings
		according to the olimex example, they need more than 600kHz and less than
		1,2MHz, they used 1MHz, we had success with 2MHz */
	outb(SPCR, (1 << SPE) | (1 << MSTR) | (1 << SPR0));
	outb(SPSR, (1 << SPI2X));
	*((volatile uint8_t *)(ARTHERCPLDSPI)) = 0xBF; /* Arthernet SPI CS6 */
	uint8_t waiting = 255;
	while (!mp3dev_dreq_get()) { //wait for dreq to get high -> can get at least 32 Byte data/commands
		if (!waiting) {
			error_general(4); //TODO: force a reset of the mp3 decoder
			break;
		}
		waiting--;
		/* Do not make a context switch here, as this may take soooo long, that the mp3
			buffer goes empty, because we already stopped the interrupt filling at this point.
		*/
		_delay_us(100.0);
	}
}

void mp3dev_deselect(void) {
	sbi(PORTB, DCS_PIN);
	*((volatile uint8_t *)(ARTHERCPLDSPI)) = 0xFF; /* Arthernet SPI */
	//restore settings for the mulit media card
	mp3dev_data_start();
	NutEventPostAsync(&myspimutex);
}

static uint8_t mp3dev_io(uint8_t val) {
	outb(SPDR, val);               /* Put byte to send in SPDR, which initiates xmit  */
	uint16_t loop = 0;
	while(!(SPSR & (1<<SPIF))) { /* wait for completion */
		loop--;
		if (!loop) {
			error_general(1);
			break;
		}
	}
	uint8_t ret = inb(SPDR);
	return ret;               /* return with byte shifted in from slave */
}

void mp3dev_sendcommand(uint8_t addr, uint16_t data) {
	mp3dev_command_select();
	mp3dev_io(0x02);     // Write
	mp3dev_io(addr);
	mp3dev_io(data >> 8);
	mp3dev_io(data & 0xff);
	mp3dev_deselect();
	_delay_us(7.0);
}

uint16_t mp3dev_getreg(uint8_t addr) {
	mp3dev_command_select();
	mp3dev_io(0x03);     // Read
	mp3dev_io(addr);
	uint16_t data = mp3dev_io(0x00) << 8;
	data |= mp3dev_io(0x00);
	mp3dev_deselect();
	return data;
}

void mp3dev_volume(uint8_t vol) {
	mp3dev_sendcommand(0x0B, (vol<<8) | vol);
}

void mp3dev_bass(uint8_t state) {
	uint8_t cmd;
	if (state) {
		cmd = 0x00C4; //+12db @ 40Hz (+5db was not hearable)
	} else
		cmd = 0;
	mp3dev_sendcommand(0x02, cmd);
}

uint16_t mp3dev_played;
uint16_t mp3dev_lastval;
uint8_t mp3dev_playoverflow;

/*
NOTE: Tests have shown that the register overflows at 2^14 and not as expected
by 2^16.
This implemementation works around this. However, to work properly, the
function must be called at least after every ~4 hours of playing.
NOTE: It looks like there are very rare cases, were wrong bits are read.
(observed once in ~100 hours). This is bad if this results in a wrong
mp3dev_playoverflow update. To work around this, the data are read twiche.
The second readout must be equal or +1 greater than the first readout in order
to be accepted. Between the two readouts, the counter could be overflow.
*/
uint16_t mp3dev_playtimeget(void) {
	uint16_t vsplaytime = mp3dev_getreg(0x04);
	uint16_t vsplaytime2 = mp3dev_getreg(0x04);
	if ((vsplaytime == vsplaytime2) ||
		  (vsplaytime == ((vsplaytime2-1) & 0x3FFF))) {
		if (vsplaytime < mp3dev_lastval) {
			mp3dev_playoverflow++;
		}
		mp3dev_lastval = vsplaytime;
	}
	vsplaytime |= mp3dev_playoverflow<<14;
	return (vsplaytime - mp3dev_played);
}

void mp3dev_playtimereset(void) {
	/*according to the datasheet, the time must be set by writing the register
    twice, but it does not work (and the notes differ within the datasheet).
    So we use a trick here:
    Reading the time and substract as offset later.
  */
	mp3dev_played = mp3dev_getreg(0x04);
	mp3dev_lastval = mp3dev_played;
	mp3dev_playoverflow = 0;
	/* Not working code:
	uint8_t i;
	for (i = 0; i < 50; i++) {
		mp3dev_sendcommand(0x04, 0x0000);
	}
	*/
}

/*Sample rates in kb/s, first 16 numbers: ID0,1,2; lower 16 numbers: id3
according to the datasheet, rate must be 1 ... 14 0 and 15 are invalid.
the calling functions uses the returned value 0 as "invalid

The info is somewhat useless if there are variable bit rates
*/

uint8_t PROGMEM kbrates[] = {
0, 1, 2, 3, 4, 5,  6,  7,  8, 10, 12, 14, 16, 18, 20, 0,
0, 4, 5, 6, 7, 8, 10, 12, 14, 16, 20, 24, 28, 32, 40, 0};

//gets the rate in kb/s (bytes, not bits)
uint8_t mp3dev_datarateget(void) {
	uint8_t layer = mp3dev_getreg(0x09) >> 3 & 0x3; //hdat1
	uint8_t rate = mp3dev_getreg(0x08) >> 12; //hdata0
	if (layer == 3)
		rate += 16;
	return pgm_read_byte(kbrates+rate);
}

void mp3dev_printregs(void) {
	uint8_t i;
	for (i = 0; i <= 0x0B; i++) {
		printf_P(PSTR("%i: %u\n\r"), i, mp3dev_getreg(i));
	}
}

/*Remember to set the volume and bass boost after calling. */
void mp3dev_softarereset(void) {
	mp3dev_sendcommand(0x00, 0x0804);
	NutSleep(1);
	//set up
	mp3dev_sendcommand(0x00, 0x0800); //default, but to be sure
	mp3dev_sendcommand(0x03, 0x9800); //12.288MHz (with clock doubler)
	NutSleep(100);
}

/* returns if failed. 1: failed, 0: success */
uint8_t mp3dev_init(void) {
	//set port directions of SPI and control signals
	outb(PORTB, 0x00);
	sbi(PORTB, DCS_PIN);
	outb(DDRB, 0x27); //NOTE this values may be set by mmc init again
	*((volatile uint8_t *)(ARTHERCPLDSPI)) = 0xFF; /* Arthernet SPI */
	//set up the timer3
	outb(TCCR3A, 0);
	outb(TCCR3B, 0);
	outb(TCCR3C, 0);
	TCNT3 = -100; //overflow in 100 clocks -> current SPI must have finished
	TCCR3B = 1<<CS30;
	sbi(ETIMSK, TOIE3);
	//init mp3 decoder: // Default - > SM_SDINEW on, MSB first, SM_SDISHARED off
	mp3_pause = 1; //only technical pause enabled
	//make software reset (needed for a system reboot)
	mp3dev_softarereset();
	mp3dev_volume(0x20);
	//mp3dev_printregs();
	//test if we could at least read one register correctly:
	if (mp3dev_getreg(0x03) == 0x9800) {
		return 0; //write was successful
	} else
		return 1; // something went wrong
}

/* BUG: The beep test always starts correctly, but often does not stop...
*/
#if 0
void mp3dev_beeptest(void) {
	//enables test mode
	mp3dev_sendcommand(0x00, 0x0820);
	//enters the sine test with 5168Hz
	mp3dev_data_select();
	mp3dev_io(0x53);
	mp3dev_io(0xEF);
	mp3dev_io(0x6E);
	mp3dev_io(0x7E);
	mp3dev_io(0x00);
	mp3dev_io(0x00);
	mp3dev_io(0x00);
	mp3dev_io(0x00);
	mp3dev_deselect();
	NutSleep(1000);
	//stop sine test
	mp3dev_data_select();
	mp3dev_io(0x45);
	mp3dev_io(0x78);
	mp3dev_io(0x69);
	mp3dev_io(0x74);
	//datasheet says: send 4 zeros, application note vs10xxan.pdf on page 14 does not have them
	mp3dev_io(0x00);
	mp3dev_io(0x00);
	mp3dev_io(0x00);
	mp3dev_io(0x00);
	mp3dev_deselect();
	//disables test mode
	mp3dev_sendcommand(0x00, 0x0800);
	NutSleep(1000);
}
#endif
