/*
 * Copyright 2009 by Malte Marwedel
 * Many parts of the code are based on sbi_mmcdrv.c from egnite GmbH
 * and the spimmcdrv.c from Michael Fischer, who got some functions
 * from Sylvain Bissonnette.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

/*!
 * $Log$
 * Revision 1.00  2009/04/30 09:38:15  Malte Marwedel
 * Initial version
 */


#include <cfg/arch.h>
#include <dev/board.h>
#include <sys/timer.h>
#include <dev/mmcard.h>

#include "vs1002drv.h"
#include "../error.h"

#define MMC_DEV_NAME    {'M', 'M', 'C', '0', 0, 0, 0, 0, 0}

#define SPIDDR        DDRB
#define SPIPORT       PORTB
#define SCLK          0x02
#define MOSI          0x04
#define MISO          0x08
#define SSP            0x01

typedef struct _MMCDCB {
    int cs_selected;
    uint8_t written;
} MMCDCB;

static MMCDCB mmc_dcb;

/*!
 * \brief Activate or deactivate chip select on card slot 0.
 *
 * \param on The card will be selected if 1, deselected if 0.
 *           Any other value can be used to query the status.
 *
 * \return Previous select status. 1 if selected, 0 otherwise.
 */

static int SpiMmCardSelect(int on)
{
    /* MMC select is low active. */
    if (on == 1) {
        mp3dev_data_pause();
#if defined(ARTHERNET1)
        outb(SPCR, (1 << SPE) | (1 << MSTR));  /* Enable SPI as master, set clk divider and max speed */
        outb(SPSR, (1 << SPI2X)); /* Double SPI speed */
        *((volatile uint8_t *)(ARTHERCPLDSPI)) = 0xEF; /* Arthernet SPI CS4 */
        sbi(PORTB, DCS_PIN);
#else
#warning "MMC chip select not implemented on this board"
#endif
        mmc_dcb.cs_selected = 1;
    } else if (on == 0) {
#if defined(ARTHERNET1)
        *((volatile uint8_t *)(ARTHERCPLDSPI)) = 0xFF; /* Arthernet SPI CS4 */
#else
#warning "MMC chip deselect not implemented on this board"
#endif
        mp3dev_data_start();
        mmc_dcb.cs_selected = 0;
    }
    return mmc_dcb.cs_selected;
}

/*!
 * \brief Check if card is available in slot 0.
 *
 * \return 0 if no card is detected, 1 if a card is available.
 */
static int SpiMmCardAvail(void)
{
    if ((inb(PORTB) & (1<<PB7)) == 0) {
        return 1;
    }
    puts_P(PSTR("No mmc"));
    return 0;
}

/*!
 * \brief Initialize the card in slot 0.
 *
 * Called by the MMC driver during card reset.
 *
 * \return 0 on success, -1 if no card is detected.
 */
static int SpiMmCardInit(void)
{
    SpiMmCardSelect(0);
    if (SpiMmCardAvail()) {
        return 0;
    }
    return -1;
}


/*!
 * \brief Read the previously received byte and transmit a new one.
 *
 * \param val Byte to transmit.
 *
 * \return Last byte received.
 */
static uint8_t SpiMmCardIo(uint8_t val)
{
    outb(SPDR, val);               /* Put byte to send in SPDR, which initiates xmit  */
    uint16_t loop = 0;
    while(!(SPSR & (1<<SPIF))) { /* wait for completion */
      loop--;
      if (!loop) {
        error_general(2);
        break;
      }
    }
    uint8_t ret = inb(SPDR);
    return ret;               /* return with byte shifted in from slave */
}


/*!
 * \brief Check if card in slot 0 is write protected.
 *
 * \todo Not implemented.
 *
 * \return Always 0.
 */
static int SpiMmCardWrProt(void)
{
    return 0;
}


/*!
 * \brief Initialize MMC hardware interface, by using the hardware SPI.
 *
 * This function is automatically executed during device
 * registration via NutRegisterDevice().
 *
 * \param dev Identifies the device to initialize.
 */
static int SpiMmcIfcInit(NUTDEVICE * dev)
{
		/* NOTE: If values are changed here, they have to be changed
		   in the vs1002drv.c too, because it overwrites the values
		   and restores fixed values later */
    SPIDDR |= SCLK;
    SPIDDR |= MOSI;
    SPIDDR |= SSP;
    SPIDDR &= ~MISO;
    //outb(SPIDDR, SCLK + MOSI + SSP); //overwrites other pin values as well -> not using
    outb(SPIPORT, 0x00);
    NutSleep(250);
    SpiMmCardSelect(0);
    NutSleep(250);
    mp3dev_data_pause();
    outb(SPCR, (1 << SPE) | (1 << MSTR));  /* Enable SPI as master, set clk divider and max speed */
    outb(SPSR, (1 << SPI2X)); /* Double SPI speed */
    /* start off with 80 bits of high data with card deselected */
    int i;
    for (i = 0; i < 10; i++) {
      SpiMmCardIo(0xff);
    }
    return MmCardDevInit(dev);
}

uint8_t mmcaddrmode; //for SD-HC

static int SpiGetAddrMode(void) {
	return mmcaddrmode;
}

static int SpiSetAddrMode(u_char value) {
  mmcaddrmode = value;
  return 0; //mmcard never reads this value
}

static MMCIFC mmc_ifc = {
    SpiMmCardInit,             /*!< mmcifc_in */
    SpiMmCardIo,               /*!< mmcifc_io */
    SpiMmCardSelect,           /*!< mmcifc_cs */
    SpiMmCardAvail,            /*!< mmcifc_cd */
    SpiMmCardWrProt,            /*!< mmcifc_wp */
    SpiSetAddrMode, //for SD-HC
    SpiGetAddrMode, //for SD-HC
};

static int SPIBlockWrite(NUTFILE * nfp, CONST void *buffer, int num) {
     mmc_dcb.written = 1;
     return MmCardBlockWrite(nfp, buffer, num);
}

uint8_t Spi_haswritten(void) {
     return mmc_dcb.written;
}

void Spi_writeclear(void) {
     mmc_dcb.written = 0;
}


NUTDEVICE devAVRSPIMmc0 = {
    0,                  /*!< Pointer to next device, dev_next. */
    MMC_DEV_NAME,       /*!< Unique device name, dev_name. */
    0,                  /*!< Type of device, dev_type. Obsolete. */
    0,                  /*!< Base address, dev_base. Unused. */
    0,                  /*!< First interrupt number, dev_irq. Unused. */
    &mmc_ifc,           /*!< Interface control block, dev_icb. */
    &mmc_dcb,           /*!< Driver control block used by the low level part, dev_dcb. */
    SpiMmcIfcInit,      /*!< Driver initialization routine, dev_init. */
    MmCardIOCtl,        /*!< Driver specific control function, dev_ioctl. */
    MmCardBlockRead,    /*!< Read data from a file, dev_read. */
    SPIBlockWrite,   /*!< Write data to a file, dev_write. */
#ifdef __HARVARD_ARCH__
    MmCardBlockWrite_P, /*!< Write data from program space to a file, dev_write_P. */
#endif
    MmCardMount,        /*!< Mount a file system, dev_open. */
    MmCardUnmount,      /*!< Unmount a file system, dev_close. */
    0                   /*!< Return file size, dev_size. */
};


