/* KS0108 driver
  (c) 2009 by Malte Marwedel
  Vesion 1.0
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
Pin connection:
Atmega128
PORTD: 8 Bit data port
PE2: E
PE5: RS
PE6: CS1
PE7: CS2
*/

#include <avr/io.h>
//should define F_CPU:
#include "ks0108drv.h"
#include <util/delay.h>
#include "../memmapper.h"

#define SIZEX_DATA SIZEX
#define SIZEY_DATA (SIZEY+7)/8

#define SIZE_MEM (SIZEX_DATA*SIZEY_DATA)

unsigned char * display_bitmap; //[SIZEY_DATA][SIZEX_DATA];

void ks0108_clear(void) {
	int i;
	for (i = 0; i < SIZE_MEM; i++) {
		display_bitmap[i] = 0;
	}
}

void ks0108_set_pixel(unsigned char x, unsigned char y, unsigned char color) {
	if ((x < SIZEX) && (y < SIZEY)) {
		//using a pointer here instead of two array accesses saves 54 byte flash!
		unsigned char * addr = display_bitmap+(y/8)*SIZEX_DATA+x;
		unsigned char b = *addr;
		if (color & 1) {
			b |= (1<< (y % 8));
		} else
			b &= ~(1<<(y % 8));
		*addr = b;
	}
}

void ks0108_pulse(void)__attribute__ ((noinline));
void ks0108_pulse(void) {
	/* Values according to the datasheet:
	   enable high and low width: 450ns
	   enable min cycle: 1000ns
	   adress set-up time: 140ns
	   data set-up time: 200ns

	   a nop @ 16MHZ = 62ns -> would need at least 8 - time to function call
	*/
	_delay_us(2.5);
	LCD_CONTROLPORT |= (1<<LCD_PIN_E); //enable
	_delay_us(2.5); //enable l pulse width = 117ns according to the datasheet
	LCD_CONTROLPORT &= ~(1<<LCD_PIN_E); //disableselect disabled
}

void ks0108_send_data(unsigned char data) {
	LCD_DATAPORT = data;
	LCD_CONTROLPORT |= (1<<LCD_PIN_RS); //select data
	ks0108_pulse();
}

void ks0108_send_sys(unsigned char command) {
	LCD_DATAPORT = command;
	LCD_CONTROLPORT &= ~(1<<LCD_PIN_RS); //select command
	LCD_CONTROLPORT |= (1<<LCD_PIN_CS1) | (1<<LCD_PIN_CS2);
	ks0108_pulse();
}

void ks0108_flush(void) {
	unsigned char page, x;
	for (page = 0; page < 8; page++) {
		ks0108_send_sys(0xB8 | page); //set page address
		ks0108_send_sys(0x40); //data start at address 0
		for (x = 0; x < SIZEX_DATA/2; x++) {
			LCD_CONTROLPORT &= ~(1<<LCD_PIN_CS2);
			LCD_CONTROLPORT |= (1<<LCD_PIN_CS1);
			ks0108_send_data(*(display_bitmap+page*SIZEX_DATA+x));
			LCD_CONTROLPORT &= ~(1<<LCD_PIN_CS1);
			LCD_CONTROLPORT |= (1<<LCD_PIN_CS2);
			ks0108_send_data(*(display_bitmap+page*SIZEX_DATA+x+SIZEX_DATA/2));
		}
	}
}

void ks0108_init(void) {
	LCD_DATADDR = 0xff; //pull-up alone does not work with the pollin LPT adaptor
	LCD_CONTROLDDR |= (1<<LCD_PIN_CS1) | (1<<LCD_PIN_CS2) | (1<<LCD_PIN_E) | (1<<LCD_PIN_RS);
	display_bitmap = s_cmalloc(SIZE_MEM);
	ks0108_send_sys(0x3f); //display on
	ks0108_send_sys(0xc0); //0 line as start position
	ks0108_clear();
}

