/*
SED 1560 driver.
By Malte Marwedel (c) 2009. Vesion 1.0
Idea based on the code from pollin.

Pin connection:
Atmega128
PORTD: 8 Bit data port
PE2: /CS1
PE3: R/W
PE6: A0
PE7: /RES
*/

#include <avr/io.h>
//should define F_CPU:
#include "sed1560-driver.h"
#include <util/delay.h>


#define SIZEX_DATA SIZEX
#define SIZEY_DATA SIZEY/8+1

unsigned char display_bitmap[SIZEY_DATA][SIZEX_DATA];
unsigned char sed1560_contrast = 0xf;

void sed1560_clear(void) {
	unsigned char i, j;
	for (j = 0; j < SIZEY_DATA; j++) {
		for (i = 0; i < SIZEX_DATA; i++) {
			display_bitmap[j][i] = 0;
		}
	}
}

void sed1560_set_pixel(unsigned char x, unsigned char y, unsigned char color) {
	if ((x < SIZEX) && (y < SIZEY)) {
		//using a pointer here instead of two array accesses saves 54 byte flash!
		unsigned char * addr = &(display_bitmap[y/8][x]);
		unsigned char b = *addr;
		if (color & 1) {
			b |= (1<< (y % 8));
		} else
			b &= ~(1<<(y % 8));
		*addr = b;
	}
}

void sed1560_pulse(void)__attribute__ ((noinline));
void sed1560_pulse(void) {
	/*data setup time = 20ns according to the datasheet
	  a nop @ 16MHZ = 62ns -> as long as this function is not inlined,
	  enough waiting has been made, however, it looks like there are
	  timing problems, maybe with the level shifter or so. So adding a nop.*/
	_delay_us(1.0);
	LCD_CONTROLPORT &= ~(1<<LCD_PIN_CS1); //chip select enabled = low
	_delay_us(1.0); //enable l pulse width = 117ns according to the datasheet
	LCD_CONTROLPORT |= (1<<LCD_PIN_CS1); //chip select disabled = high
}

void sed1560_send_data(unsigned char data) {
	LCD_DATAPORT = data;
	LCD_CONTROLPORT |= (1<<LCD_PIN_A0); //select data
	sed1560_pulse();
}

void sed1560_send_sys(unsigned char command) {
	LCD_DATAPORT = command;
	LCD_CONTROLPORT &= ~(1<<LCD_PIN_A0); //select command
	sed1560_pulse();
}

void sed1560_flush(void) {
	unsigned char page, x;
	for (page = 0; page < 8; page++) {
		sed1560_send_sys(0xB0 | page); //set page address
		sed1560_send_sys(0x12); //data start at address 32
		sed1560_send_sys(0x00);
		for (x = 0; x < SIZEX_DATA; x++) {
			sed1560_send_data(display_bitmap[page][x]);
		}
	}
	//for working around some problems:
	sed1560_send_sys(0x40);
	sed1560_send_sys(0x20);
	sed1560_send_sys(0xcc);
	sed1560_send_sys(0xaf);
	//and as with this the display still went dark some times
	sed1560_send_sys(0x80 | sed1560_contrast);
}

void sed1560_set_contrast(uint8_t value) {
	sed1560_contrast = value & 0x1F;
	sed1560_send_sys(0x80 | sed1560_contrast);
}

void sed1560_init(void) {
	//bytes taken from the pollin manual
	unsigned char sedcommands[12] = {0x40, 0x20, 0xcc, 0xa0, 0xa9, 0xab, 0x25, 0xed, 0x8f, 0xa4, 0xaf, 0xa6};
	/* The steps are:
	0x40: Display line = 0 (would shift output)
	0x20: cancel reverse driving line  (may not be used with internal voltage)
	0xcc: output status c = 102x64 com/seg pins and scan direction
	0xa0: view not mirrored
	0xa9: lcd duty 64
	0xab: duty +1
	0x25: power on
	0xed: turn-on sequence complete
	0x8f: contrast = average
	0xa4: normal, no display test
	0xaf: display on
	0xa6: normal, not inverted
	Problems found while controlling:
	  1. The display shifts vertical at a small random amount
	  2. The display goes dark
	  This only happens while updating the content, and not if nothing gets changed
	*/

	LCD_DATADDR = 0xff; //pull-up alone does not work with the pollin LPT adaptor
	LCD_CONTROLDDR = (1<<LCD_PIN_CS1) | (1<<LCD_PIN_RW) | (1<<LCD_PIN_A0) | (1<<LCD_PIN_RES);
	LCD_CONTROLPORT &= ~(1<<LCD_PIN_RES); //reset
	LCD_CONTROLPORT &= ~(1<<LCD_PIN_RW); //read data, not write it
	LCD_CONTROLPORT |= (1<<LCD_PIN_CS1); //chip select disabled
	_delay_us(1.0); //according to the datasheet
	LCD_CONTROLPORT |= (1<<LCD_PIN_RES); //reset off
	_delay_us(1.0); //according to the datasheet
	unsigned char i;
	for (i = 0; i < 12; i++) {
		sed1560_send_sys(sedcommands[i]);
		if (i == 6)
			LCD_WAIT120; //at least 100ms according to the datasheet
	}
	//clear special symbols
	sed1560_send_sys(0xB0 | 8); //set page address
	sed1560_send_sys(0x12); //data start at address 32
	sed1560_send_sys(0x00);
	for (i = 0; i <= (117-32); i++) {
		sed1560_send_data(0);
	}
	//test pattern
#ifdef LCD_TEST
	LCD_WAIT1000;
	sed1560_send_sys(0xa5);
	LCD_WAIT1000;
	//_delay_loop_2(65535);
	//_delay_loop_2(65535);
	sed1560_send_sys(0xa4);
#endif
}

