/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <stdio.h>

#ifdef TEST_PC
#define _GNU_SOURCE
#endif

#include <string.h>
#include <inttypes.h>
#include <sys/types.h>

#include "fileselect.h"
#include "stringhelper.h"

#if defined TEST_PC || defined __NUT_EMULATION__

#include <dirent.h>

#else

//Nut/OS only has 0 or 1 as d_type, but Linux has more variants
#define DT_DIR 1

#endif

#ifndef TEST_PC

#include <avr/wdt.h>
#include "infplay.h"
#include "statestorage.h"
#include "menu-interpreter.h"
#include "memmapper.h"

#include "bigbuff.h"
#include "error.h"

#endif

#define MAXVISIBLECHARS 36
#define MAXBUFSIZE (NUTBANK_SIZE/2)
#define MAXPATHLEN 255

char * fs_textbuffer;
uint16_t fs_bufused;
char * fs_path;
char * fs_ending;
char * fs_ending2;
uint8_t fs_allowdeep;

#ifdef TEST_PC

#include "pctests/avrmapper.h"

#define NUTBANK_SIZE 32768
#define NUTBANK_START (s_cmalloc(sizeof(char)* MAXBUFSIZE))

#define banking_displayuse()
#define banking_displayrelease()
#define wdt_reset()
#define NutThreadYield()

char * fileselect_filepart(uint16_t index);

void error_diropen(char * path) {
	printf("Can't open dir '%s'\n", path);
}

int main(int argc, char ** argv) {
	if ((argc != 6) && (argc != 7)) {
		printf("Usage: PATH ALLOWDEPTH(0/1) SORT(0/1) SELECTNUM ENDING1 (ENDING2)\n");
		return 1;
	}
	fs_textbuffer = malloc(sizeof(char)*MAXBUFSIZE);
	if (fs_textbuffer == 0) {
		printf("No RAM free\n");
		return 1;
	}
	char * secondending = NULL;
	if (argc == 7) {
		secondending = argv[6];
	}
	char * 	filelist = fileselect_init(argv[1], argv[5], secondending, atoi(argv[2]), atoi(argv[3]));
	printf("==Filelist==\n%s\n====\n", filelist);
	char * x = fileselect_filepart(atoi(argv[4]));
	printf("filepart: '%s'\n", x);
	free(x);
	uint8_t isdir;
	char * selectionpath = fileselect_filepath(atoi(argv[4]),  &isdir);
	if (isdir) {
		printf("Selected directory with path: '%s'\n", selectionpath);
	} else {
		printf("Selected file with path: '%s'\n", selectionpath);
	}
	if (selectionpath) {
		char * filename = strrchr(selectionpath, '/');
		if (filename) {
			filename++;
		} else
			filename = selectionpath;
		uint16_t fileindex = fileselect_getindex(filename);
		printf("If searched for the filename '%s', %i is returned as index\n", filename, fileindex);
		free(selectionpath);
	}
	free(fs_textbuffer);
	return 0;
}

#endif


char * fileselect_getpath(void) {
	return fs_path;
}

/* There are some restrictions for the sorting algorithm here.
1. The algorithm must be in-place, as we dont have more memory free
2. Exchanging two elements directly near to each other is a lot easier, than
   elements with someother between them
3. Finding a spezific index is slow, but finding the next or previous is fast.
Shakesort is one of the algorithm, meeting this criteria and Bubblesort the
simpler variant of it, which is used here as saving program space is more important
than saving cpu cycles.
*/

static size_t fileselect_entrylen(char * start) {
	size_t len = strcspn(start, "\n");
	return len;
}

/* Makes a comparison if the list element matches the tocompare element.
   Takes care of the maximum lenght by cutting there.
 */
static int fileselect_elemmatch(char * buffpos, char * tocompare) {
	size_t lenc = strlen(tocompare);
	size_t lenb = fileselect_entrylen(buffpos);
	//does: lenc = min(max(lenc, lenb),MAXVISIBLECHARS);
	if (lenb > lenc) {
		lenc = lenb;
	}
	if (lenc > MAXVISIBLECHARS) {
		lenc = MAXVISIBLECHARS;
	}
	return strncmp(buffpos, tocompare, lenc);
}

static char * fileselect_compareexchange(char * first, char * second) {
	size_t lenf = fileselect_entrylen(first);
	size_t lens = fileselect_entrylen(second);
	size_t cmplen = lenf;
	if (cmplen > lens) {
		cmplen = lens;
	}
	int res = strncasecmp(first, second, cmplen);
	if (res <= 0) {
		return NULL;
	}
	//ok, it is larger, exchange the two
	char * firstdup = s_strndup(first, lenf);
	memmove(first, second, lens);
	*(first+lens) = '\n';
	memmove(first+lens+1, firstdup, lenf);
	free(firstdup);
	return first+lens+1;
}

static void fileselect_sort(void) {
uint8_t waschanged = 1;
char * first, * second;
	while (waschanged) {
		waschanged = 0;
		banking_displayuse();
		first = fs_textbuffer;
		second = strchr(first, '\n');
		while ((first != NULL) && (second != NULL)) {
			second++; //avoid the \n
			char * possec = fileselect_compareexchange(first, second);
			if (possec != NULL) {
				second = possec;
				waschanged = 1;
			}
			first = second;
			second = strchr(first, '\n');
		}
		banking_displayrelease();
		NutThreadYield();
	}
}

static uint8_t fileselect_accept(uint8_t allowdeep, struct dirent* entry, char * ending) {
	if (((allowdeep) && (entry->d_type == DT_DIR) && (strcmp(entry->d_name, "."))) ||
	    ((entry->d_type != DT_DIR) && endswith(entry->d_name, ending))) {
		return 1;
	}
	return 0;
}

void fileselect_append(char * text) {
	uint16_t tlen = strlen(text);
	if (tlen > MAXVISIBLECHARS)
		tlen = MAXVISIBLECHARS;
	if (fs_bufused + tlen >= MAXBUFSIZE-2) { //out of memory!
		return;
	}
	if (fs_bufused != 0) { //add newline, at old item, if not the first one
		fs_textbuffer[fs_bufused] = '\n';
		fs_bufused++;
	}
	strncpy(fs_textbuffer+fs_bufused, text, tlen);
	fs_bufused += tlen;
}

char * fileselect_init(char * path, char * ending, char * ending2, uint8_t allowdeep, uint8_t sort) {
	if (fs_textbuffer == NULL) {
#ifdef __NUT_EMULATION__
		fs_textbuffer = s_cmalloc(sizeof(char)* MAXBUFSIZE);
#else
		fs_textbuffer = (char *)NUTBANK_START; //external memory 2. bank
#endif
	}
	if (fs_path == NULL)
		fs_path = s_cmalloc(sizeof(char)*(MAXPATHLEN+1));
	if (fs_ending == NULL)
		fs_ending = s_cmalloc(sizeof(char)*10);
	if (fs_ending2 == NULL)
		fs_ending2 = s_cmalloc(sizeof(char)*10);
	fs_allowdeep = allowdeep;
	strncpy(fs_path, path, MAXPATHLEN);
	strncpy(fs_ending, ending, 9);
	//makes the code later simpler, if we can assume that fs_ending2 != NULL
	if (ending2 != NULL) {
		strncpy(fs_ending2, ending2, 9);
	} else {
		strncpy(fs_ending2, ending, 9);
	}
	fs_bufused = 0;
	//open directory
	DIR * d = opendir(path);
	//printf_P(PSTR("opendir(%s)\n"), path);
	if (d == NULL) {
		error_diropen(fs_path);
		error_message_P(PSTR("Open dir failed"));
		return NULL;
	}
	//go through the items
	struct dirent* entry;
	while ((entry = readdir(d)) != NULL) {
		if ((fileselect_accept(allowdeep, entry, fs_ending)) ||
		    (fileselect_accept(allowdeep, entry, fs_ending2))) {
			banking_displayuse();
			fileselect_append(entry->d_name);
			banking_displayrelease();
		}
	}
	closedir(d);
	banking_displayuse();
	fs_textbuffer[fs_bufused] = '\0';
	banking_displayrelease();
	if (sort) {
		fileselect_sort();
	}
	return fs_textbuffer;
}

/* Returns the selected file. Note: Only the short version limited to
   MAXVISIBLECHARS is returned. If the index is invalid, NULL is returned */
char * fileselect_filepart(uint16_t index) {
	if (fs_textbuffer == NULL) {
		error_general(13);
		return NULL;
	}
	//first, get the name
	banking_displayuse();
	char * pos = fs_textbuffer;
	while ((index) && (pos < fs_textbuffer+fs_bufused)) {
		if (*pos == '\n') {
			index--;
		}
		pos++;
	}
	char * res = NULL;
	if (*pos != '\0') {
		size_t len = fileselect_entrylen(pos);
		res = s_strndup(pos, len);
	}
	banking_displayrelease();
	return res;
}

char * fileselect_filepath(uint16_t index, uint8_t * isdir) {
	if (fs_path == NULL) {
		error_general(12);
		return NULL;
	}
	char * filepath = NULL;
	*isdir = 0;
	uint8_t rootlen = strlen(fs_path);
	//get name
	char * filepart = fileselect_filepart(index);
	if (filepart == NULL) {
		return NULL;
	}
	//open directory
	DIR * d = opendir(fs_path);
	if (d == NULL) {//if error
		error_diropen(fs_path);
		error_message_P(PSTR("Open dir failed"));
		return NULL;
	}
	//go through the items
	struct dirent* entry;
	while ((entry = readdir(d)) != NULL) {
#if defined TEST_PC || defined __NUT_EMULATION__
		uint16_t nlen = strlen(entry->d_name);
#else
		uint16_t nlen = entry->d_namlen;
#endif
		if (((fileselect_accept(1, entry, fs_ending)) ||
		     (fileselect_accept(1, entry, fs_ending2))) &&
		    (fileselect_elemmatch(entry->d_name, filepart) == 0 )) {
			if (entry->d_type == DT_DIR) {
				*isdir = 1;
			}
			filepath = s_cmalloc(sizeof(char)*(rootlen+nlen+2));
			strcpy(filepath, fs_path);
			path_append(filepath, entry->d_name, rootlen+nlen+2, 0);
			break;
		}
		wdt_reset(); //in the case this takes some longer time
	}
	closedir(d);
	free(filepart);
	if (filepath != NULL) {
		uint16_t plen = strlen(filepath);
		if (endswith(filepath, "/..")) {
			filepath[plen-3] = '\0';
			char * slash = strrchr(filepath, '/');
			if (slash) {
				*slash = '\0';
			}
		} else if (endswith(filepath, "/.")) {
			filepath[plen-1] = '\0';
		}
		if (endswith(filepath, "//")) {
			filepath[plen-1] = '\0';
		}
	}
	return filepath;
}

uint16_t fileselect_getindex(char * filename) {
	if ((filename == NULL) || (fs_textbuffer == NULL)) {
		return 0;
	}
	banking_displayuse();
	uint16_t index = 0;
	char * pos = fs_textbuffer;
	uint8_t found = fileselect_elemmatch(pos, filename);
	while ((pos < fs_textbuffer+fs_bufused) && (found != 0)) {
		if (*pos == '\n') {
			index++;
			found = fileselect_elemmatch(pos+1, filename);
		}
		pos++;
	}
	if (found) { //means not found
		index = 0;
	}
	banking_displayrelease();
	return index;
}


