/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "filemanager.h"
#include "stringhelper.h"

/*
Organization:

Entrys: Num
Path: P,P,P
Dir:int P,P,P,P,P,P,P,P

P = Position found by opendir

Organization within the array:
down to high: filepos, path
high to down: entrydir_t


*/

#define MANAGER_MAXDEPTH 10
#define MANAGER_MAXPATHLEN 500
#define MANAGER_NOENT 0xFFFF

//#define DEBUG

typedef struct entrydir {
	uint8_t depth;
	uint16_t entrys;
	uint16_t * path;
	uint16_t * filepos;
} entrydir_t;

#if defined(TEST_PC) || defined (__NUT_EMULATION__)
void * managermemory;
#else
void * managermemory = (void *)MAN_STARTADDR;
#endif

uint16_t * man_entryptr;
entrydir_t * man_minpathptr;
uint16_t * man_tempway;
char * man_temppath;
uint16_t man_dirnum;

uint8_t volatile man_buildup_busy = 1;
uint8_t man_abortrequest = 0;

#ifdef TEST_PC

#include "pctests/avrmapper.h"

#define MAN_LOCK
#define MAN_UNLOCK
#define MAN_BE_NICE

void error_diropen(char * path) {
	printf("Can't open dir '%s'\n", path);
}

void printdatastruct(void) {
	uint16_t i, j;
	entrydir_t * pathptr = (entrydir_t *)(managermemory+MANAGER_MAXMEM-1-sizeof(entrydir_t));
	printf("Dirs: %i\n", man_dirnum);
	fflush(stdout);
	for (i = 0; i < man_dirnum; i++) {
		printf("Dir %i, has way: ", i);
		for (j = 0; j < pathptr->depth; j++) {
			printf("%i, ", (pathptr->path)[j]);
		}
		printf("\n  Entrys: ");
		for (j = 0; j < pathptr->entrys; j++) {
			printf("%i, ", (pathptr->filepos)[j]);
		}
		printf("\n");
		pathptr--;
	}
}

int main(int argc, char ** argv) {
	if (argc < 2) {
		printf("Usage: PATHNAME [sort(0/1)]\n");
		return 1;
	}
	if (strlen(argv[1]) == 0) {
		printf("Pathname too short\n");
		return 1;
	}
	if (argv[1][strlen(argv[1])-1] != '/') {
		printf("Pathname must end with a slash (/)\n");
		return 1;
	}
	managermemory = malloc(MANAGER_MAXMEM*sizeof(uint8_t));
	if (managermemory) {
		chdir(argv[1]);
		manager_buildup(argv[1]);
		if ((argc == 3) && (argv[2][0] == '1')) {
			manager_sortnames(argv[1]);
		}
		printdatastruct();
		manager_fileidget("./123.mp3");
		uint16_t id = manager_fileidget("./abc/def.mp3");
		uint16_t id2 = manager_fileidget("./abc/ghi.mp3");
		char * f = manager_filenameget(id, argv[1]);
		printf("Id %i has name: '%s'\n", id, f);
		free(f);
		f = manager_filenameget(id2, argv[1]);
		printf("Id %i has name: '%s'\n", id, f);
		free(f);
		printf("Files: %i\n", manager_filecount());
		//test get next file
		printf("Test next file\n");
		uint16_t after1 = manager_nextfileid("./abc.mp3");
		printf("%s\n", manager_filenameget(after1, argv[1]));
		//test invalid file
		printf("Test invalid\n");
		uint16_t inv1 = manager_nextfileid("./blub");
		printf("%s\n", manager_filenameget(inv1, argv[1]));
		//test randomfile
		printf("Test random\n");
		uint16_t rand1 = manager_randomfileid();
		printf("%s\n", manager_filenameget(rand1, argv[1]));
		//test near random file
		printf("Test near random\n");
		uint16_t near1 = manager_randomfilenearid("./cd/abc/123/xyz.mp3");
		printf("%s\n", manager_filenameget(near1, argv[1]));
		//test folder iterate
		printf("Test folder iterate\n");
		uint8_t i;
		uint16_t myid = manager_fileidget("./klm/nop.mp3");
		for (i = 0; i < 20; i++) {
			char * nam = manager_filenameget(myid, argv[1]);
			printf("%s\n", nam);
			*(nam+strlen(argv[1])-1) = '.'; //nextfolderid may only get the path relative to argv[1]
			myid = manager_nextfolderfileid(nam+strlen(argv[1])-1);
			free(nam);
		}
		if (argc > 2) { //print sorting
			for (i = 0; i < 225; i++) {
				char * nam = manager_filenameget(i, argv[1]);
				printf("id %i: %s\n", i, nam);
				free(nam);
			}
		}
		free(managermemory);
		return 0;
	}
	return 1;
}

#else

//according to http://www.ethernut.de/api/structdirent.html
#ifndef DT_DIR
#define DT_DIR 1
#endif

#include "memmapper.h"

#endif

uint8_t manger_seekedfile(char * filename) {
	return endswith(filename, ".mp3");
}

/*
Observe: because the AVR has 16 Bit adresses and we got to the upper limit,
but never to the lower one, we have to substract on the left, instead of add
on the right side to prevent overflows.
*/
static uint8_t manager_memcheck(uint16_t size) {
	if ((man_entryptr-size) >= (uint16_t *)man_minpathptr) {
		puts_P(PSTR("Out of mem"));
		return 0;
	}
	return 1;
}

void manager_recursive(uint8_t depth) {
	struct dirent* entry;
	//first pass to manage files
	DIR * d  = opendir(man_temppath);
	if (d == NULL) {
		error_diropen(man_temppath);
		return;
	}
	uint16_t i = 0, k = 0;
	uint16_t * entryptr = man_entryptr;
	while ((entry = readdir(d)) != NULL) {
		i++;
		if (manger_seekedfile(entry->d_name)) {
#ifdef DEBUG
				printf("Found '%s'\n", entry->d_name);
#endif
			if (!manager_memcheck(sizeof(uint16_t))) {
				break;
			}
			k++;
			MAN_LOCK
			*man_entryptr = i;
			MAN_UNLOCK
			man_entryptr++;
		}
	}
	closedir(d);
	//update entrydir
	if (k) {
		if (!manager_memcheck(depth*sizeof(uint16_t))) {
			return;
		}
		MAN_LOCK
		memcpy(man_entryptr, man_tempway, depth*sizeof(uint16_t));
		entrydir_t * pathptr = man_minpathptr;
		pathptr->depth = depth;
		pathptr->entrys = k;
		pathptr->path = man_entryptr;
		pathptr->filepos = entryptr;
		MAN_UNLOCK
		man_entryptr += depth;
		man_dirnum++;
		man_minpathptr--;
	}
	//second pass to manage subdirs
	d = opendir(man_temppath);
	if (d == NULL) {
		error_diropen(man_temppath);
		return;
	}
	if (depth < (MANAGER_MAXDEPTH-1)) {
		i = 0;
		while ((entry = readdir(d)) != NULL) {
			i++;
			if ((entry->d_type == DT_DIR) && (strcmp(entry->d_name, ".")) &&
			    (strcmp(entry->d_name, "..")) && (!man_abortrequest)) {
				*(man_tempway+depth) = i;
				uint16_t length = strlen(man_temppath);
				path_append(man_temppath, entry->d_name, MANAGER_MAXPATHLEN, 1);
				manager_recursive(depth+1);
				*(man_temppath+length) = '\0'; //clear old append
				if (!manager_memcheck(0)) {
					return;
				}
			}
		}
		closedir(d);
	}
}

void manager_buildup(char *rootdir) {
	puts_P(PSTR("Build DB..."));
	man_buildup_busy = 1;
	man_abortrequest = 0;
	man_tempway = s_cmalloc(MANAGER_MAXDEPTH*sizeof(uint16_t));
	man_temppath = s_cmalloc((MANAGER_MAXPATHLEN+1)*sizeof(char));
	strcpy(man_temppath, rootdir);
	man_dirnum = 0;
	uint8_t i;
	for (i = 0; i < MANAGER_MAXDEPTH; i++) {
		*(man_tempway+i) = MANAGER_NOENT;
	}
	man_minpathptr = (entrydir_t *)(managermemory+MANAGER_MAXMEM-1-sizeof(entrydir_t));
	man_entryptr = managermemory;
	manager_recursive(0);
	free(man_tempway);
	free(man_temppath);
	man_buildup_busy = 0;
	if (!man_abortrequest) {
		puts_P(PSTR("DB ready"));
	} else {
		puts_P(PSTR("DB build aborted"));
	}
}

uint8_t manager_buildup_busy_get(void) {
	return man_buildup_busy;
}

//may only be called within a MAN_LOCK, MAN_UNLOCK block
uint8_t manager_idcompare(entrydir_t * dir, uint16_t * path, uint8_t depth) {
	uint8_t i;
	if (dir->depth != (depth))
		return 0;
	for (i = 0; i < dir->depth; i++) {
		if ((dir->path)[i] != path[i]) {
			return 0;
		}
	}
	return 1;
}

uint16_t manager_fileidget(char * filename) {
	char * ftemp = s_strdup(filename);
	uint16_t way[MANAGER_MAXDEPTH+1];
	uint8_t depth = 0;
	char * st = NULL;
	char * ed = strchr(ftemp, '/');
	if (ed) {
		for (depth = 0; depth < MANAGER_MAXDEPTH; depth++) {
			st = ed;
			ed = strchr(ed+1, '/');
			if (ed != NULL) {
				*(ed) = '\0';
			}
			//get position
			struct dirent* entry;
			char ct = *(st+1);
			*(st+1) = '\0';
#ifdef DEBUG
			printf_P(PSTR("Seeking in '%s'"), ftemp);
#endif
			DIR * d  = opendir(ftemp);
			*(st+1) = ct;
#ifdef DEBUG
			printf_P(PSTR(" for '%s'\n"), st+1);
#endif
			if (d) {
				uint16_t pos = 0;
				while ((entry = readdir(d)) != NULL) {
					pos++;
					if (strcmp(entry->d_name, st+1) == 0) {
						break;
					}
				}
				way[depth] = pos;
				closedir(d);
			} else
				error_diropen(ftemp);
			*(st) = '/';
			if (ed != NULL) {
				*(ed) = '/';
			} else
				break;
		}
	}
	free(ftemp);
#ifdef DEBUG
	{
		puts_P(PSTR("File trace:"));
		uint8_t i;
		for (i = 0; i <= depth; i++) {
			printf("%i, ", way[i]);
		}
		printf("\n");
	}
#endif
	//now get the id from our database
	uint16_t i;
	uint16_t id = 0;
	entrydir_t * k = (entrydir_t *)(managermemory+MANAGER_MAXMEM-1-sizeof(entrydir_t));
	MAN_LOCK
	for (i = 0; i < man_dirnum; i++) {
		//if this is our id?
		if (manager_idcompare(k, way, depth)) {
			uint16_t j;
			for (j = 0; j < k->entrys; j++) {
				if ((k->filepos)[j] == way[depth]) {
					MAN_UNLOCK
					return id += j;
				}
			}
		} else
			id += k->entrys;
		k--;
	}
	MAN_UNLOCK
	id = MANAGER_NOENT;
#ifdef DEBUG
	puts_P(PSTR("File not found"));
#endif
	return id;
}

uint16_t manager_filecount(void) {
	uint16_t i;
	uint16_t nums = 0;
	entrydir_t * k = (entrydir_t *)(managermemory+MANAGER_MAXMEM-1-sizeof(entrydir_t));
	MAN_LOCK
	for (i = 0; i < man_dirnum; i++) {
		nums += k->entrys;
		k--;
	}
	MAN_UNLOCK
	return nums;
}

char * manager_get_path(char * rootdir, uint8_t depth, uint16_t * way, uint8_t isdir) {
	char * filename = s_cmalloc((MANAGER_MAXPATHLEN+1)*sizeof(char));
	filename[0] = '\0';
	path_append(filename, rootdir, MANAGER_MAXPATHLEN, 1);
	uint8_t j;
	for (j = 0; j < depth; j++) {
		DIR * d  = opendir(filename);
		if (d) {
#ifdef DEBUG
			printf_P(PSTR("'%s' opened\n"), filename);
#endif
			uint16_t pos = 0;
			struct dirent* entry;
			while ((entry = readdir(d)) != NULL) {
				pos++;
				if (pos == way[j]) {
					path_append(filename, entry->d_name, MANAGER_MAXPATHLEN, isdir);
					//strncat(filename, entry->d_name, MANAGER_MAXPATHLEN-strlen(filename));
					break;
				}
			}
			closedir(d);
		} else {
			error_diropen(filename);
			break;
		}
	}
	return filename;
}


char * manager_filenameget(uint16_t id, char * rootdir) {
#ifdef DEBUG
	printf_P(PSTR("Seek path for id %i\n"), id);
#endif
	uint16_t i;
	uint16_t way[MANAGER_MAXDEPTH+1];
	uint8_t depth = 0;
	entrydir_t * k = (entrydir_t *)(managermemory+MANAGER_MAXMEM-1-sizeof(entrydir_t));
	//get the way
	MAN_LOCK
	for (i = 0; i < man_dirnum; i++) {
		if ((k->entrys) > id) { //found it
			depth = k->depth;
			memcpy(way, k->path, depth*sizeof(uint16_t));
			way[depth] = (k->filepos)[id];
			break;
		}
		id -= k->entrys;
		k--;
	}
	MAN_UNLOCK
#ifdef DEBUG
	{
		printf_P(PSTR("File trace:"));
		uint8_t i;
		for (i = 0; i <= depth; i++) {
			printf("%i, ", way[i]);
		}
		printf("\n");
	}
#endif
	return manager_get_path(rootdir, depth+1, way, 0);
}

static uint8_t manager_getpos(char * pathname, entrydir_t * entry, char * namea) {
	DIR * d2 = opendir(pathname);
	if (d2 == NULL) {
		error_diropen(pathname);
		return 0;
	}
	uint8_t i;
	uint16_t d2pos = 0;
	uint8_t position = 0;
	struct dirent* dent;
	MAN_LOCK
	uint8_t entrys = entry->entrys;
	MAN_UNLOCK
	for (i = 0; i < entrys; i++) {
		MAN_LOCK
		uint16_t id_to_test = entry->filepos[i];
		MAN_UNLOCK
		//get corresponding filename
		while ((dent = readdir(d2)) != NULL) {
			d2pos++;
			if (d2pos == id_to_test) {
				if (strcasecmp(namea, dent->d_name) > 0) {
					position++;
				}
				break;
			}
		}
	}
	//printf("%s: %i\n", namea, position);
	closedir(d2);
	return position;
}

void manager_sortfiles(char * pathname, entrydir_t * entry) {
	MAN_LOCK
	uint16_t entrys = entry->entrys;
	MAN_UNLOCK
	if ((entrys < 2) || (entrys > 100)) {
		return;
	}
	uint16_t * targetmem = s_cmalloc(entrys*sizeof(uint16_t));
	uint8_t i;
	DIR * d1 = opendir(pathname);
	if (d1 == NULL) {
		error_diropen(pathname);
		return;
	}
	uint16_t d1pos = 0;
	struct dirent* dent;
	for (i = 0; i < entrys; i++) {
		MAN_LOCK
		uint16_t id_to_place = entry->filepos[i];
		MAN_UNLOCK
		//get corresponding filename
		while ((dent = readdir(d1)) != NULL) {
			d1pos++;
			if (d1pos == id_to_place) {
				//insert to proper position
				uint16_t pos = manager_getpos(pathname, entry, dent->d_name);
				if (pos < entrys) {
					targetmem[pos] = id_to_place;
				}
				break;
			}
		}
		if (man_abortrequest) {
			break;
		}
	}
	closedir(d1);
	//copy new memory over old memory
	if (!man_abortrequest) {
		MAN_LOCK
		memcpy(entry->filepos, targetmem, entrys*sizeof(uint16_t));
		MAN_UNLOCK
	}
	free(targetmem);
}


/* Will take seconds to minutes to finish*/
void manager_sortnames(char * rootdir) {
	puts_P(PSTR("Sorting..."));
	//go through every filenamelist
	uint16_t i;
	uint16_t way[MANAGER_MAXDEPTH+1];
	uint8_t depth;
	entrydir_t * k = (entrydir_t *)(managermemory+MANAGER_MAXMEM-1-sizeof(entrydir_t));
	for (i = 0; i < man_dirnum; i++) {
		//get the path of the directory
		MAN_LOCK
		depth = k->depth;
		memcpy(way, k->path, depth*sizeof(uint16_t));
		MAN_UNLOCK
		char * pathname = manager_get_path(rootdir, depth, way, 1);
		printf_P(PSTR("Sorting: '%s'\n"), pathname);
		//sort
		manager_sortfiles(pathname, k);
		//free memory and go to next enty
		free(pathname);
		k--;
		if (man_abortrequest) {
			puts_P(PSTR("Sort aborted"));
			return;
		}
	}
	//TODO: sort directories
	puts_P(PSTR("Sorted"));
}

uint16_t manager_nextfileid(char * filename) {
	uint16_t id = manager_fileidget(filename);
	id++;
	if (id == manager_filecount()) {
		id = 0;
	}
	return id;
}

uint16_t manager_prevfileid(char * filename) {
	uint16_t id = manager_fileidget(filename);
	if (id == 0) {
		id = manager_filecount();
	}
	id--;
	return id;
}

#define MAN_RANDAVOID 10
uint16_t man_randlast[MAN_RANDAVOID];

uint8_t manager_waslastrand(uint16_t id) {
	uint8_t i;
	for (i = 0; i < MAN_RANDAVOID; i++) {
		if (man_randlast[i] == id) {
			return 1;
		}
	}
	for (i = 0; i < MAN_RANDAVOID-1; i++) {
		man_randlast[i+1] = man_randlast[i];
	}
	man_randlast[0] = id;
	return 0;
}

uint16_t manager_randomfileid(void) {
	uint16_t top = manager_filecount();
	uint16_t nextid;
	uint8_t i;
	for (i = 0; i < 10; i++) { //10 retrys to avoid already heard files
		nextid = random() % top;
		if (!manager_waslastrand(nextid)) {
			break;
		}
	}
	return nextid;
}

uint16_t manager_randomfilenearid(char * filename) {
	uint16_t top = manager_filecount();
	uint16_t id = manager_fileidget(filename);
	uint16_t nextid;
	uint8_t i;
	for (i = 0; i < 10; i++) { //10 retrys to avoid already heard files
		nextid = id + (random() % 25) - 10; //slowly move forward
		nextid %= top;
		if (!manager_waslastrand(nextid)) {
			break;
		}
	}
	return nextid;
}

uint16_t manager_nextfolderfileid(char * filename) {
	uint16_t id = manager_fileidget(filename);
	uint16_t i;
	uint16_t offset = 0;
	uint16_t nextid = MANAGER_NOENT;
	entrydir_t * k = (entrydir_t *)(managermemory+MANAGER_MAXMEM-1-sizeof(entrydir_t));
	//get the dir
	MAN_LOCK
	for (i = 0; i < man_dirnum; i++) {
		if ((k->entrys) > (id-offset)) { //found it
			nextid = id+1;
			if (nextid >= offset+(k->entrys)) {
				nextid -= k->entrys;
			}
			break;
		}
		offset += k->entrys;
		k--;
	}
	MAN_UNLOCK
	return nextid;
}

void manager_abort(void) {
	man_abortrequest = 1;
}
