/* Connect to an url(c) 2009 by Malte Marwedel
   http://www.marwedels.de/malte

   Terms of use: GPL Version 2 or later
   Example for reading id3 information from mp3 files

compile for pc: gcc -o connurl connurl.c -Wall -Wextra -DTEST_PC
run on pc: ./connurl <your url file>
*/


#include <stdlib.h>
#include <inttypes.h>
#include <stdio.h>
#include <string.h>
#include <arpa/inet.h>
#include "connurl.h"

#ifdef TEST_PC
#include <unistd.h>

#else
//add your avr specific includes here
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <avr/pgmspace.h>

#include "memmapper.h"

#endif

//===== the PC test program ===

#ifdef TEST_PC

//for testing on pc only:
#include "pctests/avrmapper.h"
#include "pctests/net_nuttounix.h"

int main(int argc, char ** argv) {
	if (argc < 2) {
		printf("Give an url file as param\n");
		return 1;
	}
	if (connurl_retrieve(argv[1])) {
		printf("Test: Connection is a success\n");
		connurl_terminate();
	} else {
		printf("Test: Connection failed\n");
	}
	return 0;
}

#endif

//==== the url parser =================

#define MAXURLLEN 384

static prog_char http[] = "http://";

char * emptystring = "\0";

FILE * conn_stream;
TCPSOCKET * conn_socket;


uint8_t connurl_terminate(void) {
	if ((!conn_stream) && (!conn_socket)) {
		return 0; //nothing has been changed
	}
	if (conn_stream) {
		fclose(conn_stream);
		conn_stream = NULL;
	}
	if (conn_socket) {
#if defined TEST_PC || defined __NUT_EMULATION__
		if (1) {
#else
		if (conn_socket->so_state != TCPS_CLOSED) {
#endif
			if (NutTcpCloseSocket(conn_socket)) {
				puts_P(PSTR("closing socket failed"));
			}
		}
		conn_socket = NULL;
	}
	return 1;
}

/*as we pass pointers to these variables, I am not sure if the content or the
  reference to it is saved within the functions*/

//returns 1 if succeed, 0 if an error occured
uint8_t connurl_connect(char * url) {
	connurl_terminate();
	uint16_t port = 80;
	//extract the url into his parts
	//  get domain
	char * domain = strstr_P(url, http);
	if (domain == NULL) {
		printf_P(PSTR("no http in '%s'\n"), url);
		return 0;
	}
	domain += strlen_P(http);
	//  end at new line
	char * nline = strchr(domain, '\n');
	if (nline)
		*nline = '\0';
	nline = strchr(domain, '\r');
	if (nline)
		*nline = '\0';
  // get name (if there)
	char * name = strchr(domain, '/');
	if (name) {
		*name = '\0'; //makes sure the : for the port is not found within the name
		name++; //one after the slash
	} else {
		name = emptystring;
	}
	// get port (if there)
	char * ports = strchr(domain, ':');
	if (ports) {
		port = atol(ports+1); //atoi would only allow 2^15 on an AVR
		*ports = '\0';
	}
	printf_P(PSTR("domain: '%s'\n"), domain);
	printf_P(PSTR("port: '%i'\n"), port);
	printf_P(PSTR("name: '%s'\n"), name);
	//resolve ip
	uint32_t ip = inet_addr(domain);
	//I looked into the source of inet_addr, it returns -1 if an error occured
	if (ip == (uint32_t)-1) {
		ip = NutDnsGetHostByName((unsigned char*)domain);
		if (!ip) {
			puts_P(PSTR("Could not resolve domain"));
			return 0;
		}
	}
	//connect
	conn_socket = NutTcpCreateSocket();
	if (conn_socket == NULL) {
		puts_P(PSTR("socket err"));
		return 0;
	}
	//I looked into the Nut/OS code: the content is stored, not a reference
	uint16_t tcpbufsiz = 8712; //10k large buf sizes result in hanging...
	uint32_t tcptimeout = 1500;
	uint16_t mss = 1452; //1452 maximum for ethernet with adsl
	NutTcpSetSockOpt(conn_socket, SO_RCVTIMEO, &tcptimeout, sizeof(tcptimeout));
	NutTcpSetSockOpt(conn_socket, SO_RCVBUF, &tcpbufsiz, sizeof(tcpbufsiz));
	NutTcpSetSockOpt(conn_socket, TCP_MAXSEG, &mss, sizeof(mss));
	if (NutTcpConnect(conn_socket, ip, port)) {
		puts_P(PSTR("conn failed"));
		return 0;
	}
	//open stream
	if ((conn_stream = _fdopen((int) conn_socket, "r+b")) == 0) {
		puts_P(PSTR("stream open failed"));
		return 0;
	}
	//request http
	fprintf_P(conn_stream, PSTR("GET /%s HTTP/1.0\r\nIcy-Metadata:1\r\nHost: %s\r\n\r\n"), name, domain);
	//printf_P(PSTR("GET /%s HTTP/1.0\r\nIcy-Metadata:1\r\nHost: %s\r\n\r\n"), name, domain);
	fflush(conn_stream);
	return 1;
}

uint8_t connurl_retrieve(char * filename) {
	//get the url
	FILE * f = fopen(filename, "r");
	if (f == NULL) {
		puts_P(PSTR("could not open file"));
		return 0; //no need to free() here
	}
	char * url = s_cmalloc((MAXURLLEN+1)*sizeof(char)); //is already null terminated
	fread(url, sizeof(char), MAXURLLEN, f);
	fclose(f);
	uint8_t ret = connurl_connect(url);
	free(url);
	return ret;
}



