/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <dev/board.h>
#include <compiler.h>
#include <sys/timer.h>
#include <string.h>
#include <stdint.h>
//for F_CPU:
#include "infplay.h"
#include <util/delay.h>
#include "hardware/vs1002drv.h"
#include "bigbuff.h"
#include "error.h"

/* There are 512 KB ext memory, of which 64KB are in use otherwise.
The memory is segmented into 16 32KB blocks.
So we may use 14 blocks = 448KB.

wp pointers show to the next free location
rp pointers show to the next readable location.
read bytes if rp != wp

avariable bytes if next wp != rp


*/

//14 blocks

volatile uint8_t buff_block_rp;
volatile BUFF_SEGDATATYPE buff_byte_rp;

uint8_t buff_block_rpf;
BUFF_SEGDATATYPE buff_byte_rpf;

volatile uint8_t buff_block_wp;
volatile BUFF_SEGDATATYPE buff_byte_wp;

uint8_t buff_dual_rp_mode;

void buff_set_dual_rp_mode(uint8_t state) {
	buff_dual_rp_mode = state & 1;
	if (buff_dual_rp_mode) {
		mp3dev_pause3(0xFF);
		buff_block_rpf = buff_block_rp;
		buff_byte_rpf = buff_byte_rp;
		mp3dev_pause3(0);
	}
}

/* tocopy is the distance between the two read pointers */
void buff_copy_tofile(FILE * f, uint32_t tocopy) {
	//check if there is enough data in the buffer
	uint32_t used = BUFF_MAXRAM-buff_free();
	BUFF_SEGDATATYPE byte_rpf = buff_byte_rpf;
	uint8_t block_rpf = buff_block_rpf;
	if (used < tocopy) {
		tocopy = used;
	}
	while (tocopy) {
		//set block pos
		banking_set(block_rpf);
		//copy data
		BUFF_SEGDATATYPE lmaxw;
		if (tocopy > (BUFF_TOOBIGADDR-byte_rpf)) {
			lmaxw = BUFF_TOOBIGADDR-byte_rpf;
		} else
			lmaxw = tocopy;
		if (lmaxw > 6144) {
			/*fwrite can not write 32768 byte at the same time on an AVR,
			  because it uses 16 bit integer internally with a maximum value
			  of 32767.
			  Moreover cutting down in smaller pieces might help the other
			  threads to get some time.
			  And for uninterrupted playback, fwrite may not be busy too long.
			  It should be less than 80ms busy (at 128Kbit/s streams), which
			  means that no more than 6,4KB (80KB/s write speed) should be
			  copied in a block and that there must be a pause of at least
			  7,5ms in order to refill the VS1002 buffer.
			  Using 4096 bytes and 10ms delay to be safe.
			  (However it looks like the bus is sometimes still longer blocked,
			  resulting in some musik interrupts.)
			*/
			lmaxw = 4096;
		}
		size_t lcop = fwrite((uint8_t *)byte_rpf, sizeof(uint8_t), lmaxw, f);
		tocopy -= lcop;
		//update rpf pos
		byte_rpf += lcop;
		if ((!(byte_rpf & BUFF_INBLOCKMASK)) && (byte_rpf != NUTBANK_START)) {
			byte_rpf = NUTBANK_START;
			block_rpf++;
			if (block_rpf > BUFF_ENDBLOCK)
				block_rpf = BUFF_STARTBLOCK;
		}
		if (tocopy) {
			NutSleep(10);
		}
		//check on errors
		if (lcop != lmaxw) {
			error_general(15);
			break;
		}
	}
}

void buff_copy_tofile_follow(FILE * f) {
	//calculate the distance between the two pointers
	mp3dev_pause3(0xFF);
	BUFF_SEGDATATYPE byte_rp = buff_byte_rp;
	uint8_t block_rp = buff_block_rp;
	mp3dev_pause3(0);
	//calc distance
	uint32_t rp = ((uint32_t)(block_rp-BUFF_STARTBLOCK)<<15)+(byte_rp & BUFF_INTERNALADDR);
	uint32_t rpf = ((uint32_t)(buff_block_rpf-BUFF_STARTBLOCK)<<15)+(buff_byte_rpf & BUFF_INTERNALADDR);
	uint32_t tofollow;
	if (rp >= rpf) {
		tofollow = rp-rpf;
	} else
		tofollow = BUFF_MAXRAM-(rpf-rp)+1; //BUFF_MAXRAM includes -1
	//copy and update
	buff_copy_tofile(f, tofollow);
	buff_block_rpf = block_rp;
	buff_byte_rpf = byte_rp;
}

volatile uint8_t bank_c = 0x10; //should be the same as in arch/avr/os/nutinit.c
uint8_t bank_old = 0x10;

void banking_set(uint8_t bank) {
	cli();
	bank_c = bank << 4;
	*((volatile uint8_t *)(ARTHERCPLDSTART)) = bank_c;
	sei();
}

void banking_displayuse(void) {
	cli();
	bank_old = bank_c;
	banking_set(1); //will do sei()
}

void banking_displayrelease(void) {
	banking_set(bank_old >> 4); //needs a bank, not a double bank
}

//serves as stop function too
void buff_init(void) {
	mp3dev_pause3(0xFF);
	buff_block_rp = BUFF_STARTBLOCK;
	buff_byte_rp = NUTBANK_START;
	buff_block_wp = BUFF_STARTBLOCK;
	buff_byte_wp = NUTBANK_START;
	mp3dev_pause3(0);
}

uint32_t buff_free(void) {
	uint32_t fr;
	uint32_t rwp;
	uint32_t rrp;
	rwp = ((uint32_t)(buff_block_wp-BUFF_STARTBLOCK)<<15)+(buff_byte_wp & BUFF_INTERNALADDR);
	if (!buff_dual_rp_mode) {
		mp3dev_pause3(0xFF);
		rrp = ((uint32_t)(buff_block_rp-BUFF_STARTBLOCK)<<15)+(buff_byte_rp & BUFF_INTERNALADDR);
		mp3dev_pause3(0);
	} else {
		rrp = ((uint32_t)(buff_block_rpf-BUFF_STARTBLOCK)<<15)+(buff_byte_rpf & BUFF_INTERNALADDR);
	}
	if (rrp > rwp) {
		fr = rrp-rwp-1; //-1: otherwise pointer could not divide between completely empty and full
	} else
		fr = BUFF_MAXRAM-(rwp-rrp); //BUFF_MAXRAM includes -1
	return fr;
}

size_t buff_filecopy(FILE * f, size_t s) {
	/* The copying has to be divided into up to two
	   separate freads */
	size_t cop = 0;
	while(s) {
		//set block pos
		banking_set(buff_block_wp);
		//copy data
		BUFF_SEGDATATYPE lmaxr;
		if (s > BUFF_TOOBIGADDR-buff_byte_wp) {
			lmaxr = BUFF_TOOBIGADDR-buff_byte_wp;
		} else
			lmaxr = s;
		size_t lcop = fread((uint8_t *)buff_byte_wp, sizeof(uint8_t), lmaxr, f);
		cop += lcop;
		s -= lcop;
		//update wp pos
		mp3dev_pause3(0xFF);
		buff_byte_wp += lcop;
		if ((!(buff_byte_wp & BUFF_INBLOCKMASK)) && (buff_byte_wp != NUTBANK_START)) {
			buff_byte_wp = NUTBANK_START;
			buff_block_wp++;
			if (buff_block_wp > BUFF_ENDBLOCK)
				buff_block_wp = BUFF_STARTBLOCK;
		}
		mp3dev_pause3(0);
		//check on errors
		if (lcop != lmaxr)
			break;
	}
	return cop;
}

#if defined(BUFF_TESTMODE) || defined (__NUT_EMULATION__)

uint8_t buff_get(void) {
	uint8_t retval = 0;
	if ((buff_block_rp != buff_block_wp) || (buff_byte_rp != buff_byte_wp)) {
		banking_set(buff_block_rp);
		retval =  *((uint8_t *)buff_byte_rp);
		//increment rp
		buff_byte_rp++;
		if (!(buff_byte_rp & BUFF_INBLOCKMASK)) {
			buff_byte_rp = NUTBANK_START;
			buff_block_rp++;
			if (buff_block_rp > BUFF_ENDBLOCK)
				buff_block_rp = BUFF_STARTBLOCK;
		}
	}
	return retval;
}

#endif

#ifdef BUFF_TESTMODE

void buff_put(uint8_t * data, uint16_t datalen) {
	/* The copying has to be divided into up to two
	   separate freads */
	size_t cop = 0;
	while(datalen) {
		//set block pos
		banking_set(buff_block_wp);
		//copy data
		BUFF_SEGDATATYPE lmaxr;
		if (datalen > BUFF_TOOBIGADDR-buff_byte_wp) {
			lmaxr = BUFF_TOOBIGADDR-buff_byte_wp;
		} else
			lmaxr = datalen;
		memcpy((uint8_t *)buff_byte_wp, data+cop, lmaxr);
		cop += lmaxr;
		datalen -= lmaxr;
		//update wp pos
		buff_byte_wp += lmaxr;
		if (!(buff_byte_wp & BUFF_INBLOCKMASK)) {
			buff_byte_wp = NUTBANK_START;
			buff_block_wp++;
			if (buff_block_wp > BUFF_ENDBLOCK)
				buff_block_wp = BUFF_STARTBLOCK;
		}
	}
}

void buff_test(void) {
	buff_init();
	FILE * f = fopen("PHAT0:/mp3/The Modern Divide.mp3", "rb");
	if (f == NULL)
		return;
	printf("buf_free(): %li byte\n\r", buff_free());
	uint8_t i;
	for (i = 0; i < 100; i++) {
		BUFF_SEGDATATYPE r = rand();
		if (r > buff_free())
			r = buff_free();
		printf("Filling %u bytes, free: %li\n\r", r, buff_free());
		BUFF_SEGDATATYPE fi = buff_filecopy(f, r);
		if (fi != r) {
			printf("Filled only %u, free: %li\n\r", fi, buff_free());
			break;
		}
		if (r == 0) //buff full
			break;
	}
	fclose(f);
	f = fopen("PHAT0:/mp3/The Modern Divide.mp3", "rb");
	if (f == NULL)
		return;
	uint32_t fo;
	uint32_t fn = buff_free();
	printf("Reading...\n\r");
	uint8_t m = 0;
	while(buff_avail()) {
		m++;
		uint8_t k;
		if (!m) {
			printf("%li\n\r", buff_free());
		}
		BUFF_SEGDATATYPE ffr = fread(&k, sizeof(uint8_t), 1, f);
		if (ffr != 1) {
			printf("File reading error\n\r");
			break;
		}
		uint8_t v = buff_get();
		if (k != v) {
			printf("Error at %i.%u: %i vs %i\n\r", buff_block_rp, buff_byte_rp, k, v);
			break;
		}
		fo = fn;
		fn = buff_free();
		if (fo == fn) {
			printf("Error: Nothing cleared\n\r");
			break;
		}
	}
	fclose(f);
	printf("Block test done\n\r");
}
#endif
