/* Infplay
  (c) 2009-2010 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <avr/wdt.h>
#include <string.h>
#include <netdb.h>
#include <fs/phatvol.h>
#include <compiler.h>

#ifdef __NUT_EMULATION__
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#endif

#include "action.h"
#include "infplay.h"
#include "statestorage.h"
#include "menu-interpreter.h"
#include "fileselect.h"
#include "hardware/vs1002drv.h"
#include "bigbuff.h"
#include "streamlogic.h"
#include "memmapper.h"
#include "networkedit.h"
#include "stringhelper.h"
#include "error.h"
#include "recorder.h"
#include "id3.h"
#include "network.h"

uint8_t visual_showstats;

void visual_print_number(uint8_t idx, int16_t number) {
	sprintf((char *)(menu_strings[idx]), "%i", number);
}

void visual_remote_upd(void) {
	visual_print_number(MENU_TEXT_ircode,
	  state_rc5_code_get(menu_listindexstate[MENU_LISTINDEX_remote]));
}

void visual_volume_upd(void) {
	visual_print_number(MENU_TEXT_mp3volume, state_mp3_volume_get());
}

void visual_backlight_upd(void) {
	visual_print_number(MENU_TEXT_backlight, state_backlight_get());
}

void visual_network_upd(void) {
	strncpy((char *)menu_strings[MENU_TEXT_IP], my_inet_ntoa(confnet.cdn_ip_addr), INITIAL_TEXT_LENGTH-1);
	uint32_t ip;
	NutIpRouteQuery(0, &ip);
	strncpy((char *)menu_strings[MENU_TEXT_GATEWAY], my_inet_ntoa(ip), INITIAL_TEXT_LENGTH-1);
	NutDnsGetConfig2(NULL, NULL, &ip, NULL);
	strncpy((char *)menu_strings[MENU_TEXT_DNS], my_inet_ntoa(ip), INITIAL_TEXT_LENGTH-1);
	u_char * mac = confnet.cdn_mac;
	sprintf_P((char *)(menu_strings[MENU_TEXT_MAC]), PSTR("%02x:%02x:%02x:%02x:%02x:%02x"),
		mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
}

void visual_init_upd(void) {
	//update init states
	menu_checkboxstate[MENU_CHECKBOX_init_eth1] = state_devinit_get(INIT_NET_DONE);
	menu_checkboxstate[MENU_CHECKBOX_init_eth2] = state_devinit_get(INIT_NETCONFIG_DONE);
	menu_checkboxstate[MENU_CHECKBOX_init_sd1] = state_devinit_get(INIT_BLKDEV_DONE);
	menu_checkboxstate[MENU_CHECKBOX_init_sd2] = state_devinit_get(INIT_MOUNT_DONE);
	menu_checkboxstate[MENU_CHECKBOX_init_mp3] = state_devinit_get(INIT_MP3_DONE);
	menu_checkboxstate[MENU_CHECKBOX_init_clock] = state_devinit_get(INIT_CLOCK_DONE);
}

extern TCPSOCKET *tcpSocketList;
void visual_stats_upd(void) {
	sprintf_P((char *)menu_strings[MENU_TEXT_traffic], PSTR("%luKB"), (long unsigned int)state_stats_traffic_get());
	sprintf_P((char *)menu_strings[MENU_TEXT_freeheap], PSTR("%05uByte"), NutHeapAvailable());
	visual_print_number(MENU_TEXT_reboots, state_reboots_get());
	sprintf_P((char *)menu_strings[MENU_TEXT_totaltraffic], PSTR("%luMB"),
	           (long unsigned int)state_totaltraffic_get());
	uint16_t socks = 0;
#ifndef __NUT_EMULATION__
	TCPSOCKET *sp;
	for (sp = tcpSocketList; sp; sp = sp->so_next) {
		if (sp->so_state == TCPS_ESTABLISHED) {
			socks++;
		}
	}
#endif
	visual_print_number(MENU_TEXT_sockets, socks);
	uint32_t uptime = NutGetSeconds()/60;
	sprintf_P((char *)menu_strings[MENU_TEXT_uptime], PSTR("%lu:%02luh"), (long unsigned int)uptime/60, (long unsigned int)uptime%60);
	sprintf_P((char *)menu_strings[MENU_TEXT_operatinghours], PSTR("%luh"),
	           (long unsigned int)state_operatingminutes_get()/60);
	//get MMC size and free space
	PHATVOL *vol = (PHATVOL *) ((&FSDEV)->dev_dcb);
	if (vol) {
		uint64_t fssize = vol->vol_clustsz*vol->vol_sectsz;
		uint64_t fsfree = vol->vol_numfree;
		fsfree *= fssize;
		fssize *= (vol->vol_last_clust) - (vol->vol_root_clust);
		fssize /= (1000L*1000L);
		fsfree /= (1000L*1000L);
		sprintf_P((char *)menu_strings[MENU_TEXT_mmcard], PSTR("%u/%uMB"), (uint16_t)fsfree, (uint16_t)fssize);
	}
}

void visual_pauseplay_upd(void) {
	unsigned char * ogfx = menu_gfxdata[MENU_GFX_playpause];
	if ((stream_state.laststate == STREAM_STOPPED) || (stream_state.laststate == STREAM_STOP) || (mp3dev_pause2_get() != 0x00)) {
		menu_gfxdata[MENU_GFX_playpause] = gfx_play;
	} else {
		menu_gfxdata[MENU_GFX_playpause] = gfx_pause;
	}
	if (ogfx != menu_gfxdata[MENU_GFX_playpause]) {
		state_event_put(EVENT_REDRAW);
	}
}

void visual_recordbutton_upd(void) {
	if (stream_state.recording == STREAM_RECORDING_STOP) {
		sprintf_P((char *)menu_strings[MENU_TEXT_RecordState], PSTR("Start"));
	} else
		sprintf_P((char *)menu_strings[MENU_TEXT_RecordState], PSTR("Stop"));
}

//type: 0: file, 1: stream
void action_filebrowse(uint8_t type) {
	uint8_t isdir = 0;
	uint8_t stateindex = MENU_LISTINDEX_mp3files;
	uint8_t menutext = MENU_TEXT_mp3list;
	char * ending = ending_mp3;
	char * ending2 = NULL;
	if (type == 1) {
		stateindex = MENU_LISTINDEX_stations;
		menutext = MENU_TEXT_stationlist;
		ending = ending_url;
		ending2 = ending_lfm;
	}
	char * filepath = fileselect_filepath(menu_listindexstate[stateindex], &isdir);
	if (filepath == NULL) {
		error_general(17);
		return;
	}
	if (isdir) {
		size_t lenold = s_strlen(fileselect_getpath());
		size_t lennew = s_strlen(filepath);
		char * oldname = s_strdup(extractfilename(fileselect_getpath()));
		char * text = fileselect_init(filepath, ending, ending2, 1, state_sortfilenames_get());
		menu_strings[menutext] = (unsigned char *)text;
		if (lenold > lennew) { //went up in folder tree
			menu_listindexstate[stateindex] = fileselect_getindex(oldname);
		} else {
			menu_listindexstate[stateindex] = 0;
		}
		free(oldname);
	} else {
		printf_P(PSTR("Should start playing now %s\n"), filepath);
		if (type == 1) {
			state_mp3_urlplay(filepath);
		} else {
			state_mp3_fileplay(filepath);
		}
		state_event_put(EVENT_CLOSE_SELECT);
	}
	free(filepath);
}

void action_fileview(uint8_t type) {
	char * path = NULL;
	char * filename = NULL;
	uint8_t stateindex = MENU_LISTINDEX_mp3files;
	uint8_t menutext = MENU_TEXT_mp3list;
	char * ending = ending_mp3;
	char * ending2 = NULL;
	if (type == 1) {
		stateindex = MENU_LISTINDEX_stations;
		menutext = MENU_TEXT_stationlist;
		ending = ending_url;
		ending2 = ending_lfm;
	}
	if (((stream_state.sourcetype == STREAM_TYPE_FILE) && (type == 0)) ||
			((stream_state.sourcetype == STREAM_TYPE_NETWORK) && (type == 1))) {
		filename = extractfilename(stream_state.source);
		path = extractfilepathdup(stream_state.source);
		//check if path is valid, otherwise the user could end up with an empty dir
		struct stat s;
		if (stat(path, &s) == -1) {
			printf_P(PSTR("Could not browse '%s'\n"), path);
			//failed
			free(path);
			path = NULL;
		}
	}
	if (!path) {
		if (type == 1) {
			path = s_strdup(FSDEV_ROOT"stations");
		} else {
			path = s_strdup(FSDEV_ROOT); //why is there no strdup_P?
		}
	}
	menu_strings[menutext] = (unsigned char *)fileselect_init(path, ending, ending2, 1, state_sortfilenames_get());
	menu_listindexstate[stateindex] = fileselect_getindex(filename);
	free(path);
}

static void action_record(void) {
	uint8_t recordmode = state_recordmode_get();
	if (recordmode == 1) {
		state_record_trychange(STREAM_RECORDING_CONTINUOUS);
	} else {
		state_record_trychange(STREAM_RECORDING_TITLE);
	}
}

static void action_recordtoggle(void) {
	if (stream_state.recording == STREAM_RECORDING_STOP) {
		action_record();
	} else {
		state_record_change(STREAM_RECORDING_STOP);
	}
}

unsigned char menu_action(unsigned short action) {
	unsigned char redraw = 0;
	if ((state_player_stopped) && (action != MENU_ACTION_reboot)) {
		return 0;
	}
	banking_displayrelease();
	if (action == MENU_ACTION_ircodeshow) {
		state_rc5_learn_start(menu_listindexstate[MENU_LISTINDEX_remote]);
	}
	if (action == MENU_ACTION_LISTINDEXCHANGE_remote) {
		visual_remote_upd();
	}
	if (action == MENU_ACTION_remotelearnabort) {
		state_rc5_learn_stop();
	}
	if (action == MENU_ACTION_remotelearn) {
		visual_remote_upd();
	}
	if (action == MENU_ACTION_reboot) {
		inf_shutdown();
		wdt_enable(WDTO_15MS);
#ifdef __NUT_EMULATION__
	pc_exit();
#endif
		for (;;); //let the watchdog do the rest
	}
	if (action == MENU_ACTION_mp3volumeup) {
		state_mp3_volume_inc();
		visual_volume_upd();
		redraw = 1;
	}
	if (action == MENU_ACTION_mp3volumedown) {
		state_mp3_volume_dec();
		visual_volume_upd();
		redraw = 1;
	}
	if (action == MENU_ACTION_backlightmore) {
		state_backlight_inc();
		visual_backlight_upd();
		redraw = 1;
	}
	if (action == MENU_ACTION_backlightless) {
		state_backlight_dec();
		visual_backlight_upd();
		redraw = 1;
	}
	if (action == MENU_ACTION_mp3fileview) {
		action_fileview(0);
	}
	if (action == MENU_ACTION_mp3stationview) {
		action_fileview(1);
	}
	if (action == MENU_ACTION_fileselect) {
		action_filebrowse(0);
		redraw = 1;
	}
	if (action == MENU_ACTION_stationselect) {
		action_filebrowse(1);
		redraw = 1;
	}
	if (action == MENU_ACTION_mp3playpause) {
		if (stream_state.laststate == STREAM_STOPPED) {
			state_mp3_replay();
		} else {
			state_pause_toggle();
		}
	}
	if (action == MENU_ACTION_mp3time) {
		state_mp3_playtimetoggle();
	}
	if (action == MENU_ACTION_mp3stop) {
		state_mp3_stop();
	}
	if (action == MENU_ACTION_eepsave) {
		state_save_settings();
	}
	if (action == MENU_ACTION_dhcpchange) {
		state_dhcpset(menu_checkboxstate[MENU_CHECKBOX_usedhcp]);
	}
	if (action == MENU_ACTION_stats_upd) {
		visual_stats_upd();
		visual_showstats = 1;
	}
	if (action == MENU_ACTION_stats_leave) {
		visual_showstats = 0;
	}
	if (action == MENU_ACTION_shutdown) {
		inf_shutdown();
	}
	if (action == MENU_ACTION_mp3prev) {
		stream_dbfileselect(4);
	}
	if (action == MENU_ACTION_mp3next) {
		if (stream_state.sourcetype == STREAM_TYPE_NETWORK) {
			state_mp3_lastfmskip();
		} else {
			stream_dbfileselect(0);
		}
	}
	if (action == MENU_ACTION_bassboost) {
		state_bassboost_set(menu_checkboxstate[MENU_CHECKBOX_bassboost]);
	}
	if (action == MENU_ACTION_autorestore) {
		state_autorestore_set(menu_checkboxstate[MENU_CHECKBOX_autorestore]);
	}
	if (action == MENU_ACTION_sortfilenames) {
		state_sortfilenames_set(menu_checkboxstate[MENU_CHECKBOX_sortfilenames]);
	}
	if (action == MENU_ACTION_playrestore) {
		if (stream_state.sourcetype == STREAM_TYPE_UNDEFINED) {
			if (state_load_playingstate() == 0) {
				error_message_P(PSTR("Select file or stream"));
			}
		} else if (stream_state.sourcetype == STREAM_TYPE_FILE) {
			state_event_put(EVENT_SHOW_FILEPLAYER);
		} else if (stream_state.sourcetype == STREAM_TYPE_NETWORK) {
			state_event_put(EVENT_SHOW_NETWORKPLAYER);
		} else {
			error_general(7);
		}
	}
	if (action == MENU_ACTION_edit_ip) {
		neted_start_ip();
	}
	if (action == MENU_ACTION_edit_gw) {
		neted_start_gw();
	}
	if (action == MENU_ACTION_edit_dns) {
		neted_start_dns();
	}
	if (action == MENU_ACTION_edit_net_ok) {
		neted_accept();
	}
	if (action == MENU_ACTION_edit_ip1) {
		neted_set1();
	}
	if (action == MENU_ACTION_edit_ip2) {
		neted_set2();
	}
	if (action == MENU_ACTION_edit_ip3) {
		neted_set3();
	}
	if (action == MENU_ACTION_edit_ip4) {
		neted_set4();
	}
	if (action == MENU_ACTION_ipup) {
		neted_inc();
		redraw = 1;
	}
	if (action == MENU_ACTION_ipdown) {
		neted_dec();
		redraw = 1;
	}
	if (action == MENU_ACTION_mp3recordlater) {
		//the function does not contain thread switches, so no strdup needed
		recorder_add(state_id3_get(ID3_ARTIST), state_id3_get(ID3_TITLE));
	}
	if (action == MENU_ACTION_RecordToggle) {
		action_recordtoggle();
	}
	if (action == MENU_ACTION_mp3record) {
		action_record();
	}
#ifdef MENU_ACTION_playdecide
	//only needed if newer menu is used
	if (action == MENU_ACTION_playdecide) {
		if (stream_state.sourcetype == STREAM_TYPE_NETWORK) {
			state_event_put(EVENT_SHOW_NETWORKPLAYER);
		} else
			state_event_put(EVENT_SHOW_FILEPLAYER);
	}
#endif
	banking_displayuse();
	return redraw;
}

void visual_init(void) {
	//make empty strings, give everyone 20 chars, make larger later if needed
	uint8_t i;
	for (i = 0; i < MENU_TEXT_MAX; i++) {
		if ((i != MENU_TEXT_mp3list) && (i != MENU_TEXT_stationlist)) {
			menu_strings[i] = s_cmalloc(sizeof(char)*INITIAL_TEXT_LENGTH);
		}
	}
	//One-time setup
	menu_checkboxstate[MENU_CHECKBOX_usedhcp] = state_usedhcp();
	visual_remote_upd();
	visual_volume_upd();
	visual_backlight_upd();
	state_mp3_secupdate();
	visual_pauseplay_upd();
	visual_recordbutton_upd();
}
