/*
AVR High-voltage Serial Fuse Reprogrammer and Flasher
This program has many authors:
The code part for communication and reading + writing fuses is from
https://arduinodiy.wordpress.com/2015/05/16/high-voltage-programmingunbricking-for-attiny/
which again notes that it is adapted from code and design by Paul Willoughby 03/20/2010
http://www.rickety.us/2010/03/arduino-avr-high-voltage-serial-programmer/
rickety.us states that most of the work on its website is under
  Creative Commons Attribution 3.0 United States License.
  And you are free:
    to Share — to copy, distribute and transmit the work
    to Remix — to adapt the work
The post is from the son of the webpage owner and notes that the original is
inspired by
http://mightyohm.com/blog/2008/09/arduino-based-avr-high-voltage-programmer/
but here I did not find any sourcecode similarities.
So I assume the creative common license for this program.

In this vesion me, Malte Marwedel, added the reading/writing capability of the
flash and eeprom. Its a simple-as-possible addition. Feel free to improve.
My page: www.marwedels.de/malte

Compile: Use gcc and define F_CPU. gcc parameter: -DF_CPU=8000000
This program has been tested on an Atmega8 with the pin connection:
red led: PORTB.2
green led: PORTB.0
serial port: 19200 baud, hardware RX, TX
RST PORTC.5 // Output to level shifter done with two transistors
SCI PORTC.4
SDO PORTC.3
SII PORTC.2
SDI PORTC.1
VCC PORTC.0

See reset converter:
                               12V

                               |
                             |<
                     - ------|   BC557
                     |       |\
                    .-.        |
                    | |        |
                    | | 10k   .-.
                    '-'       | | 82
                     |        | |
                     |        '-'
                     |         |---- Reset of target chip
          10k        |         |
          ___      |/         .-.
PORTC.5 -|___|-----|  BC547   | |
                   |>         | | 1200
                     |        '-'
                     |         |
                     |         |
                    ===       ===
                    GND       GND
(created by AACircuit v1.28.6 beta 04/19/05 www.tech-chat.de)
*/

// Fuse Calc:
//   http://www.engbedded.com/fusecalc/

#include <avr/io.h>
#include <inttypes.h>
#include <util/delay.h>
#include <avr/pgmspace.h>

//use PORTC 0...C5
#define  RST     21    // Output to level shifter for RESET from transistor (PC5)
#define  SCI     20    // Target Clock Input (PC4)
#define  SDO     19    // Target Data Output (PC3)
#define  SII     18    // Target Instruction Input (PC2)
#define  SDI     17    // Target Data Input (PC1)
#define  VCC     16    // Target VCC (PC0)

#define  HFUSE  0x747C
#define  LFUSE  0x646C
#define  EFUSE  0x666E

// Define ATTiny series signatures
#define  ATTINY13   0x9007  // L: 0x6A, H: 0xFF             8 pin
#define  ATTINY24   0x910B  // L: 0x62, H: 0xDF, E: 0xFF   14 pin
#define  ATTINY25   0x9108  // L: 0x62, H: 0xDF, E: 0xFF    8 pin
#define  ATTINY44   0x9207  // L: 0x62, H: 0xDF, E: 0xFFF  14 pin
#define  ATTINY45   0x9206  // L: 0x62, H: 0xDF, E: 0xFF    8 pin
#define  ATTINY84   0x930C  // L: 0x62, H: 0xDF, E: 0xFFF  14 pin
#define  ATTINY85   0x930B  // L: 0x62, H: 0xDF, E: 0xFF    8 pin

#define OUTPUT 1
#define INPUT 0
#define HIGH 1
#define LOW 0

#define BAUDRATE 19200

#define UARTNUMBER (F_CPU/(BAUDRATE*16l)-1)


typedef unsigned char byte;

//------------------- basic IO functions, replacement functions for arduino lib

void led_red(void) {
	PORTB &= (1<<2);  //Clear Bit
	DDRB |= (1<<2);  //Set Bit
	PORTB |= (1<<2);  //Set Bit
}

void led_green(void) {
	PORTB &= ~(1<<2); //Clear Bit
	DDRB |= (1<<0);  //Set Bit
	PORTB |= (1<<0);  //Set Bit
}

void led_yellow(void) {
	DDRB |= (1<<2) | (1<<0);  //Set Bit
	PORTB |= (1<<2) | (1<<0);  //Set Bit
}

void led_off(void) {
	PORTB &= ~((1<<0)|(1<<2)); //Clear Bit
}

void pinMode(unsigned int pin, unsigned int value) {
	unsigned char pi = pin & 0x7;
	unsigned char po = pin >> 3;
/*
	if (po == 0) {
		if (value) {
			DDRA |= 1<<pi;
		} else {
			DDRA &= ~(1<<pi);
		}
	}
*/
	if (po == 1) {
		if (value) {
			DDRB |= 1<<pi;
		} else {
			DDRB &= ~(1<<pi);
		}
	}
	if (po == 2) {
		if (value) {
			DDRC |= 1<<pi;
		} else {
			DDRC &= ~(1<<pi);
		}
	}
}

void digitalWrite(unsigned char pin, unsigned char value) {
	unsigned char pi = pin & 0x7;
	unsigned char po = pin >> 3;
/*
	if (po == 0) {
		if (value) {
			PORTA |= 1<<pi;
		} else {
			PORTA &= ~(1<<pi);
		}
	}
*/
	if (po == 1) {
		if (value) {
			PORTB |= 1<<pi;
		} else {
			PORTB &= ~(1<<pi);
		}
	}
	if (po == 2) {
		if (value) {
			PORTC |= 1<<pi;
		} else {
			PORTC &= ~(1<<pi);
		}
	}
}

unsigned char digitalRead(unsigned char pin) {
	unsigned char pi = pin & 0x7;
	unsigned char po = pin >> 3;
	unsigned char ret = 0;
/*
	if (po == 0) {
		if (PINA & (1<<pi)) {
			ret = 1;
		}
	}
*/
	if (po == 1) {
		if (PINB & (1<<pi)) {
			ret = 1;
		}
	}
	if (po == 2) {
		if (PINC & (1<<pi)) {
			ret = 1;
		}
	}
	return ret;
}

void delayMicroseconds(unsigned int delay) {
	while (delay) {
		_delay_us(1.0);
		delay--;
	}
}

//--------------------------------- UART i/o functions -------------------------

void uart_put(uint8_t value ) {
	loop_until_bit_is_set(UCSRA,5);
	UDR = value;
}

uint8_t uart_get(void) {
	if (bit_is_set(UCSRA,RXC)) {
		return UDR;
	} else {
		return 0;
	}
}

void print(char* s) {
	uint8_t i;
	for (i = 0; i < 255; i++) {
		uart_put(*s++);
		if (*s == 0) { //Null terminated string
			break;
		}
	}
}

void print_P(PGM_P s) {
	uint8_t i;
	char c;
	for (i = 0; i < 255; i++) {
		c = pgm_read_byte(s);
		s++;
		uart_put(c);
		if (c == 0) { //Null terminated string
			break;
		}
	}
}

void printHex4(unsigned char value) {
	if (value < 10) {
		uart_put('0' + value);
	} else {
		uart_put('A' + value - 10);
	}
}

void printHex8(unsigned int value) {
	printHex4((value >> 4 ) & 0xF);
	printHex4(value  & 0xF);
}


void printHex16(unsigned int value) {
	printHex8(value >> 8);
	printHex8(value & 0xFF);
}



unsigned char to4B(unsigned char c) {
	if ((c >= '0') && (c <= '9')) {
		return c - '0';
	}
	if ((c >= 'a') && (c <= 'f')) {
		return c - 'a' + 10;
	}
	if ((c >= 'A') && (c <= 'F')) {
		return c - 'A' + 10;
	}
	return 0;
}

unsigned char toHex(unsigned char high, unsigned char low) {
	return ((to4B(high) << 4) | to4B(low));
}

int16_t uartGetHex4(void) {
	unsigned char c;
	do {
		c = uart_get();
	} while (c == 0);
	if ((c >= '0') && (c <= '9')) {
		return c - '0';
	}
	if ((c >= 'a') && (c <= 'f')) {
		return c - 'a' + 10;
	}
	if ((c >= 'A') && (c <= 'F')) {
		return c - 'A' + 10;
	}
	if (c =='\b') {
		return -2; //backspace
	}
	if (c =='\r') {
		return -3; //new line
	}
	if (c == 'x') {
		return -4; //abort
	}
	return -1; //invalid input
}

int16_t uartGetHex8(void) {
	uint8_t a = 0, b = 0;
	uint8_t mode = 0;
	int16_t input;
	while(1) {
		input = uartGetHex4();
		if (input == -3) {
			break;
		}
		if (input >= 0) {
			if (mode == 0) {
				printHex4(input);
				a = input;
				mode = 1;
			} else if (mode == 1) {
				printHex4(input);
				b = input;
				mode = 2;
			}
		} else if (input == -2) { //if backspace
			if (mode == 2) {
				print_P(PSTR("\b"));
				b = 0;
				mode = 1;
			} else if (mode == 1) {
				print_P(PSTR("\b"));
				a = 0;
				mode = 0;
			}
		}
		if (input == -4) {
			return -1; //immediate abort
		}
	}
	if (mode == 1) { //only one digit input
		return a;
	}
	return ((a << 4) | b); //mode == 2, 2 digit input
}

//--------------------- hv prog functions --------------------------------------

byte shiftOut (byte val1, byte val2) {
	int inBits = 0;
	//Wait until SDO goes high
	while (!digitalRead(SDO));
	unsigned int dout = (unsigned int) val1 << 2;
	unsigned int iout = (unsigned int) val2 << 2;
	for (int ii = 10; ii >= 0; ii--)  {
		digitalWrite(SDI, !!(dout & (1 << ii)));
		digitalWrite(SII, !!(iout & (1 << ii)));
		inBits <<= 1;         inBits |= digitalRead(SDO);
		digitalWrite(SCI, HIGH);
		digitalWrite(SCI, LOW);
	}
	return inBits >> 2;
}


void writeFuse (unsigned int fuse, byte val) {
	shiftOut(0x40, 0x4C);
	shiftOut( val, 0x2C);
	shiftOut(0x00, (byte) (fuse >> 8));
	shiftOut(0x00, (byte) fuse);
}

void readFuses (uint8_t * fuses) {
	shiftOut(0x04, 0x4C);  // LFuse
	shiftOut(0x00, 0x68);
	fuses[0] = shiftOut(0x00, 0x6C);
	shiftOut(0x04, 0x4C);  // HFuse
	shiftOut(0x00, 0x7A);
	fuses[1] = shiftOut(0x00, 0x7E);
	shiftOut(0x04, 0x4C);  // EFuse
	shiftOut(0x00, 0x6A);
	fuses[2] = shiftOut(0x00, 0x6E);
}

uint16_t readSignature () {
	uint16_t sig = 0;
	byte val;
	for (int ii = 1; ii < 3; ii++) {
		shiftOut(0x08, 0x4C);
		shiftOut(  ii, 0x0C);
		shiftOut(0x00, 0x68);
		val = shiftOut(0x00, 0x6C);
		sig = (sig << 8) + val;
	}
	return sig;
}

void setup() {
	pinMode(VCC, OUTPUT);
	pinMode(RST, OUTPUT);
	pinMode(SDI, OUTPUT);
	pinMode(SII, OUTPUT);
	pinMode(SCI, OUTPUT);
	pinMode(SDO, OUTPUT);     // Configured as input when in programming mode
#ifdef RSTINV
	digitalWrite(RST, HIGH);  // Level shifter is inverting, this shuts off 12V
#else
	digitalWrite(RST, LOW);  // Level shifter, this shuts off 12V
#endif
	UBRRL = UARTNUMBER; //speed
	UCSRB = 0x18;      //rx + tx
	led_yellow();
}

void enterProgramming(void) {
	led_red();
	pinMode(SDO, OUTPUT);     // Set SDO to output
	digitalWrite(SDI, LOW);
	digitalWrite(SII, LOW);
	digitalWrite(SDO, LOW);
#ifdef RSTINV
	digitalWrite(RST, HIGH);  // 12v Off
#else
	digitalWrite(RST, LOW);  // 12v Off
#endif
	digitalWrite(VCC, HIGH);  // Vcc On
	delayMicroseconds(20);
#ifdef RSTINV
	digitalWrite(RST, LOW);   // 12v On
#else
	digitalWrite(RST, HIGH);   // 12v On
#endif
	delayMicroseconds(10);
	pinMode(SDO, INPUT);      // Set SDO to input
	delayMicroseconds(300);
}

void powerDown(void) {
	digitalWrite(SCI, LOW);
	digitalWrite(VCC, LOW);    // Vcc Off
#ifdef RSTINV
	digitalWrite(RST, HIGH);   // 12v Off
#else
	digitalWrite(RST, LOW);   // 12v Off
#endif
	led_green();
}

void chipErase(void) {
	shiftOut(0x80, 0x4C);
	shiftOut(0x00, 0x64);
	shiftOut(0x00, 0x6C);
	while (digitalRead(SDO) == 0); //wait until out gets high
	shiftOut(0x00, 0x4C); //nop
}

uint8_t readLock(void) {
	shiftOut(0x04, 0x4C);
	shiftOut(0x00, 0x78);
	uint8_t lock = shiftOut(0x00, 0x6C);
	return lock & 0x3;
}

uint8_t readCalibration(void) {
	shiftOut(0x08, 0x4C);
	shiftOut(0x00, 0x0C);
	shiftOut(0x00, 0x78);
	uint8_t cal = shiftOut(0x00, 0x7C);
	return cal;
}


//------- high level hv prog functions -----------------------------------------


uint16_t getFlashWords(void) {
	uint16_t words = 512; //default 1K in Tiny13
	uint16_t sig = readSignature();
	if ((sig == ATTINY24) ||  (sig == ATTINY25))
		words = 1024;
	if ((sig == ATTINY44) ||  (sig == ATTINY45))
		words = 2048;
	if ((sig == ATTINY84) ||  (sig == ATTINY85))
		words = 4096;
	return words;
}

uint8_t getPageBytes(void) {
	uint8_t bytes = 32; //default for 1K or 2K devices
	uint16_t sig = readSignature();
	if ((sig == ATTINY44) ||  (sig == ATTINY45) ||
	    (sig == ATTINY84) ||  (sig == ATTINY85))
		bytes = 64;
	return bytes;
}

uint16_t getEepromBytes(void) {
	uint16_t bytes = 64; //default for Tiny13
	uint16_t sig = readSignature();
	if ((sig == ATTINY24) ||  (sig == ATTINY25))
		bytes = 128;
	if ((sig == ATTINY44) ||  (sig == ATTINY45))
		bytes = 256;
	if ((sig == ATTINY84) ||  (sig == ATTINY85))
		bytes = 512;
	return bytes;
}

void dumpFlashSub(void) {
	uint16_t maxWords = getFlashWords();
	shiftOut(0x02, 0x4C); // load read flash command 1
	print_P(PSTR("Dump flash:"));
	for (unsigned int word = 0; word < maxWords; word++) {
		if ((word % 8) == 0) {
			print_P(PSTR("\r\n"));
		}
		shiftOut(word & 0xFF, 0x0C);               //read flash low and high bytes instr 1 of 6
		if ((word % 256) == 0) {
			shiftOut(word >> 8, 0x1C);               //read flash low and high bytes instr 2 of 6
		}
		shiftOut(0x00, 0x68);                      //read flash low and high bytes instr 3 of 6
		unsigned char low = shiftOut(0x00, 0x6C);  //read flash low and high bytes instr 4 of 6
		shiftOut(0x00, 0x78);                      //read flash low and high bytes instr 5 of 6
		unsigned char high = shiftOut(0x00, 0x7C); //read flash low and high bytes instr 6 of 6
		printHex16((low << 8) | high); //looks like inverted, but now it fits with .hex
	}
	print_P(PSTR("\r\n"));
}

void dumpEepromSub(void) {
	uint16_t maxBytes = getEepromBytes();
	shiftOut(0x03, 0x4C); // load read eeprom command
	print_P(PSTR("Dump eeprom:"));
	for (uint16_t i = 0; i < maxBytes; i++) {
		if ((i % 16) == 0) {
			print_P(PSTR("\r\n"));
		}
		shiftOut(i & 0xFF, 0x0C);               //read eeprom instr 1 of 4
		if ((i % 256) == 0) {
			shiftOut(i >> 8, 0x1C);               //read eeprom instr 2 of 4
		}
		shiftOut(0x00, 0x68);                      //read eeprom instr 3 of 4
		uint8_t data = shiftOut(0x00, 0x6C);  //read eeprom instr 4 of 4
		printHex8(data);
	}
	print_P(PSTR("\r\n"));
}

//page size for tiny13: 16words (each word 16 bit) = 32byte
void writeFlash(void) {
	enterProgramming();
	chipErase();
	uint16_t maxWords = getFlashWords();
	uint8_t pagesize = getPageBytes();
	//1. load command write flash
	shiftOut(0x10, 0x4C);
	uint8_t buffer[pagesize*2];
	uint8_t bufferptr = 0;
	uint16_t word = 0;
	led_yellow();
	print_P(PSTR("Chip erased. Enter flash data. x terminates.\r\n"));
	print_P(PSTR("Only fully entered flash pages are written. (Prints new line)\r\n"));
	while(1) { //repeat 2. 3. until the end
		int16_t value = uartGetHex4();
		if (value == -4) {
			break;
		} else if (value >= 0) {
			buffer[bufferptr] = value;
			bufferptr++;
			printHex4(value);
		} else if (value == -2) {
			if (bufferptr > 0) {
				print_P(PSTR("\b"));
				bufferptr--;
				continue;
			}
		}
		if (bufferptr == (pagesize*2)) {
			print_P(PSTR("-\r\n"));
			bufferptr = 0;
			led_red();
			//2. load flash page buffer
			for (uint8_t i = 0; i < pagesize/2; i++) {
				uint8_t lowbyte = (buffer[i*4] << 4) | (buffer[i*4 + 1]);
				uint8_t highbyte = (buffer[i*4 + 2] << 4) | (buffer[i*4 + 3]);
				shiftOut(word & 0xFF, 0x0C); //address low bits
				//3. load flash high address and program page
				if ((word  % 256) == 0) {
					shiftOut(word >> 8, 0x1C);
				}
				shiftOut(lowbyte, 0x2C); //data low bits
				shiftOut(highbyte, 0x3C); //data high bits
				shiftOut(0x00, 0x7D);
				shiftOut(0x00, 0x7C);
				word++;
			}
			shiftOut(0x00, 0x64);
			shiftOut(0x00, 0x6C);
			while (digitalRead(SDO) == 0); //wait until out gets high
			led_yellow();
		}
		if (word >= maxWords) {
			print_P(PSTR("Full\r\n"));
			break;
		}
	}
	//5. end programming with a nop operation
	shiftOut(0x00, 0x4C); //nop
	powerDown();
	print_P(PSTR("Done\r\n"));
}

void writeEeprom(void) {
	enterProgramming();
	uint16_t maxBytes = getEepromBytes();
	uint16_t signature = readSignature();
	uint8_t pagesize = 4; //4 bytes for all devices
	//1. load command write flash
	shiftOut(0x11, 0x4C);
	uint8_t buffer[pagesize*2];
	uint8_t bufferptr = 0;
	uint16_t byte = 0;
	led_yellow();
	print_P(PSTR("Enter eeprom data. x terminates.\r\n"));
	print_P(PSTR("Only fully entered eeprom pages are written. (Prints new line)\r\n"));
	while(1) {
		int16_t value = uartGetHex4();
		if (value == -4) {
			break;
		} else if (value >= 0) {
			buffer[bufferptr] = value;
			bufferptr++;
			printHex4(value);
		} else if (value == -2) {
			if (bufferptr > 0) {
				print_P(PSTR("\b"));
				bufferptr--;
				continue;
			}
		}
		if (bufferptr == (pagesize*2)) {
			print_P(PSTR("-\r\n"));
			bufferptr = 0;
			led_red();
			//2. load eeprom page buffer
			for (uint8_t i = 0; i < pagesize; i++) {
				uint8_t lowbyte = (buffer[i*2] << 4) | (buffer[i*2 + 1]);
				shiftOut(byte & 0xFF, 0x0C); //address low bits
				if (signature != ATTINY13) {
					shiftOut(byte >> 8, 0x1C); //address high bits
				}
				shiftOut(lowbyte, 0x2C); //data low bits
				shiftOut(0x00, 0x6D);
				shiftOut(0x00, 0x6C);
				byte++;
			}
			//program eeprom page
			shiftOut(0x00, 0x64);
			shiftOut(0x00, 0x6C);
			while (digitalRead(SDO) == 0); //wait until out gets high
			led_yellow();
		}
		if (byte >= maxBytes) {
			print_P(PSTR("Full\r\n"));
			break;
		}
	}
	//5. end programming with a nop operation
	shiftOut(0x00, 0x4C); //nop
	powerDown();
	print_P(PSTR("Done\r\n"));
}

void dumpInfo(void) {
	enterProgramming();
	uint16_t sig = readSignature();
	uint8_t fuses[3];
	readFuses(fuses);
	uint8_t lock = readLock();
	uint8_t calib = readCalibration();
	powerDown();
	print_P(PSTR("Signature=0x"));
	printHex16(sig);
	print_P(PSTR("\r\nFuse Low=0x"));
	printHex8(fuses[0]);
	print_P(PSTR("\r\nFuse High=0x"));
	printHex8(fuses[1]);
	print_P(PSTR("\r\nFuse Ext=0x"));
	printHex8(fuses[2]);
	print_P(PSTR("\r\nLock=0x"));
	printHex8(lock);
	print_P(PSTR("\r\nCalib=0x"));
	printHex8(calib);
	print_P(PSTR("\r\n"));
}

void setFuses(void) {
	uint8_t fuses[3];
	enterProgramming();
	uint16_t sig = readSignature();
	led_yellow();
	print_P(PSTR("\r\nFuse Low=0x"));
	fuses[0] = uartGetHex8();
	print_P(PSTR("\r\nFuse High=0x"));
	fuses[1] = uartGetHex8();
	if (sig != ATTINY13) {
		print_P(PSTR("\r\nFuse Ext=0x"));
		fuses[2] = uartGetHex8();
	}
	led_red();
	writeFuse(LFUSE, fuses[0]);
	writeFuse(HFUSE, fuses[1]);
	if (sig != ATTINY13) {
		writeFuse(EFUSE, fuses[2]);
	}
	powerDown();
	print_P(PSTR("\r\nDone\r\n"));
}

void fixFuses(void) {
	enterProgramming();
	unsigned int sig = readSignature();
	if (sig == ATTINY13) {
		writeFuse(LFUSE, 0x6A);
		writeFuse(HFUSE, 0xFF);
	} else if (sig == ATTINY24 || sig == ATTINY44 || sig == ATTINY84 ||
	           sig == ATTINY25 || sig == ATTINY45 || sig == ATTINY85) {
		writeFuse(LFUSE, 0x62);
		writeFuse(HFUSE, 0xDF);
		writeFuse(EFUSE, 0xFF);
	}
	powerDown();
	print_P(PSTR("\r\nDone\r\n"));
}

void dumpFlash(void) {
	enterProgramming();
	dumpFlashSub();
	powerDown();
}

void dumpEeprom(void) {
	enterProgramming();
	dumpEepromSub();
	powerDown();
}

void testTarget(void) {
	led_yellow();
	digitalWrite(VCC, HIGH);  // Vcc On
	print_P(PSTR("Press any key to power down\r\n"));
	while (uart_get() == 0);
	digitalWrite(VCC, LOW);  // Vcc Off
	led_green();
	print_P(PSTR("Done\r\n"));
}

int main(void) {
	setup();
	print_P(PSTR("YASAP - Yet another silly AVR programmer. V:1.0\r\nCreative common license\r\n"));
	print_P(PSTR("(c) 2015 by Malte Marwedel, based on code from Paul Willoughby\r\n"));
	print_P(PSTR("WARNING: Wrong fuses will brick your chip!\r\nSelect:\r\n"));
	print_P(PSTR("i: read signature, lock, calib, fuses\r\n"));
	print_P(PSTR("f: write new fuses\r\n"));
	print_P(PSTR("r: reset fuses to default\r\n"));
	print_P(PSTR("d: dump current flash\r\n"));
	print_P(PSTR("p: program new flash\r\n"));
	print_P(PSTR("e: dump current eeprom\r\n"));
	print_P(PSTR("w: write new eeprom\r\n"));
	print_P(PSTR("t: test target - 5V power and reset low\r\n"));
	while(1) {
		led_green();
		unsigned char c = uart_get();
		if (c == 'i') {
			dumpInfo();
		}
		if (c == 'f') {
			setFuses();
		}
		if (c == 'r') {
			fixFuses();
		}
		if (c == 'd') {
			dumpFlash();
		}
		if (c == 'p') {
			writeFlash();
		}
		if (c == 'w') {
			writeEeprom();
		}
		if (c == 'e') {
			dumpEeprom();
		}
		if (c == 't') {
			testTarget();
		}

	}
}
