/*LRCD Meter
  Version 1.0
  (c) 2017-2018 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  Features:
  Measure capacity, resistance and diode current
  Inductivity can not be measured - it would require additional external parts.

  Pin connection:
  PINB.0 = Output, LCD E
  PINB.1 = Output, LCD Backlight enabled
  PINB.2 = Output, LCD D7
  PINB.3 = Output, LCD D6
  PINB.4 = Output, LCD D5
  PINB.5 = Output, LCD D4
  PINB.6 = Crystal
  PINB.7 = Crystal
  PINC.0 = Input, Analog or Comparator
  PINC.1 = Output, 1kOhm
  PINC.2 = Output, 10kOhm
  PINC.3 = Output, 100kOhm
  PINC.4 = Output, 1MOhm
  PINC.5 = Output, 10MOhm
  PINC.6 = n/a
  PINC.7 = n/a
  PIND.0 = Input/Output, Input used by debug RS232 and bootloader.
                         Output used for charging L.
  PIND.1 = Output, debug RS232, bootloader
  PIND.2 = Input, Button B (green)
  PIND.3 = Input, Button A (red)
  PIND.4 = Output, LCD RS
  PIND.5 = Output, 249Ohm
  PIND.6 = Input, AIN0 50% VCC reference
  PIND.7 = Input, L measurement

  Hardware:
  timer0: Used for LCD update
  timer1: Used for C measurement
  timer2: Used for main control and button press
  AD converter: Used for R and D measurements
  Analog comparator: Used for C measurements
*/

#include "main.h"
#include <string.h>
#include <stdlib.h>
#include <avr/eeprom.h>

#include "gui.h"
#include "rs232.h"
#include "avr_f64.h"
//Global variables, within this file

#ifndef BUILD_DATE
	#define BUILD_DATE "Unknown date"
	#warning "BUILD_DATE not set in CFLAGS"
#endif


#define EEPROM_LCONFIG_POS (void*)0

//you might want to calibrate these...
#define MODE_R 0
#define MODE_C 1
#define MODE_L 2
#define MODE_D 3
#define MODE_MAX 4

#define RESISTORS 6

const char splash[] PROGMEM = "!\rLCRD Meter 1.0.1\nBuild:"BUILD_DATE;
const char about[] PROGMEM = "!\rGPL v2 (c) by\n Malte Marwedel";

const uint32_t g_r1[RESISTORS] = {249, 1000, 10000, 100000, 1000000, 10000000};

volatile uint8_t g_overflows;
volatile uint8_t g_captured;

//see calculation below
#define LUNCALIBRATED 11513770868373ULL
uint64_t g_LCalibration;



void waitms(uint16_t ms) {
	while(ms) {
		_delay_ms(1.0);
		ms--;
	}
}

ISR(TIMER1_OVF_vect) {
	g_overflows++;
}

ISR(ANALOG_COMP_vect) {
	TIMSK1 = 0; //make sure there is no overflow after capture
	g_captured = 1;
}

static void enableResistor(uint8_t res) {
	if (res == 0) {
		DDRC = 0;
		PORTC = 0;
		DDRD |= (1<<PD5);
		PORTD |= (1<<PD5);
	} else {
		DDRD &= ~(1<<PD5);
		PORTD &= ~(1<<PD5);
		DDRC = 1<<(res);
		PORTC = 1<<(res);
	}
}

static void disableResistors(void) {
	DDRC = 0;
	PORTC = 0;
	DDRD &= ~(1<<PD5);
	PORTD &= ~(1<<PD5);
}

static void discharge(uint8_t waittime) {
	DDRC = 0;
	PORTC = 0;
	PORTD &= ~(1<<PD5);
	DDRD |= 1<<PD5;
	waitms(waittime); //discharge
}

//48*65565cycles = can take up to 200ms for a measurement
uint32_t acquireTimeToLevel(uint8_t resistor, uint8_t toHigh) {
	uint32_t ticks = 0xFFFFFFFF;
	if (toHigh) {
		ACSR = (1<<ACIS1) | (1<<ACIC); //Falling edge (neg input above pos input), Timer1 capture
	} else {
		ACSR = (1<<ACIS1) | (1<<ACIS0) | (1<<ACIC); //Rising edge (neg input below pos input), Timer1 capture
	}
	ADCSRA = 0; //ADEN must be off
	ACSR |= (1<<ACIE); //enable comparator interrupt
	ADCSRB = (1<<ACME); //analog comparator multiplexer enable
	ADMUX = 0; //use pin 0 as negative input
	TCNT1 = 0;
	ICR1 = 0;
	PORTC = 0;
	g_overflows = 0;
	g_captured = 0;
	TIMSK1 = 1<<TOIE1;
	cli();
	TCCR1A = 0;
	TCCR1B = 1<<CS10; //run without prescaler
	TCCR1C = 0;
	enableResistor(resistor);
	sei();
	//lcd_backlightOff(); //good replacement as debug LED
	while (g_captured == 0) {
		if (g_overflows > 48) { //too big cap, or a resistor?
			break;
		}
	}
	//lcd_backlightOn();
	if (g_overflows <= 48) {
		ticks = ICR1 + (uint32_t)g_overflows*0x10000L;
	}
	ADCSRB = 0;
	ACSR = 0;
	TCCR1B = 0;
	TIMSK1 = 0;
	discharge(0); //leave low for next round
	return ticks;
}

/*
R2 is the unknown resistor we want to have. R1 the known one.
Ux = Vcc/(R1+R2)*R2
Ux = Vcc/1023*ADval
<=> Vcc/(R1+R2)*R2 = Vcc/1023*Adval
<=>   1/(R1+R2)*R2 = 1/1023*Adval
<=>        1023*R2 = Adval*(R1+R2)
<=> R2(1023-Adval) = Adval*R1
<=> R2 = (Adval*R1)/(1023-Adval)
Note: AD resolution has been increased to 2046 as max value.
*/
static int64_t measureResistor(uint32_t * valRaw, uint8_t * valResit) {
	uint8_t i;
	uint8_t vres;
	uint16_t raw;
	//acquire
	ad_init(7); //fits for 8...16MHz
	for (i = 0; i < RESISTORS; i++) {
		enableResistor(i);
		waitms(1);
		//raw = sample_ad(0, 32);
		raw = samle_adQuartil(0);
		disableResistors();
		vres = i;
		if (raw < 900) {
			break;
		}
	}
	//store
	*valRaw = raw;
	*valResit = vres;
	int64_t val = 0x7FFFFFFFFFFFFFFF;
	//calculate
	if (raw < 2046) {
		val = (int64_t)raw*(int64_t)g_r1[vres]/(int64_t)(2046-raw);
	}
	return val;
}

#define NUMSAMPLES 10

int comparatorFuncUint32(const void * pa, const void * pb) {
	uint32_t a = *(uint32_t*)pa;
	uint32_t b = *(uint32_t*)pb;
	if (a > b) return 1;
	if (a < b) return -1;
	return 0;
}

static uint32_t measureCapacityQuartil(uint8_t resistor) {
	//high noise, make some 50% average quartil
	uint32_t raws[NUMSAMPLES];
	uint8_t i;
	uint32_t sum = 0;
	for (i = 0; i < NUMSAMPLES; i++) {
		discharge(1); //discharge (low cap, can discharge faster)
		raws[i] = acquireTimeToLevel(resistor, 1);
		sum += raws[i];
		if (sum > (F_CPU/3)) {
			return 0; //timeout, otherwise the watchdog might hit
		}
	}
	//sort it
	qsort(raws, NUMSAMPLES, sizeof(uint32_t), &comparatorFuncUint32);
	//get average of 50% quartil
	sum = 0;
	for (i = 0; i < NUMSAMPLES/2; i++) {
		sum += raws[i + NUMSAMPLES/4];
	}
	return sum/(NUMSAMPLES/2);
}

/*
    Ux = U0 + Vcc*(1-e^(-t/(R1*C)))
<=>  C = -t/(R1*ln(1-Ux/Vcc + U0/Vcc))
    Ux = 0.5*Vcc, U0 = 0:
 =>  C = -t/(R1*ln(0.5))
 =>  C = -t/(R1*-0.69314718056)
<=>  C = t/(R1*0.69314718056)
 =>  C = t*1.442695/R1
     t = raw/F_CPU
 =>  C = raw*1.442695/(R1*F_CPU)
 to femto F:
     C = 1442695040890000*raw/(R1*F_CPU)
     C = 144269504089*raw/(R1*(F_CPU/10000))

If capacity is high and fully discharging costs too much time:
    Ux = 0.5*Vcc, U0 = 0.25 * Vcc:
 =>  C = -t/(R1*ln(0.5 + 0.25)
 =>  C = -t/(R1*-0.287682072)
<=>  C = t/(R1*0.287682072)
 =>  C = t*3.476059497/R1
     t = raw/F_CPU
 =>  C = raw*3.476059497/(R1*F_CPU)
 to femto F:
     C = 3476059497000000*raw/(R1*F_CPU)
     C = 347605949700*raw/(R1*(F_CPU/10000))

Returned value is in fF
*/
static int64_t measureCapcity(uint32_t * valRaw, uint8_t * valResit) {
	uint8_t i;
	uint8_t vres;
	uint8_t quarterstart = 0;
	uint32_t raw;
	//acquire
	for (i = 0; i < RESISTORS; i++) {
		discharge(20);
		raw = acquireTimeToLevel(i, 1);
		vres = i;
		if (raw > 10000) {
			break;
		}
	}
	if (vres >= 2) {
		uint32_t raw2 = measureCapacityQuartil(vres);
		if (raw2) { //use simple value in the case of the timeout
			raw = raw2;
		}
	}
	//store
	*valResit = vres;
	*valRaw = raw;
	//calculate
	//raw can be up to 2^16*24, so the multiplication fits within 2^63
	int64_t val = 0; //error case
	if ((raw < 63931543) && (!quarterstart)) { // smaller than 2^63/144269504089
		val = 144269504089LL*(int64_t)raw/((int64_t)g_r1[vres]*(F_CPU/10000LL));
	} else if ((raw < 26533987) && (quarterstart)) { // smaller than 2^63/347605949700
		val = 347605949700LL*(int64_t)raw/((int64_t)g_r1[vres]*(F_CPU/10000LL));
	}
	return val;
}

/*
Theory of operation
RX Pin is used as charge output pin (so no PC control possible anylonger)
PinD.7 is used as input. The comparator is low as long as the voltage is
below Vref. So measuring the time, gives us half the period.
http://electronoobs.com/eng_arduino_tut10_3.php
Gives the formula to get from the time to the inductance:
L=1/(4*Pi^2*f^2*C)
C=2.2µF
L=1/(.0000868524*f^2) = (1/.0000868524)*(1/f^2)
L[H]=11513.77086/(f^2)
L[pH]=11513770868373/(f^2)

Since we get the number of CPU cycles, we need to convert them to a frequency
F_CPU/measuredTicks=Frequency


*/

//default: PIND
#define IND_IN_PIN PIND
//default PORTD
#define IND_IN_PORT PORTD
//default DDRD
#define IND_IN_DDR DDRD
//default: PD7
#define IND_IN_BIT PD7

#define INDUCTORSAMPLES 5
static int64_t measureInductivity(uint32_t * valRaw, uint8_t * valResit) {
	uint8_t i;
	uint32_t raws[INDUCTORSAMPLES];
	uint8_t rawSel = 0;
	uint8_t uartstate = UCSR0B;
	uint16_t pinDown1L, pinUpL, pinDown2L;
	uint8_t pinDown1H, pinUpH, pinDown2H;
	UCSR0B &= ~(1<<RXEN0); //disable RX pin, which is PD0 on atmega328
	//measure 5 times
	for (i = 0; i < INDUCTORSAMPLES; i++) { //acquire
		//charge the coil
		DDRD |= 1<<PD0; //output
		IND_IN_DDR &= ~(1<<IND_IN_BIT); //input
		PORTD |= 1<<PD0;
		IND_IN_PORT &= ~(1<<IND_IN_BIT); //no pullup
		waitms(10); //wait for fully charged
		//prepare time measurement
		pinDown1L = 0;
		pinDown1H = 0;
		pinDown2L = 0;
		pinDown2H = 0;
		pinUpL = 0;
		pinUpH = 0;
		cli();
		g_overflows = 0;
		TIMSK1 = 1<<TOIE1;
		TCCR1A = 0;
		TCCR1B = 1<<CS10; //run without prescaler
		TCCR1C = 0;
		TCNT1 = 0;
		PORTD &= ~(1<<PD0); //let go
		sei();
		/*Measure timing
			As this measurement is done in software (and not hardware pin capture)
			only and interrupts are not blocked
			there will be some loss of precision and jitter if an interrupt has hit
			us while the pin toggled
		*/
		while (g_overflows < 6) { //244 would be one second @ 16MHz, so timeout=25ms
			if ((IND_IN_PIN & (1<<IND_IN_BIT)) == 0) {
				cli();
				pinDown1L = TCNT1;
				pinDown1H = g_overflows;
				sei();
				break;
			}
		}
		while (g_overflows < 12) { //244 would be one second @ 16MHz, so timeout=25ms
			if (IND_IN_PIN & (1<<IND_IN_BIT)) {
				cli();
				pinUpL = TCNT1;
				pinUpH = g_overflows;
				sei();
				break;
			}
		}
		while (g_overflows < 18) { //244 would be one second @ 16MHz, so timeout=25ms
			if ((IND_IN_PIN & (1<<IND_IN_BIT)) == 0) {
				cli();
				pinDown2L = TCNT1;
				pinDown2H = g_overflows;
				sei();
				break;
			}
		}
		//stop timer
		TCCR1B = 0;
		TIMSK1 = 0;
		//calculate
		uint32_t pinDown1 = (uint32_t)pinDown1L + (((uint32_t)pinDown1H)<<16);
		uint32_t pinUp = (uint32_t)pinUpL + (((uint32_t)pinUpH)<<16);
		uint32_t pinDown2 = (uint32_t)pinDown2L + (((uint32_t)pinDown2H)<<16);
		if (pinUp > pinDown1) {
			if (pinDown2 > pinUp) {
				//we dected a full wave (preferred as this is more precise)
				raws[i] = (pinDown2 - pinDown1); //full wave measured, possible different rise and fall times of OP amp are canceled out
				rawSel |= 1;
			} else {
				raws[i] = (pinUp - pinDown1)*2; //*2 because only half the wave was measured
				rawSel |= 2;
			}
		} else {
			raws[i] = 0; //error case
			rawSel = 0;
		}
/*
		printf_P(PSTR("%u%-%u -> %u%-%u -> %lu %lu -> %lu\r\n"),
		         pinDownH, pinDownL, pinUpH, pinUpL,
		         (unsigned long)pinDown, (unsigned long)pinUp, (unsigned long)raws[i]);
*/
		printf_P(PSTR("%u-%u -> %u-%u -> %u-%u\r\n"),
		         pinDown1H, pinDown1L, pinUpH, pinUpL, pinDown2H, pinDown2L);
	}
	DDRD &= ~(1<<PD0); //input
	waitms(10); //let pins normalize before restart RX
	UCSR0B = uartstate;
	//take median value
	qsort(raws, INDUCTORSAMPLES, sizeof(uint32_t), &comparatorFuncUint32);
	//store
	*valResit = rawSel;
	*valRaw = raws[INDUCTORSAMPLES/2];
	//calculate
	if (*valRaw) {
		uint64_t frequency = F_CPU / *valRaw;
		if (frequency) {
			return (g_LCalibration/(frequency*frequency));
		}
	}
	return 0; //error case
}

/*
We simply want to have Vx.
Vx = Vcc/1023*Adval
*/
static int64_t measureDiode(uint32_t * valRaw, uint8_t valResitSelect) {
	enableResistor(valResitSelect);
	//acquire
	ad_init(7); //fits for 8...16MHz
	waitms(1);
	uint16_t raw = samle_adQuartil(0);
	disableResistors();
	//store
	*valRaw = raw;
	//calculate
	int32_t val = 5000L*(int32_t)raw/2046L; //Vcc = 5000mV might want calibration later
	return val;
}

static uint8_t userInput(void) {
	char key = keys_pressed();
		if (key) {
			printf_P(PSTR("Key %i pressed\r\n"), key);
		}
		if (key == 0) { //debug only
			key = rs232_key();
			if (key) {
				printf_P(PSTR("Serial %i pressed\r\n"), key);
			}
		}
		return key;
	}

static void messageWaitUser(PGM_P text) {
	lcdPutsP(text);
	while (userInput() == 0) {
		wdt_reset();
	}
	lcd_clear();
}

static void showAbout(void) {
	messageWaitUser(about);
}

//buffer must be at least 21chars in size to manage all numbers
static void printUint64(char * buffer, size_t blen, uint64_t input) {
	size_t idx = blen - 1;
	buffer[idx] = '\0';
	while ((input) && (blen)) {
		idx--;
		char digit = (input % 10) + '0';
		buffer[idx] = digit;
		input /= 10;
	}
	//move to buffer start
	for (uint8_t i = 0; i < blen; i++) {
		buffer[i] = buffer[i + idx];
		if (buffer[i] == '\0') {
			break;
		}
	}
}


/*we support 0.7 to 7µF
Intended C is 2.2µF.
However a 1µF or 4.7µF capacitor can be used too.
*/
#define C5MIN     700000000LL
#define C5MAX    7000000000LL
static void calibrateL(void) {
	uint8_t valResit = 0;
	uint32_t valRaw = 0;
	int64_t valCalc = measureCapcity(&valRaw, &valResit);
	if ((valCalc >= C5MIN) && (valCalc <= C5MAX)) {
		//1. calculate constant value
		/*
			L=1/(4*Pi^2*f^2*C)
			<=> L=(1/(4*Pi^2*C))*(1/(f^2))
			1000000000000000 converts the fF to F.
			1000000000 gives a constant for a result in pH and not H
			multiplied: 1000000000000000000000000 -> cant be done in a float
		*/
#if 0
		float t1 = 4 * M_PI * M_PI * valCalc;
		float t2 = (float)1000000000000000/t1;
		t2 *= 1000000000;
		uint64_t constant = (uint64_t)(t2 + 0.5);
#else
		/*TODO: We should have functions to convert float64_t directly to uint64_t
		        without loosing precision by casting to float first.
		*/
		float64_t t1 = f_mult(f_mult(f_mult(f_sd(4.0), f_sd(M_PI)), f_sd(M_PI)), f_sd(valCalc));
		float64_t t2 = f_div(f_sd(1000000000000000.0), t1);
		t2 = f_mult(f_sd(1000000000.0), t2);
		t2 = f_add(t2, f_sd(0.5));
		uint64_t constant = (uint64_t)(f_ds(t2));
#endif
		//2. save to eeprom
		uint64_t block[2];
		block[0] = constant;
		block[1] = ~constant;
		eeprom_write_block(block, EEPROM_LCONFIG_POS, sizeof(uint64_t)*2);
		char buffer64[21];
		printUint64(buffer64, 21, constant);
		printf_P(PSTR("Calibrated L=%s\r\n"), buffer64);
		//3. display
		char buffer[32];
		printUint64(buffer64, 21, valCalc);
		sprintf_P(buffer, PSTR("!\n%sfF"), buffer64);
		lcdPuts(buffer);
		messageWaitUser(PSTR("\rCal saved"));
		g_LCalibration = constant;
	} else {
		messageWaitUser(PSTR("!\rCal out of range\nConnect L and C"));
	}
}

static void calibrateLoad(void) {
	uint64_t block[2];
	eeprom_read_block(block, EEPROM_LCONFIG_POS, sizeof(uint64_t)*2);
	if (block[0] == ~block[1]) {
		g_LCalibration = block[0];
	} else {
		g_LCalibration = LUNCALIBRATED;
		messageWaitUser(PSTR("!\rLoad calib for\nL failed."));
	}
}

int main(void) {
	//status variables
	uint8_t backlightOn = 1; // 0 or 1
	uint8_t mode = MODE_R; //0 = R, 1 = C, 2 = L, 3 = Diode
	uint8_t showRaw = 0; // 0 or 1
	int64_t valCalc = 0;
	uint32_t valRaw = 0;
	uint8_t valResit = 0;
	uint8_t valResitSelect = 0; //for measure diode
	int64_t calOffset = 0;
	//clear whats left from some bootloader...
	DDRB = 0;
	DDRC = 0;
	DDRD = 0;
	PORTB = 0;
	PORTC = 0;
	wdt_reset();
	wdt_enable(WDTO_2S); //cap measurement can take loooong...
	lcd_backlightOn();
	waitms(100);
	wdt_reset();
	lcd_init(); //as sideeffect: provides starup delay
	keys_init();
	uart_init();
	sei(); //Enable interrupts
	//show some messages
	waitms(500);
	lcdPutsP(splash);
	waitms(1000);
	wdt_reset();
	waitms(1000);
	calibrateLoad();
	//enable timer2 for periodic update
	lcd_clear();
	for (;;) { //Main loop
		cli();
		g_tick = 0;
		sei();
		wdt_reset();
		//evaluate user input
		uint8_t key = userInput();
		if (key == 1) { //red button short -> change mode
			lcd_clear();
			mode++;
			if (mode == MODE_MAX)
				mode = 0;
			calOffset = 0;
		} else if (key == 2) { //red button long -> show raw values
			showRaw = 1 - showRaw;
		} else if (key == 3) { //green button short -> zero values
			if (mode == MODE_D) {
				valResitSelect++;
				if (valResitSelect >= RESISTORS) {
					valResitSelect = 0;
				}
			} else {
				calOffset = valCalc;
			}
		} else if (key == 4) { //green button long -> toggle backlight
			backlightOn = 1 - backlightOn;
			if (backlightOn) {
				lcd_backlightOn();
			} else {
				lcd_backlightOff();
			}
		} else if (key == 5) {
			if (mode == MODE_L) {
				calibrateL();
			}
		} else if (key == 6) {
			showAbout();
		}
		if (key) {
			printf_P(PSTR("Key: %i, new mode:%i\n\r"), key, mode);
		}
		//measure
		uint8_t delta = 0;
		if (calOffset) {
			delta = 1;
		}
		if (mode == MODE_R) { //R
			valCalc = measureResistor(&valRaw, &valResit);
			showResistor(valCalc - calOffset, valRaw, valResit, showRaw, delta);
		} else if (mode == MODE_C) { //C
			valCalc = measureCapcity(&valRaw, &valResit);
			showCapacity(valCalc - calOffset, valRaw, valResit, showRaw, delta);
		} else if (mode == MODE_L) { //L
			valCalc = measureInductivity(&valRaw, &valResit);
			showInductivity(valCalc - calOffset, valRaw, valResit, showRaw, delta);
		} else if (mode == MODE_D) { //Diode
			valCalc = measureDiode(&valRaw, valResitSelect);
			showDiode(valCalc - calOffset, valRaw, valResitSelect, showRaw, delta);
		}
		while(g_tick < (F_CPU/1024/256/2)); //iterate the main loop every 0.5 sec
	}
}
