/* lcd.c

Copyright (C) 2006-2008, 2017 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "main.h"

#define LCD_SIZE 32
uint8_t volatile lcd_virtual[LCD_SIZE];
uint8_t volatile lcd_modify_flags[(LCD_SIZE+7)/8];
uint8_t lcd_pointer = 0;


// ------------- functions running within the interrupt -------------

static void lcd_set_4bit(uint8_t ch) {
	PORTB |= (1<<PB0); //set E
	asm volatile ("nop"); //62.5µs @ 16MHz
	asm volatile ("nop"); //62.5µs @ 16MHz
	asm volatile ("nop"); //62.5µs @ 16MHz
	asm volatile ("nop"); //62.5µs @ 16MHz
	asm volatile ("nop"); //62.5µs @ 16MHz
	PORTB &= ~(1<<PB5);	//clear data pins
	PORTB &= ~(1<<PB4);
	PORTB &= ~(1<<PB3);
	PORTB &= ~(1<<PB2);
	//set the pins according to ch
	if (ch & 0x01) //lowest bit
		PORTB |= (1<<PB5);
	if (ch & 0x02) //2. bit
		PORTB |= (1<<PB4);
	if (ch & 0x04) //3. bit
		PORTB |= (1<<PB3);
	if (ch & 0x08) //4. bit
		PORTB |= (1<<PB2);
	//set-> clear E
	asm volatile ("nop"); //62.5µs @ 16MHz
	asm volatile ("nop"); //62.5µs @ 16MHz
	asm volatile ("nop"); //62.5µs @ 16MHz
	asm volatile ("nop"); //62.5µs @ 16MHz
	asm volatile ("nop"); //62.5µs @ 16MHz
	PORTB &= ~(1<<PB0); //clear E
}

static void lcd_send_nodelay(uint8_t ch) {
	//the LCD reads the data in when switching E from high to low
	//put high 4 bits on port
	lcd_set_4bit(ch >> 4);
	//put low 4 bits on port
	lcd_set_4bit(ch);
}


static void lcd_send_char_nodelay(uint8_t ch) {
	PORTD |= (1<<PD4);	//set RS
	lcd_send_nodelay(ch);
}

static void lcd_send_sys_nodelay(uint8_t sys) {
	PORTD &= ~(1<<PD4);	//clear RS
	lcd_send_nodelay(sys);
}

ISR(TIMER0_OVF_vect, ISR_NOBLOCK) {
	static uint8_t disp_wp = 0; //internal display write pointer
	static uint8_t k = 0; //buffer read pointer
	if (k >= LCD_SIZE)
		k = 0;
	while (k < LCD_SIZE) {
		uint8_t flags = lcd_modify_flags[k/8];
		if ((flags & (1<<(k%8))) != 0) {  //if char is modified
			//check address
			uint8_t lcd_target_address = k;
			if (lcd_target_address >= (LCD_SIZE/2)) {
				lcd_target_address += (0x40-LCD_SIZE/2); //second line address
			}
			if (lcd_target_address != disp_wp) {
				//set LCD to proper adress
				lcd_send_sys_nodelay(0x80 | lcd_target_address);
				disp_wp = lcd_target_address;
				TCNT0 = 0xff-(F_CPU/1024UL/100UL); // the next ISR starts in 1/100s = 10ms
				return;
			}
			flags &= ~(1<<(k%8));           //clear flag
			lcd_modify_flags[k/8] = flags;  //write back
			uint8_t ch = lcd_virtual[k];  //read out before returning from ISR
			//write char
			lcd_send_char_nodelay(ch);
			//increase address
			disp_wp++;
			k++;
			TCNT0 = 0xff-(F_CPU/1024UL/100UL); // the next ISR starts in 1/100s = 10ms
			return;
		}
		k++;
	}
	TCNT0 = 0;
}

// -------------- functions running outside of the interrupt

void lcd_send_char(uint8_t ch) {
	lcd_send_char_nodelay(ch);
	_delay_ms(10);
}

static void lcd_send_sys(uint8_t sys) {
	lcd_send_sys_nodelay(sys);
	_delay_ms(10);
}

#define LCD_SYMBOLS_C 16
const uint8_t lcd_symbols[LCD_SYMBOLS_C] PROGMEM =
      {0x00, 0x0e, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x00,  //battery symbol
       0x00, 0x06, 0x06, 0x0b, 0x0b, 0x11, 0x1f, 0x00}; //delta symbol

static void lcd_placesymbols(void) {
	//starts with character 1 placing the bytes in the CG RAM
	uint8_t syscmd = 0x48;
	uint8_t k;
	for (k = 0; k < LCD_SYMBOLS_C; k++) {
		lcd_send_sys(syscmd+k);
		uint8_t ch = pgm_read_byte(&lcd_symbols[k]);
		lcd_send_char(ch);
	}
	lcd_send_sys(0x80);	//back to DD RAM
}

void lcd_init(void) {
	_delay_ms(10); //function supports up to 16ms @ 16MHz.
	_delay_ms(10);
	DDRB = 0x3F;
	DDRD |= (1<<PD4);
	lcd_send_sys(0x33);	//general init
	lcd_send_sys(0x32);	//general init -> 4bit interface
	lcd_send_sys(0x28);	//two lines and 5x7 font
	lcd_send_sys(0x0c);	//display on
	lcd_send_sys(0x01);	//display clear
	lcd_send_sys(0x06);	//entry mode set, auto-increment address pointer
	lcd_send_sys(0x02);	//return home
	_delay_ms(10);
	//place battery symbol as custom character one
	lcd_placesymbols();
	_delay_ms(10);
	//set up the timer0
	TCCR0A = 0;
	TCCR0B = (1<<CS02) | (1<<CS00); //prescaler = 1024
	TCNT0 = 0;
	TIMSK0 = (1<<TOIE0); //interrupt enabled
}

static void lcd_put_char_virtual(uint8_t ch) {
	if (lcd_pointer < LCD_SIZE) {
		if (lcd_virtual[lcd_pointer] != ch) {
			lcd_virtual[lcd_pointer] = ch; //write char
			cli();
			uint8_t flags = lcd_modify_flags[lcd_pointer/8]; //load flags
			flags |= 1<<(lcd_pointer%8); //set flag
			lcd_modify_flags[lcd_pointer/8] = flags; //write back
			sei();
		}
		lcd_pointer++;
	}
}

void lcd_putchar(uint8_t ch) {
	if (ch == '\n') { //new line -> go to 2. line
		lcd_pointer = LCD_SIZE/2;
	} else if (ch == '\r') { //return to home -> go to 1. line
		lcd_pointer = 0;
	} else if ((ch >= 0x80) && (ch < 0xa0)) { //set as address
		//lcd_pointer = ch-0x80;
	} else if (ch == '!') { //clear lcd
		uint8_t k;
		lcd_pointer = 0;
		for (k = 0; k < LCD_SIZE; k++) {
			lcd_put_char_virtual(' ');
		}
		lcd_pointer = 0;
	} else {
		lcd_put_char_virtual(ch);
	}
}

void lcd_clear(void) {
	lcd_putchar('!');
}

void lcd_backlightOn(void) {
	PORTB |= (1<<PB1);
}

void lcd_backlightOff(void) {
	PORTB &= ~(1<<PB1);
}

