
#include <stdint.h>
#include <string.h>
#include <inttypes.h>
#include <avr/pgmspace.h>
#include <stdio.h>

#include "lcd.h"
#include "main.h"

/*16chars per line + zero termination + 1 char for start of line indicator
  + 1char for unicode character overhead
*/
#define LCDLINEBUFFERLEN 19

/* There are simple commands for the LCD:
\r: Go to the first line
\n: Go to the second line
!: Clear LCD
0x80...0xA0: Go to character on LCD
*/

#define SCALERS 11
const char * g_suffix[SCALERS] = {"a", "f", "p", "n", "µ", "m", "", "k", "M", "G", "T"};

//this does minimal utf-8 to LCD char conversion
void lcdPuts(char * buffer) {
	printf("%s\r\n", buffer); //for debug
	while(*buffer) {
		uint8_t b0 = *buffer;
		buffer++;
		uint8_t b1 = *buffer;
		if ((b0 == 0xCE) && (b1 == 0xA9)) { // Ω
			b0 = 0xF4;
			buffer++;
		} else if ((b0 == 0xC2) && (b1 == 0xB5)) { // µ
			b0 = 0xE4;
			buffer++;
		} else if ((b0 == 0xCE) && (b1 == 0x94)) { // Δ
			b0 = 0x2;
			buffer++;
		}
		lcd_putchar(b0);
	}
}

void lcdPutsP(PGM_P text) {
	uint8_t i = 0;
	char buffer[36];
	//Count the numbers of parameters in the string
	do {
		buffer[i] = pgm_read_byte(text);
		text++;
		i++;
	} while ((buffer[i-1]) && (i < 35));
	lcdPuts(buffer);
}


static void printRaw(char * buffer, uint32_t raw, uint8_t valResit) {
	snprintf_P(buffer, LCDLINEBUFFERLEN, PSTR("\nM=%i raw=%lu       "), valResit, raw);
}

static void append(char * a, const char * b) {
	int increments = LCDLINEBUFFERLEN;
	while ((*a) && (increments)) {
		a++;
		increments--;
	}
	do {
		if (increments > 1) {
			*a = *b;
			a++;
			b++;
			increments--;
		}
		else {
			*a = '\0';
			break;
		}
	} while (*(b-1));
}


static void print4Suf(int64_t value, int8_t radix, char * output) {
	char bufferA[24];
	char bufferB[24];
	memset(bufferA, 0, sizeof(char)*24);
	memset(bufferB, 0, sizeof(char)*24);
	uint8_t i = 0;
	//0. mask out sign
	uint8_t negative = 0;
	if (value < 0) {
		negative = 1;
		value = -value; //mask out negative sign
	}

	uint8_t digits = 0;
	//1. convert to decimal
	for (i = 0; i < 19; i++) { //maximum number 9223372036854775808 has 19 digits
		bufferA[i] = value % 10 + '0';
		value /= 10;
		if (value == 0) {
			digits = i+1;
			break;
		}
	}
	radix += digits - 1;
	//2. get first 4 most significant digits and flip them
	for (i = 0; i < 4; i++) {
		if (digits) {
			digits--;
			bufferB[i] = bufferA[digits];
		} else {
			bufferB[i] = '0';
		}
	}
	uint8_t suf = (radix + 62) /3 - 15;
	uint8_t shift = 2- (radix + 62) % 3;
#ifdef PC_TEST
	if (suf < SCALERS) {
		printf("%s radix %i suf %i shift %i -> %s   ", bufferB, radix, suf, shift, g_suffix[suf]);
	} else {
		printf("%s radix %i suf %i shift %i   ", bufferB, radix, suf, shift);
	}
#endif
	//3. add prefix
	if (shift == 1) {
		bufferB[4] = bufferB[3];
		bufferB[3] = '.';
	}
	if (shift == 2) {
		bufferB[4] = bufferB[3];
		bufferB[3] = bufferB[2];
		bufferB[2] = '.';
	}
	if (negative) {
		append(output, "-");
	}
	append(output, bufferB);
	if (suf < SCALERS) {
		append(output, g_suffix[suf]);
	}
}

//value is in Ohm
void showResistor(int64_t value, uint32_t raw, uint8_t valResit, uint8_t showRaw, uint8_t delta) {
	char buffer[LCDLINEBUFFERLEN];
	lcdPutsP(PSTR("\rResistor"));
	if (showRaw) {
		printRaw(buffer, raw, valResit);
	} else {
		buffer[0] = 0;
		if (delta) {
			append(buffer, "\nΔR=");
		} else {
			append(buffer, "\nR=");
		}
		print4Suf(value, 0, buffer);
		append(buffer, "Ω           ");
	}
	lcdPuts(buffer);
}

//value is in 1fF
void showCapacity(int64_t value, uint32_t raw, uint8_t valResit, uint8_t showRaw, uint8_t delta) {
	char buffer[LCDLINEBUFFERLEN];
	lcdPutsP(PSTR("\rCapacity"));
	if (showRaw) {
		printRaw(buffer, raw, valResit);
	} else {
		buffer[0] = 0;
		if (delta) {
			append(buffer, "\nΔC=");
		} else {
			append(buffer, "\nC=");
		}
		print4Suf(value, -15, buffer);
		append(buffer, "F           ");
	}
	lcdPuts(buffer);
}

//value is in pH
void showInductivity(int64_t value, uint32_t raw, uint8_t valResit, uint8_t showRaw, uint8_t delta) {
	char buffer[LCDLINEBUFFERLEN];
	lcdPutsP(PSTR("\rInductivity"));
	if (showRaw) {
		printRaw(buffer, raw, valResit);
	} else {
		buffer[0] = 0;
		if (delta) {
			append(buffer, "\nΔL=");
		} else {
			append(buffer, "\nL=");
		}
		print4Suf(value, -12, buffer);
		append(buffer, "H           ");
	}
	lcdPuts(buffer);
}

//value is in mV
void showDiode(int64_t value, uint32_t raw, uint8_t valResit, uint8_t showRaw, uint8_t delta) {
	char buffer[LCDLINEBUFFERLEN];
	uint32_t valR1 = g_r1[valResit];
	snprintf_P(buffer, LCDLINEBUFFERLEN, PSTR("\rDiode@"));
	print4Suf(valR1, 0, buffer);
	append(buffer, "Ω");
	if (showRaw) {
		append(buffer, "         ");
		lcdPuts(buffer);
		printRaw(buffer, raw, valResit);
	} else {
		int v16 = value;
		long int imy = ((5000UL-value)*1000UL)/valR1;
		if (delta) {
			append(buffer, "Δ"); //never used for diodes, so no problem its not fully fitting on the screen
		}
		append(buffer, " UF=    ");
		lcdPuts(buffer);
		snprintf_P(buffer, LCDLINEBUFFERLEN, PSTR("\n%imV I=%liµA       "), v16, imy);
	}
	lcdPuts(buffer);
}

