/*
Simple four channel signal sampler version 2.1

    Copyright (C) 2006  by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


Supported comands:

In main Loop:
^HD\n: Disable UART Transmitter
^HR\n: Performs a Reset
^HI\n: Prints ID and SW Version
^HM1\n: Selects the fast four channel digital sampler
^HM2\n: Selects the oszilloscope mode
^HM3\n: Selects the multimeter mode
^HM4\n: Selects the signal generator mode
^HA001\n: Activates the UART Transmitter
^HB009600\n: Selects 9600 Baud for the UART transmitter
^HB115200\n: Selects 115200 Baud for the UART transmitter
^HB230400\n: Selects 230400 Baud for the UART transmitter
^HV\n: Gets the state of the input voltage

In all four M1..M4 modes:
^HM\n: Goes back to the main loop

*/

#include "main.h"

u08 recindex;
char recbuf[10];

u08 device_sel = 0;

u08 volatile isr_rec_state = 0;
register unsigned char back_to_main asm("r4");

const char bc1[7] PROGMEM = "B009600";
const char bc2[7] PROGMEM = "B115200";
const char bc3[7] PROGMEM = "B230400";


ISR(USART_RXC_vect) {		//UART Receive Interrupt
u08 zeichen;
zeichen = UDR;
if (zeichen == 0x08) {		//new command
  isr_rec_state = 1;
} else
if ((zeichen == 'M') && (isr_rec_state == 1)) { //Back to main menu
  isr_rec_state = 2;
} else
if ((zeichen == 0x0D) && (isr_rec_state == 2)) { //Command finished
  back_to_main = 1;
} else { //other character
  isr_rec_state = 0;
}
}

void delayms(u16 ms) {
while (ms > 0) {
  _delay_ms(1.0); //Goes only up to about 23ms
  ms--;
}
}

int main(void) {
//We need to take care of which reset source triggered the reset.
//On a watchdog resed we still have to use the old baudrate
u08 zeichen;
u08 temp;
u16 data;
if (MCUCSR) {			//If real reset
  UBRRL = baud9600val;
}
MCUCSR = 0;
//Set outputs
DDRD = 0x80;			//LED output
//Set pull-up resistors
PORTC = 0x30;
PORTD = 0x7c;
//UART Init
UCSRC = (1<<URSEL)|(1<<UCSZ1)|(1<<UCSZ0)|(1<<USBS);		//Two stop bits
UCSRB = (1<<RXEN);		//Activates UART receiver
//AD enabled
ADCSRA = (1<<ADEN)|(1<<ADPS2)|(1<<ADPS1)|(1<<ADPS0);
ADMUX = (1<<REFS0) | 14;	//1.23V (Bandgap) with internal reference
ADCSRA |= (1<<ADSC);		//Start next conversion
//Timer1 enabled
TCCR1A = 0;
TCNT1 = 0;
TCCR1B = (1<<CS12) | (1<<CS10);	//Prescale: 1024
for (;;) {			//The endless loop
  //Check voltage
  if ((ADCSRA & (1<<ADSC)) == 0) {	//A/D conversion finished
    /*We can not use checkvoltage() here, because the busy waiting loop is too
      long, resulting in RS232 receive data loss */
    data = ADCL;
    data |= ADCH<<8;
    ADCSRA = (1<<ADEN)|(1<<ADPS2)|(1<<ADPS1)|(1<<ADPS0);
    ADMUX = (1<<REFS0) | 14;	//1.23V (Bandgap) with internal reference
    ADCSRA |= (1<<ADSC);	//Start next conversion
    if (calcvoltage(data) == 0) { //If volage ok
      led_on();
    } else			//Voltage wrong lets the LED blink
    if (TCNT1 > 3418) {
      TCNT1 = 0;
      led_toogle();
    }
  }
  //Receives the values from the UART and calls the corresponding sub function
  if (bit_is_set(UCSRA,RXC)) {	//Receive
    zeichen = UDR;
    if (zeichen == 0x08) {	//Backspace
      recindex = 0;
    } else
    if (zeichen == 0x0D) {	//If command ends (Return)
      if (recindex == 1) {	//If command has a length of one character
        if (recbuf[0] == 'D') { //Deactivate UART sender
          UCSRB &= ~(1<<TXEN);
        }
        if (recbuf[0] == 'R') { //Hard reset
          if (device_sel) {
            print("\rK\n");
          }
          wdt_enable(WDTO_15MS);
        }
        if ((recbuf[0] == 'I') && (device_sel)){ //Show info
          print("\ri");
          print(devicetype);
          print(devicesw);
          print("\n");
        }
        if ((recbuf[0] == 'V') && (device_sel)) { //Voltage check
          temp = checkvoltage();
          if (temp == 0) {	//Voltage is ok
            print("\rK\n");
          }
          if (temp == 1) {	//Voltage is too high
            print("\rU\n");
          }
          if (temp > 1) {	//Voltage is too low
            print("\ru\n");
          }
        }
      }
      if (recindex == 2) {	//If command has a length of two characters
        if ((recbuf[0] == 'M') && (device_sel)) { //Modus select
          if (recbuf[1] == '1') {
            print("\rK\n");
            delayms(100);
            led_off();
            UCSRB |= (1<<RXCIE); //UART receive interrupt activate
            fastdigit_loop();
          }
          if (recbuf[1] == '2') {
            print("\rK\n");
            delayms(100);
            led_off();
            UCSRB |= (1<<RXCIE); //UART receive interrupt activate
            fastad_loop();
          }
          if (recbuf[1] == '3') {
            print("\rK\n");
            delayms(100);
            led_off();
            UCSRB |= (1<<RXCIE); //UART receive interrupt activate
            slowad_loop();
          }
          if (recbuf[1] == '4') {
            print("\rK\n");
            delayms(100);
            led_off();
            UCSRB |= (1<<RXCIE); //UART receive interrupt activate
            dds_loop();
          }
        }
      }
      if (recindex == 4) {	//If command has a length of four characters
        if ((recbuf[0] == 'A') && (recbuf[1] == '0') && (recbuf[2] == '0')
            && (recbuf[3] == '1')) {
          UCSRB |= 1<<TXEN;
          device_sel = 1;
          print("\rK\n");
        }
      }
      if (recindex == 7) {	//If command has a length of seven characters
        if (strncmp_P(recbuf,bc1,7) == 0) { //Set to 9600 baud
          print("\rK\n");
          _delay_ms(2.0);
          UBRRL = baud9600val;
        }
        if (strncmp_P(recbuf,bc2,7) == 0) { //Set to 115200 baud
          print("\rK\n");
          _delay_ms(2.0);
          UBRRL = baud115200val;
        }
        if (strncmp_P(recbuf,bc3,7) == 0) { //Set to 230400 baud
          print("\rK\n");
          _delay_ms(2.0);
          UBRRL = baud230400val;
        }
      }
    } else {
      if (recindex < 10) {
        recbuf[recindex] = zeichen;
        recindex++;
      }
    } //End: Common character
  } //End: Something in the UART
  //Check Voltage and toolge LED

} //End: for
}

void print(char* s) {
u08 nun;
for (nun = 0;nun < 255;nun++) { //Limmit to a length of 255 characters
  uart_put(*s++);
  if (*s == 0) { //Null terminating string
    break;
  }
}
}

//Sends a character with the UART, as soon as he is free
void uart_put(char value) {
if (UCSRB & (1<<TXEN)) { //Otherwise the following could be an endless loop
  loop_until_bit_is_set(UCSRA,UDRE);
UDR = value;
}
}

char uart_wait_get(void) {
loop_until_bit_is_set(UCSRA,RXC);
return UDR;
}

u08 hextoint(char c) {
u08 num;
num = c-48;
if (num > 9) {
  num -= 7;
}
return num;
}

u08 checkvoltage(void) {
u16 data;
//Checks the voltage with the help of the internal reference
ADCSRA = (1<<ADEN)|(1<<ADPS2)|(1<<ADPS1)|(1<<ADPS0);
ADMUX = (1<<REFS0) | 14;	//1.23V (Bandgap) with internal reference
ADCSRA |= (1<<ADSC);		//Start next conversion
while (ADCSRA & (1<<ADSC));	//Wait until conversion is complete
data = ADCL;
data |= ADCH<<8;
return calcvoltage(data);
}

u08 calcvoltage(u16 data) {
if (data < 228) {		//WARNING: TOO HIGH VOLTAGE > 5.5V
  return 1;
}
if (data < 279) {		//Good voltage
  return 0;
}
if (data < 314) {		//4.5..4.0V
  return 2;
}
if (data < 359) {		//4.0..3.5V
  return 3;
}
return 4;			//Below 3.5V
}

void putnumber(u08 val) {
//puts a 3 digit number to uart. for testing puporse only
uart_put(val/100+48);
uart_put((val/10)%10+48);
uart_put(val%10+48);
}
