/*
RS232 I/O routines
    Copyright (C) 2006  by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "main.h"
#include <errno.h>

const char portnames[portno][14] = {
           "/dev/ttyUSB0\0",
           "/dev/ttyS0\0",
           "/dev/ttyUSB1\0",
           "/dev/ttyS1\0",
           "/dev/ttyS2\0",
           "/dev/ttyS3\0"};

char port_to_use[255];		//Stores the name of the port

int fd_rs232;			//File descriptor for the port
char rs232recbuf[255];		//Receive buffer
struct termios options_port;	//Control the serial port behaviour
struct termios options_port_backup;	//Restore on close

void serial_clear(void) {
//Removes everything from the input Buffer and clears rs232recbuf
int nun;
while (read(fd_rs232,rs232recbuf,254) > 0) {};
for (nun = 0; nun < 255; nun++) { //A real clean
  rs232recbuf[nun] = 0;
}
}

int serial_speed(speed_t newbaud) {
//Returns: 0 if successful, returns something below zero if an error occurs
int res;
res = cfsetispeed(&options_port, newbaud);	//Set input speed
res += cfsetospeed(&options_port, newbaud);	//Set output speed
tcsetattr(fd_rs232, TCSANOW, &options_port);    //Apply options to port
return res;
}

int serial_open(const char *port_wish) {
//Returns: The file descriptor
//readwrite, nodelay and no control lines open
if (port_wish == NULL) {
  if (rs232_manymessages)
    printf("serial_open: No ports specified, will try common ports\n");
  strcpy(port_to_use, portnames[0]);
  fd_rs232 = open(port_to_use, O_RDWR|O_NONBLOCK|O_NOCTTY);
  if (fd_rs232 < 0) {
    if (rs232_manymessages)
      printf("serial_open: Unable to open %s; will try other port\n",port_to_use);
    strcpy(port_to_use,portnames[1]);
    fd_rs232 = open(port_to_use, O_RDWR |O_NONBLOCK|O_NOCTTY);
    if ((fd_rs232 < 0) && (rs232_manymessages)) {
      printf("serial_open: Unable to open %s\n",port_to_use);
    }
  }
} else {		//There is a port as parameter
  if (strlen(port_wish) > 254) {
    if (rs232_manymessages)
      printf("serial_open: Specified port parameter is too long, exiting  %s\n",port_to_use);
    exit(0);
  }
  strcpy(port_to_use, port_wish);
  fd_rs232 = open(port_to_use, O_RDWR |O_NONBLOCK|O_NOCTTY);
  if ((fd_rs232 < 0) && (rs232_manymessages)) {
    printf("serial_open: Unable to open %s\n",port_to_use);
  }
}
if (fd_rs232 >= 0) {			//Port opened
  if (rs232_manymessages)
    printf("serial_open: %s successfully opened\n",port_to_use);
  tcgetattr(fd_rs232, &options_port_backup);	//load current settings for backup
  fcntl(fd_rs232, F_SETFL, O_NDELAY | O_NONBLOCK ); //No delay and non block device
  tcgetattr(fd_rs232, &options_port);	//load current settings into struct
  serial_speed(baud);			//Set speed
  options_port.c_iflag = 0;		//Clear all Bits
  options_port.c_oflag = 0;		//Clear all Bits
  options_port.c_cflag &= ~CSIZE;	//Clear -> deselected bits for datawidth
  options_port.c_cflag &= ~PARENB;	//Clear -> No use of parity bits
  options_port.c_cflag |= CSTOPB;	//Set -> Use of second stop bit
  options_port.c_cflag |= CREAD;	//Set -> Enable receiver
  options_port.c_cflag |= CLOCAL;	//Set -> Ignore modem status lines
  options_port.c_cflag |= CS8;		//Set -> Data width to 8 Bit
  options_port.c_lflag &= ~ICANON;	//Clear -> No line based input
  options_port.c_lflag &= ~ECHO;	//Clear -> No echo send
  tcsetattr(fd_rs232, TCSANOW, &options_port); //Apply options to port
}
return (fd_rs232);
}

int serial_put(char* s) {
//Returns: 0 on success, -1 if write had failed
int n, wrote;
if (s == NULL) { //not a valid pointer
  return -1;
}
n = strlen(s);
if (rs232_manymessages == 2)
  printf("serial_put: Write %s\n",s);
wrote = write(fd_rs232,s, n);
if ((wrote != n) && (rs232_manymessages)) {
  printf("serial_put: write() of serial port failed!\n");
  return -1;
}
return 0;
}

int serial_get(void) {
int readed = 0, n, waitloop;
for (waitloop = 0; waitloop < 50; waitloop++) { //Wait some milliseconds
  usleep(300);	//0,3ms
  n = read(fd_rs232,rs232recbuf,254);
  readed = n;
  if (readed > 0) {		//We got the data
    while (readed < 254) { //In the case the transmitter is still transmitting
      usleep(2000); 		//2ms
      n = read(fd_rs232,&(rs232recbuf[readed]),254-readed);
      if (n <= 0) { 		//No, not transmitting anylonger
        break;
      }
     readed += n;
    }
    break;
  }
}
int e = errno;
if (readed < 0) {
  if (rs232_manymessages == 2) {
    printf("serial_get: Error while reading serial port. ");
    if (e == EAGAIN) {
      printf("Error code: EAGAIN\n");
    } else
    if (e == EBADF) {
      printf("Error code: EBADF\n");
    } else
    if (e == EFAULT) {
      printf("Error code: EFAULT\n");
    } else
    if (e == EINTR) {
      printf("Error code: EINTR\n");
    } else
    if (e == EINVAL) {
      printf("Error code: EINVAL\n");
    } else
    if (e == EISDIR) {
      printf("Error code: EISDIR\n");
    } else
      printf("Error code is unknown: %i\n",e);
  }
  return -1;
}
if (readed == 0) {	//"No data" seems to cause n= -1 and not 0 at a serial port
  if (rs232_manymessages == 2)
    printf("serial_get: No Data received by serial port\n");
  return -1;
}
if (readed > 0) {
  rs232recbuf[readed] = 0;		//Terminate the string
  if (rs232_manymessages == 2)
    printf("serial_get: Read %i bytes\n",readed);
}
return readed;
}

int serial_get_fast(void) {
int readed = 0,n, waitloop;
for (waitloop = 0; waitloop < 5;waitloop++) { //Wait some milliseconds
  n = read(fd_rs232,rs232recbuf,254);
  readed = n;
  if (readed > 0) {			//We got the data
    break;
  }
  usleep(100);
}
if (readed < 0) {
  if (rs232_manymessages == 2)
    printf("serial_get_fast: Error while reading serial port\n");
  return -1;
}
if (readed == 0) {	//"No data" seems to cause n= -1 and not 0 at a serial port
  if (rs232_manymessages == 2)
    printf("serial_get_fast: No Data received by serial port\n");
  return -1;
}
if (readed > 0) {
  rs232recbuf[readed] = 0;		//Terminate the string
  if (rs232_manymessages == 2)
    printf("serial_get_fast: Read %i bytes\n",readed);
}
return readed;
}

void serial_close(void) {
tcflush(fd_rs232, TCIOFLUSH);		//Flush all unsend data
tcsetattr(fd_rs232, TCSANOW, &options_port_backup); //Restore options
close(fd_rs232);
if (rs232_manymessages)
  printf("serial_close: %s closed     \n",port_to_use);
}
