/*
Simple four channel signal plot.
    Copyright (C) 2006  by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "main.h"

int port_no = -1;	//Which port is the right (in the portnames[] list)
int device_id = -1;	//Which id the sampler has

int device_voltage(void) {
send_dev_cmd("V");	//Request voltage
serial_clear();
usleep(20000);
serial_get();
if (rs232recbuf[1] == 'K') {	//Allright
  printf("device_voltage: Info: Operating voltage is within 4.5 and 5.5V.\n\r");
  return 0;
}
if (rs232recbuf[1] == 'U') {
  printf("device_voltage: Error: Operating voltage is above 5.5V THIS COULD DAMAGE YOUR AVR!\n\r");
  printscan_updatetext("Error: VCC OF AVR IS TOO HIGH! ( > 5.5V)");
  return -1;
}
if (rs232recbuf[1] == 'u') {
  printf("device_voltage: Warning: Operating voltage is below 4.5V, this might result in malfunction.\n\r");
  printscan_updatetext("Warning: Vcc of AVR is too low. ( < 4.5V)");
  return -2;
}
return -3;
}

int device_reactivate(void) {
//returns: 1 if success; -1 if failed
char command[5] = "A   \0";
inttohex(command+1,device_id,3);
send_dev_cmd(command);	//Reactivate device
serial_clear();
usleep(20000);
serial_get();
if (rs232recbuf[1] == 'K') {
  return 1;
}
return -1;
}

void device_mode_select(int mode) {
char command[5] = "A   \0";
if ((mode > 0) && (mode < 5)) {
  //Let the device go back to main loop
  printscan_updatetext("Setting sampler circuit into new mode...");
  device_mainmode();
  usleep(100000); //Receive_thread must stop an device needs time to reset
  if (device_reactivate() == 1) { //Device (re)activated
    command[0] = 'M';
    command[1] = mode+48;
    command[2] = '\0';
    serial_clear();
    send_dev_cmd(command);		//Select Mode
    usleep(10000);
    serial_get();
    if (rs232recbuf[1] == 'K') {	//Mode selected
      serial_clear();
      printf("device_mode_select: Info: Device now in new mode\n\r");
      printscan_updatetext("New mode active");
      op_mode = mode;
    } else {
      printf("device_mode_select: Error: Mode activation failed\n\r");
      printscan_updatetext("Error, mode activation failed");
      usleep(200000);
    }
  } else {
    printf("device_mode_select: Error: Device activation failed\n\r");
    printscan_updatetext("Error: sampler circuit did not response");
    usleep(200000);
  }
} else
  printf("device_mode_select: Error: Mode not supportet. If you read this it is most likely a bug!\n\r");
}

void device_mainmode(void) {
//The UART receive interrupt routine is too slow for using the send_dev_cmd("M");
char c[2];
op_mode = 0;
c[0] = 0x08;
c[1] = '\0';
serial_put(c);
usleep(100000);
serial_put("M");
usleep(100000);
serial_put("\r");
usleep(100000);
}

int send_dev_cmd(char *s) {
//Sends the string together with a leading 0x08 and a final \r to the device
int length = strlen(s);
char command[length+4];
strcpy(command+1,s);
command[0] = 0x08;
command[length+1] = '\r';
command[length+2] = 'I';
command[length+3] = '\0';
//printf ("%X,%X,%X,%X,%X,%X,%X,%X,%X,%X\n",command[0], command[1],command[2],command[3],command[4],command[5],command[6],command[7],command[8],command[9]);
return serial_put(command);
}

int device_type(int device) {
//Determine if the found device is useable for us
int dev_type, dev_sw;
if ((rs232recbuf[0] == '\r') && (rs232recbuf[1] == 'K') && (rs232recbuf[2] == '\n')) {
  send_dev_cmd("I");
  serial_get();
  if ((rs232recbuf[0] == '\r') && (rs232recbuf[1] == 'i')) {
    dev_type = hexextract(rs232recbuf+2, 3);
    dev_sw = hexextract(rs232recbuf+5, 3);
    if (dev_type == supported_dev_type) {
      if (dev_sw == supported_sw) {
        printf("device_type: Info: Supported device found: ID: %i, Type: %i, Software: %i\n\r", device, dev_type, dev_sw);
        return 1;
      } else {
        printf("device_type: Error: Right device found but software version mismatch: ID: %i, Type: %i, Software: %i", device, dev_type, dev_sw);
        printf("  Please flash you Microcontroller with a proper program or change PC program\n\r");
        return -1;
      }
    } else {
      printf("device_type: Info: Unknown device found: ID: %i, Type: %i, Software: %i", device, dev_type, dev_sw);
      printf("  Maybe you have connected the wrong circuit to the serial port?\n\r");
      return 0;
    }
  } else {
    printf("device_type: Error: Connection to device %i lost\n\r",device);
    return -2;
  }
}
return 0;
}

int device_scan(void) {
//Tries to find the Simple Sampler on a serial port
int nun, nun2;
char command[6] = "A   \0";
int device_type_result;
int serial_get_result;
int slow_scan = 0;
printscan_updatetext("Searching sampler circuit... please wait");
for (nun = 0; nun < portno; nun++) {		//Probe all ports in list
  if (serial_open(&(portnames[nun][0])) > 0) {	//Open port
    while (serial_get() > 0) {};		//Clear receive buffer
    for (nun2 = 1; nun2 < 0x20; nun2++) { //Test to 0x1000 would be too slow
      inttohex(command+1, nun2, 3);
      printf("device_scan: Info: Probing: %s\r",command);
      fflush(stdout);
      serial_clear();
      if (send_dev_cmd(command) < 0) {
        printf("device_scan: Warning: Can not write to this device\n\r");
        break;
      }
      if (slow_scan) {				//Wait some time here
        usleep(100000);
      }
      serial_get_result = serial_get();
      if (serial_get_result > 2) { //If more than two chars received
        device_type_result = device_type(nun2);
        if (device_type_result == -2) {		//If connection lost
          //This may happen if the scan is too fast, retry the previous number
          if (slow_scan == 0) {
            slow_scan = 1;
            nun2 = 0;	//Restart with scan 1. (the loop will increase the zero)
            printf("device_scan: Warning: Restart with slow scan due connection lost\n\r");
          }
        }
        if (device_type_result == 1) { //If device found
          port_no = nun;
          device_id = nun2;
          printscan_updatetext("Sampler circuit found");
          return 0;
        }
      } else {
        if (serial_get_result != 0) {
          printf("\ndevice_scan: Info: Received %i bytes\n\r", serial_get_result);
        }
      }
    }
    serial_close();
    slow_scan = 0;
  }
}
return -1; //No valid device found
}

int device_switch_baud(int baudrate) {
//Changes the speed of the device and the PC port
//The Sampler supports 9600, 115200 and 230400 baud only!
speed_t baudrate_pc;
serial_clear();
if (baudrate == 9600) {
  send_dev_cmd("B009600");
  baudrate_pc = B9600;
} else
if (baudrate == 115200) {
  send_dev_cmd("B115200");
  baudrate_pc = B115200;
} else
if (baudrate == 230400) {
  send_dev_cmd("B230400");
  baudrate_pc = B230400;
} else {
  printf("device_switch_baud: Error: The Baudrate %i is not supported\n\r", baudrate);
  return -1;
}
usleep(50000); //Wait 50ms
serial_get();
if (rs232recbuf[1] == 'K') {
  if (serial_speed(baudrate_pc) >= 0) {
    printf("device_switch_baud: Info: Switching baudrate was a success\n\r");
    return 1;
  } else {
    printf("device_switch_baud: Error: Switched Baudrate of Sampler but not the PC port\n\r");
  }
} else {
  printf("device_switch_baud: Warning: Sampler did not switch baud rate\n\r");
  return 0;
}
return -1;
}
