/****************************************************************************
 *																			*
 *					RFM12 basierte, drahtlose RS232 Verbindung				*
 *																			*
 *																			*
 *								� by Benedikt								*
 *																			*
 *						Email:	benedikt83 �t gmx.net						*
 *																			*
 ****************************************************************************
 *																			*
 *	Die Software darf frei kopiert und veraendert werden, solange sie nicht	*
 *	ohne meine Erlaubnis fuer kommerzielle Zwecke eingesetzt wird.			*
 *																			*
 * This is a modified version by Malte Marwedel, www.marwedels.de/malte    *
 * Version 2.0 based on version 1.2 from Benedikt									*
 * Added features:
 *   -Eprom storage for settings
 *   -Config everything over RS232
 *   -Variable timeout for retransmit
 *   -Debug options: Dont send act. Report crc errors.
 ***************************************************************************/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <stdlib.h>
#include <portbits.h>
#include "global.h"
#include "uart.h"
#include "rf12.h"
#include "leds.h"

/*
DIP Schalter an PortC:
0-2: Baudrate: 1200,2400,9600,19200,38400,57600,115200 (hohe Baudraten nur mit entsprechendem Quarztakt moeglich !)
3-4: HF Kanal
*/

//modified by malte:
#define F_CPU 8000000UL
#include <util/delay.h>

//#define DEBUG						// anstelle der empfangenen Daten fuer jede Aktion den Status ausgeben

#define USE_SOFTCONFIG				// Konfiguration ueber ein einfaches Menue
//#define USE_DIPSWITCH				// DIP Schalter/Jumper an PortC 0-4
#define CHANNEL			1			// Sende/Empfangskanal (0-3) (nur gueltig wenn kein DIP Schalter verwendet wird)
#define RF_BAUDRATE		20000		// Baudrate des RFM12 (nur gueltig wenn kein DIP Schalter verwendet wird)
//modified by malte:
#define UART_BAUDRATE	9600		// Baudrate des UARTs (nur gueltig wenn kein DIP Schalter verwendet wird)
//modified by malte:
#define CONFIG_BAUD		9600		// Baudrate des UARTs (nur gueltig im Config Mode)

#define MAX_BUF			128			// Paket Groesse in Bytes (maximal 250)
#define TX_TIMEOUT		150			// Maximale Wartezeit auf Daten in ms (max 500)

#define ANSWER_TIMEOUT	10			// Maximale Wartezeit auf die Bestaetigung der Daten in ms (max 500)

#define RETRY			50			// Maximale Anzahl an Sendeversuchen

#define CONFIG			(PIND&4)		// Pin fuer Config Jumper (DDR beachten)

volatile unsigned char delaycnt;
unsigned char rxbuf[MAX_BUF+1];		// Puffer fuer empfangene Daten
unsigned char txbuf[MAX_BUF+1];		// Puffer fuer zu sendende Daten
unsigned char flags, tx_cnt, tx_id, tx_status, retrans_cnt;

#ifdef USE_SOFTCONFIG
	unsigned long uart_baud, rf_baud;
	unsigned char channel;
#endif
	unsigned char answertimeout = ANSWER_TIMEOUT;
	unsigned char retry = RETRY;
	unsigned char txtimeout = TX_TIMEOUT;
	unsigned char shouldactpacket = 1; //0 only recommend for debug purpose
	unsigned short waitingseed; //seed for srand. best to have different one on each device
	unsigned char debugprint;


#define WAITFORACK		1

void rx_packet(void);
void tx_packet(unsigned char retrans);
void load_settings(void);
void save_settings(void);
void configmode(void);
unsigned long read_number(unsigned long min, unsigned long max);
void show_number(unsigned long val);

int main(void)
{
#ifdef USE_SOFTCONFIG
	PORTD=254; //modified by malte
	DDRD=0x7A;  //modified by malte
	DDRB=0x03;
#else
	PORTD=15;
	PORTC=255;
	DDRD=246;
#endif
	DDRC=0x30;
	LED_TX = 1-0;
	LED_RX = 1-0;
	LED_RETRANS = 1-0;
	LED_ERR = 1-0;
	LED_GEN1 = 1-0;
	LED_GEN2 = 1-1;
	LED_INTERF = 1-0;
	sei();

#ifdef USE_DIPSWITCH
	unsigned char c;
	unsigned short baud_uart, baud_rf;
	c=PINC&7;
	if (c==0)
	{	baud_rf=2000;
		baud_uart=UART_BAUD_SELECT(1200, F_CPU);
	}
	else if (c==1)
	{	baud_rf=4000;
		baud_uart=UART_BAUD_SELECT(2400, F_CPU);
	}
	else if (c==2)
	{	baud_rf=8000;
		baud_uart=UART_BAUD_SELECT(4800, F_CPU);
	}
	else if (c==3)
	{	baud_rf=15000;
		baud_uart=UART_BAUD_SELECT(9600, F_CPU);
	}
	else if (c==4)
	{	baud_rf=25000;
		baud_uart=UART_BAUD_SELECT(19200, F_CPU);
	}
	else if (c==5)
	{	baud_rf=30000;
		baud_uart=UART_BAUD_SELECT(38400, F_CPU);
	}
	else if (c==6)
	{	baud_rf=40000;
		baud_uart=UART_BAUD_SELECT(57600, F_CPU);
	}
	else if (c==7)
	{	baud_rf=50000;
		baud_uart=UART_BAUD_SELECT(115200, F_CPU);
	}
	c=(PINC&24)/8;
    uart_init(baud_uart);
	rf12_init();									// ein paar Register setzen (z.B. CLK auf 10MHz)
	rf12_config(baud_rf, c, 0, QUIET);				// Baudrate, Kanal (0-3), Leistung (0=max, 7=min), Umgebungsbedingungen (QUIET, NORMAL, NOISY)
#else
#ifdef USE_SOFTCONFIG
	load_settings();
	if (!CONFIG)
		configmode();
	srand(waitingseed);
	uart_init(UART_BAUD_SELECT(uart_baud, F_CPU));
	rf12_init();									// ein paar Register setzen (z.B. CLK auf 10MHz)
	rf12_config(rf_baud, channel, 0, QUIET);	// Baudrate, Kanal (0-3), Leistung (0=max, 7=min), Umgebungsbedingungen (QUIET, NORMAL, NOISY)
#else
	uart_init(UART_BAUD_SELECT(UART_BAUDRATE, F_CPU));
	rf12_init();									// ein paar Register setzen (z.B. CLK auf 10MHz)
	rf12_config(RF_BAUDRATE, CHANNEL, 0, QUIET);	// Baudrate, Kanal (0-3), Leistung (0=max, 7=min), Umgebungsbedingungen (QUIET, NORMAL, NOISY)
#endif
#endif

	TCCR1A=0;
	TCCR1B=(1<<WGM12)|1;
	OCR1A=((F_CPU+2500)/500)-1;
	TIMSK=(1<<OCIE1A);

	rf12_rxmode();

	for (;;)
	{	if (rf12_data())							// Daten im RF12 RX Puffer ?
		{	rx_packet();
		}
		if (!(flags&WAITFORACK))					// Puffer nur dann fuellen, wenn die alten Daten aus dem Puffer wirklich angekommen sind
		{	if (uart_data())						// Daten im UART RX Puffer ?
			{	if (!tx_cnt)
					delaycnt=txtimeout/2;			// Nach dem ersten Byte: timeout starten
				txbuf[tx_cnt++]=uart_getchar();
			}
			if ((tx_cnt>=MAX_BUF)||((tx_cnt)&&(delaycnt==0))) // Puffer voll, oder timeout seit erstem Byte im Puffer vorbei ?  -> senden
			{	tx_status=0;						// zu sendender Status
				tx_packet(0);						// erstmaliger Transfer
#ifdef DEBUG
				uart_putc('t');
#endif
			}
		}
		else if (delaycnt==0)						// Timeout: Daten nochmal senden
		{	LED_RETRANS=1-1;
			if (retrans_cnt)
			{	retrans_cnt--;
				tx_packet(1);						// retransmit
#ifdef DEBUG
				uart_putc('r');
#endif
			}
			else									// Versuche abgelaufen
			{	LED_ERR=1-1;							// -> Fehler LED an
				tx_cnt=0;							// -> Daten verwerfen
				tx_id++;
				flags&=~WAITFORACK;					// Daten als OK markieren
#ifdef DEBUG
				uart_putc('e');
#endif
			}
		}
	}
}

void rx_packet(void)
{	static unsigned char rx_lastid=255;
	unsigned char rx_cnt,i, rx_id, status;
	rx_cnt=rf12_rxdata(rxbuf, &status, &rx_id);		// komplettes Paket empfangen
	if (rx_cnt<=MAX_BUF)							// Daten gueltig (d.h. kein CRC Fehler) ?
	{	if (status&RECEIVED_OK)						// Empfangsbestaetigung ?
		{	flags&=~WAITFORACK;						// -> "Warten auf Bestaetigung"-Flag loeschen
			tx_cnt=0;								// -> Daten als gesendet markieren
			tx_id++;
			LED_RETRANS=1-0;
			LED_ERR=1-0;
#ifdef DEBUG
				uart_putc('a');
#endif
		}
		if (rx_cnt)									// Daten empfangen
		{
			if (shouldactpacket) { //usually set
				tx_status=RECEIVED_OK;					// zu sendender Status
#ifdef DEBUG
				uart_putc('p');
#endif
				tx_packet(0);								// Empfangsbestaetigung senden
				retrans_cnt=retry;						// Retry Counter neu starten
			}
			if (rx_id!=rx_lastid)					// Handelt es sich um neue Daten ?
			{
#ifndef DEBUG
				for (i=0; i<rx_cnt; i++)			// Daten weiterleiten
					uart_putc(rxbuf[i]);
#endif
				rx_lastid=rx_id;					// Aktuelle ID speichern
			}
		}
	}
	else
	{
#ifdef DEBUG
		uart_putc('n');
#endif
		if (debugprint) {
			if (rx_cnt == 255) {
				uart_puts_P("CRC error\r\n");
			}
		}
	}
}

void tx_packet(unsigned char retrans)
{
	rf12_stoprx();									// auf TX umschalten
	if ((!retrans)&&((flags&WAITFORACK)||(tx_cnt==0))) // es wird noch auf eine Antwort vom Paket gewartet oder es sind keine Daten zu senden
	{    rf12_txdata(txbuf, 0, tx_status, 0);		// -> kein komplettes neues Paket, sondern nur Status senden
#ifdef DEBUG
		uart_putc('s');
#endif
	}
	else
	{	rf12_txdata(txbuf, tx_cnt, tx_status, tx_id); // komplettes Paket senden
		flags|=WAITFORACK;							// auf ACK warten
		delaycnt=answertimeout/2;					// Timeout Counter neu starten
		delaycnt+=(rand() % answertimeout); //results in variable delay [answertimeout/2...answertimeout*3/2]
		if (!retrans)								// erstmalige Uebertragung ?
			retrans_cnt=retry;						// -> Retry Counter neu starten
#ifdef DEBUG
		uart_putc('d');
#endif
	}
	rf12_rxmode();									// wieder auf RX umschalten
}

#ifdef USE_SOFTCONFIG
void load_settings(void)
{
	if (eeprom_read_byte((void *)1)!=0xBB)
	{	rf_baud=RF_BAUDRATE;
		uart_baud=UART_BAUDRATE;
		channel=CHANNEL;
		retry=RETRY;
		answertimeout=ANSWER_TIMEOUT;
		txtimeout=TX_TIMEOUT;
		shouldactpacket=1;
		waitingseed=42;
		debugprint=0;
		save_settings();
	}
	channel=eeprom_read_byte((void *)2);
	eeprom_read_block(&rf_baud,(void *)3,4);
	eeprom_read_block(&uart_baud,(void *)7,4);
	retry=eeprom_read_byte((void *)11);
	answertimeout=eeprom_read_byte((void *)12);
	txtimeout=eeprom_read_byte((void *)13);
	shouldactpacket=eeprom_read_byte((void *)14) & 1; //eeprom has 0xFF as default, god here
	waitingseed=eeprom_read_word((void *)15);
	debugprint=eeprom_read_byte((void *)17);
	if (debugprint != 1) {
		debugprint = 0; //eeprom has 0xFF as default, bad here
	}
}

void save_settings(void)
{	eeprom_write_byte((void *)1,0xBB);
	eeprom_write_byte((void *)2,channel);
	eeprom_write_block(&rf_baud,(void *)3,4);
	eeprom_write_block(&uart_baud,(void *)7,4);
	eeprom_write_byte((void *)11,retry);
	eeprom_write_byte((void *)12,answertimeout);
	eeprom_write_byte((void *)13,txtimeout);
	eeprom_write_byte((void *)14,shouldactpacket);
	eeprom_write_word((void *)15,waitingseed);
	eeprom_write_byte((void *)17,debugprint);
}

void configmode(void)
{	unsigned char c;
	LED_INTERF = 1-1;
	uart_init(UART_BAUD_SELECT(CONFIG_BAUD, F_CPU));
	for(;;)
	{	uart_puts_P("\r\n\r\n");
		uart_puts_P("RFM12 RS232 bridge v1.2.1\r\n");
		uart_puts_P("(c) original by Benedikt\r\n");
		uart_puts_P("(c) modified by Malte\r\n");
		uart_puts_P("Config Menue\r\n\r\n");
		uart_puts_P("Current Settings:\r\n");
		uart_puts_P("\r\n[1] RS232 baudrate:          ");
		show_number(uart_baud);
		uart_puts_P("\r\n[2] RFM12 baudrate:          ");
		show_number(rf_baud);
		uart_puts_P("\r\n[3] Channel:                 ");
		show_number(channel);
		uart_puts_P("\r\n[4] max retry:               ");
		show_number(retry);
		uart_puts_P("\r\n[5] answer timeout [ms]:     ");
		show_number(answertimeout);
		uart_puts_P("\r\n[6] tx collect timeout [ms]: ");
		show_number(txtimeout);
		uart_puts_P("\r\n[7] Act packet (default=1):  ");
		show_number(shouldactpacket);
		uart_puts_P("\r\n[8] srand seed:              ");
		show_number(waitingseed);
		uart_puts_P("\r\n[9] debug print (default=0)  ");
		show_number(debugprint);
		uart_puts_P("\r\n\r\n");
		uart_puts_P("[0] exit menu without saving\r\n");
		uart_puts_P("[s] exit menu and save settings\r\n");
		uart_puts_P("\r\n");
		for(;;)
		{	c=uart_getchar();
			if (c=='0')
			{
				LED_INTERF = 1-0;
				return;
			}
			else if (c=='s')
			{	uart_puts_P("saved!\r\nLeaving config mode...\r\n");
				save_settings();
				_delay_ms(10);
				LED_INTERF = 1-0;
				return;
			}
			else if (c=='1')
			{	uart_puts_P("Enter RS232 baudrate (300-150000)\r\n");
				uart_baud=read_number(300,150000);
				break;
			}
			else if (c=='2')
			{	uart_puts_P("Enter RFM12 baudrate (700-65000)\r\n");
				rf_baud=read_number(700,65000);
				break;
			}
			else if (c=='3')
			{	uart_puts_P("Enter RFM12 channel (0-3)\r\n");
				channel=read_number(0,3);
				break;
			}
			else if (c=='4')
			{	uart_puts_P("Enter max retry: (0-100)\r\n");
				retry=read_number(0,100);
				break;
			}
			else if (c=='5')
			{	uart_puts_P("Enter answer timeout [ms]: (0-100)\r\n");
				answertimeout=read_number(0,100);
				break;
			}
			else if (c=='6')
			{	uart_puts_P("Enter tx timeout [ms]: (0-250)\r\n");
				txtimeout=read_number(0,250);
				break;
			}
			else if (c=='7')
			{	uart_puts_P("Act valid packet (0=No 1=Yes):\r\n");
				shouldactpacket=read_number(0,1);
				break;
			}
			else if (c=='8')
			{	uart_puts_P("srand seed: (0-65565):\r\n");
				waitingseed=read_number(0,65565);
				break;
			}
			else if (c=='9')
			{	uart_puts_P("Show debug messages (0=No 1=Yes):\r\n");
				debugprint=read_number(0,1);
				break;
			}
			else
				break;
		}
		LED_INTERF = 1-0;
	}
}

void show_number(unsigned long val)
{	unsigned char i;
	char s[8]={32,32,32,32,32,32,32,0};
	i=6;
	do
	{	s[i--]=val%10+'0';
	}
	while  (val/=10);
	s[7]=0;
	uart_puts(s);
}

unsigned long read_number(unsigned long min, unsigned long max)
{	unsigned long number;
	unsigned char c,i;
	for (;;)
	{	number=0;
		for (i=0;i<6;i++)
			{	c=uart_getchar();
				if ((c==10)||(c==13)||(c==32))
					break;
			if ((c>='0')&&(c<='9'))
			{	uart_putc(c);
				number=number*10+(c-'0');
			}
		}
		if ((number>=min)&&(number<=max))
			break;
		uart_puts_P("Number out of range, try again!\r\n");
	}
	return number;
}
#endif

ISR(TIMER1_COMPA_vect)							// 500Hz Interrupt
{
	if (delaycnt)
		delaycnt--;
}



