/*
Fahrradlader for ATtiny48V or ATtiny44V
(Fahrrad (ger) = bicycle, lader (ger) = charger)
(c) 2012 by Malte Marwedel
Version 1.0
Terms of use: General Public License V 2 or later

Measured with a Shimano DH-3D30 and a 28" wheel:
Speed   Charging current   Voltage after rectifier diodes
7km/h:   25mA              6.5V
10km/h: 150mA              6.6V
13km/h: 300mA              6.7V
17km/h: 400mA              6.7V
20km/h: 450mA              6.7V
30km/h: 480mA              7.2V
*/

#include <avr/io.h>
#include <avr/interrupt.h>

#include "fahrradlader.h"

#include <util/delay.h>
#include <avr/wdt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>

#include "softtx.h"

/*I/O pins

PA0: Analog In: Temp battery A (1.1Vref)
PA1: Analog In: Temp reference (1.1Vref)
PA2: Analog In: Temp Battery B (1.1Vref)
PA3: Analog In: Measure V in with 1:11 divider (1.1Vref)
PA4: Analog In: Measure Charging current with 1Ohm (1.1Vref)
PA5: Analog In: Mesure Output Voltage with 1:2 voltage divider.
PA6: PWM Out: Set current for battery charging
PA7: Analog In: Measure Bat A- with 1:11 divider (1.1Vref)
PB0: Digigal Out: Left charging LED (battery A)
PB1: Digigal Out: Green output Ok LED, Debug Output Bootloader: Programming output
PB2: Digigal Out: Right charging LED (battery B), Bootloader: Programming input
PB3: Reset

Eeprom content in bytes:
4-7: uint32_t total charged mA min
8-11: uint32_t total charged mA min
12-13: uint16_t number of times the first two values did not match at start

Observed
*/

/*
Bugs:
The charger cant reliable detect if only one pair of batteries is charged
and therefore both LEDs are active in both cases.
It can detect if there is only one pair while powered by battery, but there
is no way to reliable detect wether it is running only on batteries.
A solution would be to wire PA7 to Bat B+.
A good solution would be an additional analog in at B-, but there is no pin
left.
*/


//off
#define LED_OFF 0
//1Hz blinking, 50% on
#define LED_SLOWBLINK 1
//5Hz blinking, 50% on
#define LED_FASTBLINK 2
//1Hz blinking, 90% on
#define LED_SHORTOFF 3
//1Hz blinking, 10% on
#define LED_SHORTON 4
//on
#define LED_ON 5

//pin definitions
#define LED_RED_LEFT 0
#define LED_GREEN 1
#define LED_RED_RIGHT 2

#define TEMP_MAX 50
//read maximum of 50° in energizer handbook for operating
//http://data.energizer.com/PDFs/nickelmetalhydride_appman.pdf
#define BAT_TEMP_MAX 40
#define BAT_TEMP_DELTA 5
//maximum charged mAs
#define BAT_MAS_MAX (2000L*60L*60L)

//minimum mV a pair of batteries may have
#define BAT_MIN 2100

//8Bit pwm. timer could support 9 or 10bit as well
#define PWM_MAX 0xFF

//minimum and maximum mV for the CPU to accept
#define MCU_MV_MIN 3150
#define MCU_MV_MAX 3400

//the CPU 1,8V, so the eeprom must write 4 bytes until voltage
//drops to this level
#define MCU_MV_EEP_MIN 2500

//minimum charge start voltage
#define CHARGE_MV_MIN 6000

//maximum input voltage considered acceptable, otherwise diodes may be damaged
#define CHARGE_MV_GOOD_MAX 9000

//current not the target and enough input voltage, ramp up PWM...
#define CHARGE_MV_MIN_RAMPUP 6800
#define CHARGE_MA_MAX_RAMPUP 500

//... but only if the current did not increase more than X mA since the last
//iteration, to avoid overswinging
#define CHARGE_MA_MAXINC_RAMPUP 10

/*Only allow PWM to ramp up to the maximum if there is some current measured
  at 1/3 of the maximum. If we are at 1/3 of the maximum, there must be some
  current or the batteries are not connected properly. If we would set the PWM
  to maximum to reach the target current and then the batteries suddently
  get connecte, we would get a too high current otherwise.
*/
#define CHARGE_MA_MIN_RAMPFURTHERUP 50

//too low voltage or too high current, ramp down PWM by one
#define CHARGE_MV_MAX_RAMPDOWN 6400
#define CHARGE_MA_MIN_RAMPDOWN 550

//very low voltage or too high current, ramp down PWM by two
#define CHARGE_MV_MAX_RAMPDOWNFAST 6000
#define CHARGE_MA_MIN_RAMPDOWNFAST 700

//way too high current, ramp down PWM to zero
#define CHARGE_MA_MIN_FULLSTOP 800

//note that the 4700µF condensator takes some time until its discharged
#define USB_MV_MIN_MODE 1000

/*USB voltage is good within the two values
  the specifications allow a maximum of 5.25V and
  6V for overshoot
  and 9V for a defective USB charger.
  So I assume that 5.3V wont do any harm.
*/

#define USB_MV_MIN_ERROR 5300
#define USB_MV_MIN_TOOLOW 4200


volatile uint8_t led_states[3];
volatile uint8_t led_counter;
volatile uint8_t debugmode = 0;

uint32_t mamininit = 0;
uint32_t maminlast = 0;

ISR(TIM0_COMPA_vect) { //10Hz timer
	uint8_t i;
	uint8_t bits = 0;
	uint8_t counterl = led_counter;
	for (i = 0; i < 3; i++) {
		uint8_t state = led_states[i];
		bits = bits << 1;
		if ((state == LED_ON) ||
		    ((state == LED_SLOWBLINK) && (counterl >= 5)) ||
		    ((state == LED_FASTBLINK) && (counterl & 1)) ||
		    ((state == LED_SHORTOFF) && (counterl > 0)) ||
		    ((state == LED_SHORTON) && (counterl == 0))) {
			bits++;
		}
	}
if (debugmode) {
		bits |= (1<<SOFTTX_PIN);  //debug mode does not send during interrupt
	}
	PORTB = bits;
	counterl++;
	if (counterl == 10) {
		counterl = 0;
	}
	led_counter = counterl;
}

static void print_u16(uint16_t number) {
	softtx_char('0'+(number/10000));
	softtx_char('0'+(number/1000)%10);
	softtx_char('0'+(number/100)%10);
	softtx_char('0'+(number/10)%10);
	softtx_char('0'+(number%10));
}

static void print_u32(uint32_t number) {
	uint32_t meaning = 1000000000; //4G is the biggest number of uint32_t
	uint8_t digit = 0;
	while (meaning) {
		if (number >= meaning) {
			number -= meaning;
			digit++;
		} else {
			softtx_char('0'+digit);
			digit = 0;
			meaning /= 10;
		}
	}
}

uint16_t temp_convert(uint16_t ad) {
	uint32_t ohm = ((uint32_t)ad*4700)/(3069-ad);
	/* 3. grade polynomia with 0, 25, 50 and 80°C:
		2.12x10^-9*ohm^3 -0.0000224786*ohm^2 + 0.128189*ohm - 157.224
		3. grade not representable in 32bit.

		2. grade polynomia: with 0, 50 and 100°C
 -0.0000070424599426781825*x^2 +0.09231465863293291*x-130.5806904123189

		Calculate with 1/10 grad and then round proper
	*/
	uint16_t gradcelsius = ((-(ohm*ohm*10/141996) + (ohm*100/108) - 1306) + 5)/10;
	if (gradcelsius > 1000) { //negative temperature!
		gradcelsius = 0;
	}
	return gradcelsius;
}

/*
Data order in array:
0: left temperature
1: reference temperature
2: right temperature
3: Charge Vcc in mV
4: charging current in mA
5: voltage at Bat1- in mV
6: voltage at USB in mV
7: MCU Vcc in mV
*/
uint16_t ad_data[8];

/*
A/D Converting takes 1MHz/8/13 = 104µs
Math converting for temperature takes ~2198 Clock cycles = 2198µs
And 678µs for all other mV and mA calculation
And 604µs for the last divide.
-> Routine needs 104µs*11+2198µs*3+678µs*4+604µs = ~11ms
*/
void ad_updateall(void) {
	uint8_t i;
	uint8_t channel = 0;
	for (i = 0; i < 11; i++)  {
		channel = i;
		if (i == 0) {
			ADCSRA = 0; //disable because we change the reference voltage
		}
		if (i == 5) {
			channel = 7;
		}
		if (i == 6) {
			ADCSRA = 0; //disable because we change the reference voltage
			channel = 5;
		}
		if (i >= 7) { //sample 4 times, becaus its has a high impedance
			channel = 0x21;
		}
		if (i < 6) { //1.1Vref
			ADMUX = channel | (1<<REFS1);
		} else { //Vcc as reference
			ADMUX = channel;
		}
		ADCSRA = (1<<ADEN) | (1<<ADSC) | (1<<ADPS1) | (1<<ADPS2);
		while (ADCSRA & (1<<ADSC)); //wait for complete
		if (i >= 8) {
			ad_data[7] = ADC;
		} else {
			ad_data[i] = ADC;
		}
	}
	//convert to °C with 1.1Vref
	ad_data[0] = temp_convert(ad_data[0]);
	ad_data[1] = temp_convert(ad_data[1]);
	ad_data[2] = temp_convert(ad_data[2]);
	//convert to mA with 1Ohm and 1.1Vref
	ad_data[4] = (uint32_t)ad_data[4]*1100/1023;
	//convert to mV with 1:11 divider and 1.1Vref
	ad_data[3] = (uint32_t)ad_data[3]*1183/100;
	ad_data[5] = (uint32_t)ad_data[5]*1183/100;
	//convert to mV with 1:2 divider and 3.3Vref
	ad_data[6] = (uint32_t)ad_data[6]*645/100;
	//convert to mV from known 1.1Vmeasure to the value of Vref
	ad_data[7] = 1125000/ad_data[7];
}

static void debugprint(uint8_t opmode, uint8_t errormode, uint16_t totalcharge) {
	static uint8_t iter = 0;
	if (iter < 8) {
		print_u16(ad_data[iter]);
		print_p(PSTR(" "));
	}
	if (iter == 9) {
		print_p(PSTR(" -- "));
		print_u16(OCR1A);
	}
	if (iter == 10) {
		print_p(PSTR(" "));
		print_u16(opmode);
	}
	if (iter == 11) {
		print_p(PSTR(" "));
		print_u16(errormode);
	}
	if (iter == 12) {
		print_p(PSTR(" "));
		print_u16(totalcharge);
		print_p(PSTR("\n\r"));
		iter = 255;
	}
	iter++;
}

static uint8_t overtemperature(void) {
	if ((ad_data[0] > BAT_TEMP_MAX) ||
	    (ad_data[1] > TEMP_MAX) ||
	    (ad_data[2] > BAT_TEMP_MAX)) {
		return 1;
	}
	return 0;
}

static uint8_t mcuvccwrong(void) {
	if ((ad_data[7] > MCU_MV_MAX) || (ad_data[7] < MCU_MV_MIN)) {
		return 1;
	}
	return 0;
}

uint16_t readeep(void) {
	mamininit = eeprom_read_dword((uint32_t *) 4);
	uint32_t mamininit2 = eeprom_read_dword((uint32_t *) 8);
	uint16_t eepbroken = eeprom_read_word((uint16_t *) 12);
	if (eepbroken == 0xFFFF) {
		eepbroken = 0;
	}
	if (mamininit == 0xFFFFFFFF) {
		mamininit = 0;
	} else if (mamininit > mamininit2) {
		mamininit = mamininit2;
		eepbroken++;
		eeprom_write_word((uint16_t *) 8, eepbroken);
	}
	maminlast = mamininit;
	return eepbroken;
}

void writeeep(uint32_t mas) {
	uint32_t mamin = mamininit + mas/60;
	if ((mamin != maminlast) && (mcuvccwrong()) && (ad_data[7] > MCU_MV_EEP_MIN)) {
		eeprom_write_dword((uint32_t *) 4, mamin);
		eeprom_write_dword((uint32_t *) 8, mamin);
		maminlast = mamin;
	}
}

int main(void) {
//Init I/O
	WDTCSR = (1<<WDP2) | (1<<WDP1) | (1<<WDP0); //2s
	wdt_reset();
	WDTCSR |= (1<<WDE);
	PORTA = 0;
	DDRA = (1<<PINA6); //PWM output
	PORTB = 0;
	DDRB = (1<<PINB0) | (1<<PINB1) | (1<<PINB2); //three LEDs
//	CLKPR = (1<<CLKPCE);
//	CLKPR = (1<<CLKPS2); //500khz clock
//early message
	print_p(PSTR("\n\rFahrradlader 1.0, by Malte Marwedel\n\r"));
//check if debugmode should be used
	if (MCUSR & (1<< EXTRF)) {
		debugmode = 1;
	}
	MCUSR = 0;
//init A/D converter
	DIDR0 = 0; //saves power
	ADCSRB = 0;
//init PWM
	TCNT1 = 0;
	OCR1A = 0; //set value 0...0xFF to control current
	TIMSK1 = 0;
	TCCR1A = (1<<COM1A1) | (1<<WGM10); //8Bit fast pwm
	TCCR1B = (1<<WGM12) | (1<<CS10); //8bit , divide by 1
	TCCR1C = 0;
//read eep
	uint16_t eepbroken = readeep();
//first sample and print results
	ad_updateall();
	uint8_t i;
	for (i = 0; i < 8; i++) {
		print_u16(ad_data[i]);
		print_p(PSTR("\n\r"));
	}
	print_u32(mamininit);
	print_p(PSTR("\n\r"));
	print_u16(eepbroken);
	print_p(PSTR("\n\r"));
//init LED blinking
	TCNT0 = 0;
	OCR0A = 98; //1000kHz/1024/98 ~=10Hz frequency
	TCCR0A = (1<<WGM01); //clear timer on compare
	TIMSK0 = (1<<OCIE0A); //compare match interrupt
	TCCR0B = (1<<CS02) | (1<<CS00); //divide by 1024
	//globally enable interrupts
	sei();
	uint8_t opmode = 0; //0 = nothing, 1 = USB, 2 = charge
 //0 = operating, 1 = battery left full, 2 = battery righ full 3 = both
	uint8_t battfull = 0;
	/*
		0: all well
		1: usb voltage too high
		2: usb voltage too low
		3: overtemperature
		4: failure of input overvoltage protection
		5: 3,3V out of range!

	*/
	uint8_t errormode = 0;
	uint16_t lastcurrent = 0;
	uint8_t secondtick = 0;
	uint32_t totalcharge = 0; //mAs
	uint16_t usbmin = 0;
	//main loop
	while (1) {
		if (debugmode) {
			debugprint(opmode, errormode, totalcharge/60/60);
		}
		lastcurrent = ad_data[4];
		ad_updateall();
		wdt_reset();
		writeeep(totalcharge);
		//determine mode of OP
		errormode = 0;
		if (ad_data[3] > CHARGE_MV_MIN) { //Vin
			//charge mode
			opmode = 2;
			if (ad_data[3] > CHARGE_MV_GOOD_MAX) { //vin overvoltage
				errormode = 4;
			} else if (overtemperature()) {
				errormode = 3;
			} else if (mcuvccwrong()) {
				errormode = 5;
			} else {
				led_states[LED_GREEN] = LED_OFF;
				if (battfull == 0) { //try charge
					if ((ad_data[3] > CHARGE_MV_MIN_RAMPUP) &&
					    (ad_data[4] < CHARGE_MA_MAX_RAMPUP) &&
					    ((lastcurrent + CHARGE_MA_MAXINC_RAMPUP) > ad_data[4]) &&
					    ((OCR1A < (PWM_MAX/3)) ||
					    ((ad_data[4] > CHARGE_MA_MIN_RAMPFURTHERUP) && (OCR1A < PWM_MAX)))) {
						/*Increase current if input voltage is high enough and
						  output current is low enough. but only if OCR1A is very low or
						  we already measured some current and value is not the maximum.
						  Moreove, the current may not increase more than 10mA per iteration
						*/
						OCR1A++;
					}
					if (((ad_data[3] < CHARGE_MV_MAX_RAMPDOWN) ||
						   (ad_data[4] > CHARGE_MA_MIN_RAMPDOWN) ||
						   ((ad_data[4] < CHARGE_MA_MIN_RAMPFURTHERUP) &&
						    (OCR1A > (PWM_MAX/3)))) && (OCR1A > 0)) {
						//decrease current
						OCR1A--;
					}
					if (((ad_data[3] < CHARGE_MV_MAX_RAMPDOWNFAST) ||
					    (ad_data[4] > CHARGE_MA_MIN_RAMPDOWNFAST)) && (OCR1A > 0)) {
						//fast decrease current
						OCR1A--;
					}
					if (ad_data[4] > CHARGE_MA_MIN_FULLSTOP) { //way too high current
						OCR1A = 0;
					}
					//check LEDs mark other batteries as full if temperatue is close to
					//threshold
					if (ad_data[4] > 10) {
						led_states[LED_RED_RIGHT] = LED_ON;
						led_states[LED_RED_LEFT] = LED_ON;
					} else {
						led_states[LED_RED_RIGHT] = LED_OFF;
						led_states[LED_RED_LEFT] = LED_OFF;
					}
					//check temperature
					if ((ad_data[0] >= ad_data[1] + BAT_TEMP_DELTA) ||
							(ad_data[0] >= BAT_TEMP_MAX-1)) {
						battfull |= 1;
						if (ad_data[2] >= ad_data[1] + BAT_TEMP_DELTA-2) {
							battfull |= 2;
						}
					}
					if ((ad_data[2] >= ad_data[1] + BAT_TEMP_DELTA) ||
					    (ad_data[2] >= BAT_TEMP_MAX-1)) {
						battfull |= 2;
						if (ad_data[0] >= ad_data[1] + BAT_TEMP_DELTA-2) {
							battfull |= 1;
						}
					}
					if (totalcharge > BAT_MAS_MAX) { //failsafe
						battfull = 3;
					}
				} else { //batteries charged
					OCR1A = 0;
					if (battfull & 1) {
						led_states[LED_RED_RIGHT] = LED_SHORTOFF;
					} else {
						led_states[LED_RED_RIGHT] = LED_OFF;
					}
					if (battfull & 2) {
						led_states[LED_RED_LEFT] = LED_SHORTOFF;
					} else {
						led_states[LED_RED_LEFT] = LED_OFF;
					}
				}
			}
		} else if (ad_data[6] > USB_MV_MIN_MODE) {//VUsb
			OCR1A = 0;
			//USB mode
			opmode = 1;
			if (ad_data[6] >= USB_MV_MIN_ERROR) { //too high VUSB
				errormode = 1;
			} else if (overtemperature()) {
				errormode = 3;
			} else if (mcuvccwrong()) {
				errormode = 5;
			} else {
				if ((ad_data[6] < USB_MV_MIN_TOOLOW) || (usbmin)) { //too low VUSB
					errormode = 2;
					if (ad_data[6] < USB_MV_MIN_TOOLOW) {
						//sometimes only single samples are below the threshold
						usbmin = 4; //signal too low for 3...4sec
					}
					led_states[LED_GREEN] = LED_SLOWBLINK;
				} else if ((ad_data[6]+39) < ad_data[3]) { //charging with more than 100mA
					//no error condition detected
					led_states[LED_GREEN] = LED_SHORTOFF;
				} else {
					//no error condition detected
					led_states[LED_GREEN] = LED_ON;
				}
				if (ad_data[5] < BAT_MIN) {
					led_states[LED_RED_LEFT] = LED_SLOWBLINK;
				} else {
					led_states[LED_RED_LEFT] = LED_OFF;
				}
				if (ad_data[3]-ad_data[5] < BAT_MIN) {
					led_states[LED_RED_RIGHT] = LED_SLOWBLINK;
				} else {
					led_states[LED_RED_RIGHT] = LED_OFF;
				}
			}
		} else {
			OCR1A = 0;
			opmode = 0;
			if (overtemperature()) {
				errormode = 3;
			} else if (mcuvccwrong()) {
				errormode = 5;
			} else {
				led_states[LED_GREEN] = LED_SHORTON;
				led_states[LED_RED_RIGHT] = LED_OFF;
				led_states[LED_RED_LEFT] = LED_OFF;
			}
		}
		//error modes with led states
		if (errormode) {
			OCR1A = 0; //power off
		}
		if (errormode == 1) { //too high usb voltage!
			led_states[LED_GREEN] = LED_FASTBLINK;
			led_states[LED_RED_RIGHT] = LED_FASTBLINK;
			led_states[LED_RED_LEFT] = LED_FASTBLINK;
		}
		if (errormode == 3) { //too high temperature
			led_states[LED_GREEN] = LED_ON;
			led_states[LED_RED_RIGHT] = LED_FASTBLINK;
			led_states[LED_RED_LEFT] = LED_FASTBLINK;
		}
		if (errormode == 4) { //too high vin
			led_states[LED_GREEN] = LED_SLOWBLINK;
			led_states[LED_RED_RIGHT] = LED_FASTBLINK;
			led_states[LED_RED_LEFT] = LED_FASTBLINK;
		}
		if (errormode == 5) { //3,3V out of range
			led_states[LED_GREEN] = LED_FASTBLINK;
			led_states[LED_RED_RIGHT] = LED_FASTBLINK;
			led_states[LED_RED_LEFT] = LED_OFF;
		}
		//count total charged current, to switch off after 2Ah
		uint8_t localcounter = led_counter; //atomic read needed
		if ((secondtick) && (localcounter < 5)) {
			//second lapsed
			totalcharge += ad_data[4];
			secondtick = 0;
			if (usbmin) {
				usbmin--;
			}
		} else if (localcounter > 5) {
			secondtick = 1;
		}
	}
}
