/* rs232_io.c
Interface for the RS232 connector

Copyright (C) 2006-2007 by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "main.h"


void rs232_init(void) {
//this has to be done before scheduling starts, because of use for debugging
UBRR0L = UARTNUMBER;		//set speed
UCSR0C = (1<<UCSZ01) | (1<<UCSZ00); //set 8Bit mode
UCSR0B = 1<<RXEN0 | 1<<TXEN0;	//TX enabled, RX enabled
}


void rs232_put_byte(u08 byte) {
while (1) {	//not that elegant, but produces smaller code
  if (UCSR0A & (1<<UDRE0))	//break if UDR is empty
    break;
  if (sched_thread < SCHED_THREADS) {
    sched();
  }
}
UDR0 = byte;
}

u08 rs232_get_nextbyte(void) {
while(1) {	//not that elegant, but produces smaller code
  if (UCSR0A & (1<<RXC0))
    break;
  sched();
}
u08 k = UDR0;
return k;
}

void rs232_sendnewline(void) {
rs232_put_byte('\r');
rs232_put_byte('\n');
}


void rs232_thread(void) {
static u08 sendbuf[15];
static u32 timing;	//for sending stats every 250ms
u08 recch;
//only in op_mode==OP_MAIN we are willing to receive commands from the interface
//in all other modes, we send out the mode, voltage and current every 0,25sec
timer_setup(&timing, 250);
for (;;) {
  if (op_mode == OP_MAIN) {
    if (UCSR0A & (1<<RXC0)) {	//we received something
       recch = UDR0;
       if (recch == 'G') {	//Get: the PC wants the config
         eep_sendconfig();
       }
       if (recch == 'P') {	//Put in a new config
         u08 ch2 = rs232_get_nextbyte();
         if (ch2 == 'd') {    //d must follow, for not accidentally receive a P
           eep_receiveconfig();
         }
       }
       if (recch == 'R') {	//Reset
         reset();
       }
     }
    /*Otherwise the timer would fire multiple times after op_mode is != OP_MAIN
      again */
    timer_setup(&timing, 250);
  } else
  if (timer_renew(&timing, 250)) {
    //Sending format: I:M;VVVVV;AAAA\r\n
    sendbuf[0] = 'I';
    sendbuf[1] = ':';
    sendbuf[2] = op_mode+48;
    sendbuf[3] = ';';
    if (op_mode == OP_PROBE) {
      wordtostr(sendbuf, get_uppervoltage(), 5, 4);
    } else
      wordtostr(sendbuf, get_outputvoltage(), 5, 4);
    sendbuf[9] = ';';
    if (op_mode == OP_PROBE) {
      wordtostr(sendbuf, get_lowervoltage(), 4, 10);
    } else
      wordtostr(sendbuf, get_current(), 4, 10);
    sendbuf[14] = '\0';
    rs232_println(sendbuf);
    recch = UDR0;	//clearing of receive register
  }
  sched();
}
}

u08 volatile print_lock = 0;

void rs232_println(u08 *text) {	//Sends a string over rs232
/*This function may only be used if the scheduler is already running,
  otherwise you must call rs232_println_atomic();
  The simple locking mechanism, works only on cooperative scheduling and would
  fail on pre-emptive scheduling.
*/
while (print_lock) {
  sched();
}
print_lock = 1;
rs232_println_atomic(text);
print_lock = 0;
}

void rs232_println_atomic(u08 *text) {	//Sends a string over rs232
/*Not really atomic, but there is no sched() in this function.
  Note that rs232_put_byte() still calls sched() if scheduling is active.
*/
u08 k = 0;
u08 ch;
while ((ch = *(text+k)) != 0) {	//for every character of *text
  rs232_put_byte(ch);
  k++;
  if (k == 0) {		//security out if the string was not null terminated
    break;
  }
}
rs232_sendnewline();
}

/* Prints a line with a number on the RS232 connection. Mainly for
debugging purpose */
void rs232_print_number(u16 val) {
u08 text[6];
wordtostr(text, val, 5, 0);
text[5] = '\0';
rs232_println(text);
}

/* Prints a line with a 12 digits number on the RS232 connection. Mainly for
debugging purpose */
void rs232_print_number_u32(u32 val) {
u08 text[15];
text[0] = 'C'; //HACK: mark as commentary for the PC software
text[1] = ':';
wordtostr(text, (val/1000000)%1000, 4, 2); //upper part
wordtostr(text, (val/1000)%1000, 4, 6); //middle part
wordtostr(text, val%1000, 4, 10); //lower part
text[14] = '\0';
rs232_println(text);
}

