/* main.c
  File description
  Title:    NIMH Battery charger
  Author:   (c) 2006-2008 by Malte Marwedel
  Date:     2008-05-07
  Version:  Final 1.0
  Purpose:  Charge of NIMH batteries with constant current
  Software: AVR-GCC
  Hardware: ATMEGA168 with 8MHZ
  License:  GNU GENERAL PUBLIC LICENSE Version 2, June 1991
            This program is free software; you can redistribute it and/or modify
            it under the terms of the GNU General Public License as published by
            the Free Software Foundation; either version 2 of the License, or
            (at your option) any later version.

            This program is distributed in the hope that it will be useful,
            but WITHOUT ANY WARRANTY; without even the implied warranty of
            MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
            GNU General Public License for more details.

            You should have received a copy of the GNU General Public License
            along with this program; if not, write to the Free Software
        Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  US

  Wer Fragen oder Anregungen zu dem Programm hat, kann an
             m.marwedel <AT> onlinehome.de mailen.
  If you have questions or suggestions, mail me:
             m.marwedel <AT> onlinehome.de
  Homepage: http://www.marwedels.de/malte/
  Code size: below 15KB
  More details: see readme.txt


Pin connection of the circuit:
PB0: Output, LCD D7
PB1: Output, PWM charge
PB2: Output, PWM discharge
PB3: Output, LCD
PB4: Output, LCD
PB5: Output, LCD
PB6: Input, not available
PB7: Input, not available

PC0: Input, measure charge - discharge current (lower voltage)
PC1: Input, measure battery voltage (upper voltage)
PC2: Input, Key 1 (pull-up active)
PC3: Input, Key 2 (pull-up active)
PC4: Input, Key 3 (pull-up active)
PC5: Input, Key 4 (pull-up active)
PC6: Input, not available
PC7: Input, not available

PD0: Input,  RXD
PD1: Output, TXD
PD2: Output, status LED
PD3: Input, not used (pull-up active)
PD4: Input, not used (pull-up active)
PD5: Output, LCD RS
PD6: Output, LCD E
PD7: Input, not used (pull-up active)

*/

#include "main.h"

void reset(void) {
//An endless loop, the Watchdog will do the rest
sched_thread = WAS_WISHED;
  while(1);
}

const char starterror[] PROGMEM = "\r\nE: J   ";

/*
The etext variable is placed on the stack and if this would happen in the
main function, the part of the stack would be used once, and after
printing the content, it would stay there forever without being used because
the main function does not end.
So the noinline here simply saves some stack space.
*/
void tell_reset_source(u08 mcusr, u08 sched_thread_o)
                                                     __attribute__ ((noinline));
void tell_reset_source(u08 mcusr, u08 sched_thread_o) {
u08 erroroccured = 0;
u08 etext[10];
memcpy_P(&etext,&starterror,sizeof(etext));
if (mcusr & (1<<BORF)) {
  //brown out reset
  etext[5] = 'B';
  erroroccured = 1;
} else //Other errors show the task number
  wordtostr(etext, sched_thread_o ,2 ,7);
if (mcusr == 0) {
  //indicates that there was a direct jump to 0x0000 and no reset
  erroroccured = 2;
}
if ((mcusr & (1<<WDRF)) && (sched_thread_o < MAX_THREADS)) {
  //unplanned watchdog reset
  etext[5] = 'W';
  erroroccured = 1;
}
if (erroroccured) {		//print the error message
  rs232_println_atomic(etext);
  while (erroroccured > 1);	//Endless loop or jump to 0x0000 error
}
}

int main(void) {		//main routine
u08 mcusr = MCUSR;		//copy reset source
MCUSR = 0;			//clear for next use
//configure the watchdog
asm volatile ("wdr");		//resets the watchdog
WDTCSR = (1<<WDCE) | (1<< WDE);	//prepare watchdog
WDTCSR = (1<<WDE) | (1<< WDP3);	//enable with 4,0s
_delay_ms(30.0);		//wait after power on
//I/O Init
DDRB = 0x3f;
DDRC = 0x00;
DDRD = 0x66;
//pull-up
PORTC = 0x3c;
PORTD = (1<<3)|(1<<4)|(1<<7);
//early setup of threading status
u08 sched_thread_o = sched_thread;
sched_thread = NOT_RUNNING;
//early init of RS232 for debugging
rs232_init();
//display a message according to the reset source
tell_reset_source(mcusr, sched_thread_o);
//go on with init
valid_prepare(570);
eep_loadconf();
clock_init();
sei();
//decrease watchdog timeout period
asm volatile ("wdr");		//resets the watchdog
WDTCSR = (1<<WDCE) | (1<< WDE);	//prepare watchdog
WDTCSR = (1<<WDE);	//enable with 16ms
sched_init();
/*sched_init() will never return anyway, but the compiler does not know this.
  So the while(1); only tells the compiler that main() does not return and that
  the compiler does not need to load values for the returned int into the
  registers. Or to say it short: This saves 6 bytes of flash memory.
*/
while(1);
}
