package view;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.KeyStroke;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import control.GraphDrawer;
import control.Main;
import control.Session;
import control.Tools;
import data.DataEntry;

public class RecordWindow implements ComponentListener, ActionListener, AdjustmentListener,
                                     ListSelectionListener, WindowListener {

  private final int yoffset = 40; //place for the header
  private int imagesizex;
  private int imagesizey;
  private int windowsizex = 500;
  private int windowsizey = 400;
  
  private GraphDrawer gd;
  
  private JFrame frame;           //The main frame
  private JPanel panelmain;       //The main panel with everything
  private JScrollBar scrollzoom;  //How much is visible
  private JScrollBar scrollpartvisible;  //which part is visible 
  private JScrollPane scrolllist; 
  private JList listdata;
  private int listdataElems;
  private JMenuBar jmainmenu;
  private JLabel jlimage;
  
  private JLabel lvolts;
  private JLabel lmilis;
  private JLabel lpower;
  
  private JCheckBoxMenuItem itvolts;
  private JCheckBoxMenuItem itmilis;
  private JCheckBoxMenuItem itpower;
  private JMenu itavg;
  private JCheckBoxMenuItem itavauto;
  private ButtonGroup bgav;
  private JRadioButtonMenuItem itav1;
  private JRadioButtonMenuItem itav2;
  private JRadioButtonMenuItem itav4;
  private JRadioButtonMenuItem itav8;
  private JRadioButtonMenuItem itav16;
  private JRadioButtonMenuItem itav32;
  private JRadioButtonMenuItem itav50s;
  private JMenu menuadv;
  private JMenuItem iadvadv;
    
  private Vector<DataEntry> vde;
  
  private Session sess;
  
  private boolean stopRedrawing = true; //may not be needed, but makes things a lot faster
  
  public RecordWindow(Session s) {
    sess = s;
    //Create and set the window
    frame = new JFrame();
    frame.addWindowListener(this);
    setFilename("Unknown");  
    //frame.setDefaultCloseOperation(JFrame.);
    //Generate the Layout
    panelmain = new JPanel();
    panelmain.setLayout(null);
    //Make menus
    jmainmenu = new JMenuBar();
    JMenu menur = new JMenu("Record");
    JMenu menuview = new JMenu("View");
    menuadv = new JMenu("Advanced");
    jmainmenu.add(menur);
    jmainmenu.add(menuview);
    jmainmenu.add(menuadv);
    JMenuItem itselectall = new JMenuItem("Select all");
    itselectall.setActionCommand("selectall");
    itselectall.addActionListener(this);
    itvolts = new JCheckBoxMenuItem("Show voltage", true);
    itvolts.setActionCommand("togglevolts");
    itvolts.addActionListener(this);
    itmilis = new JCheckBoxMenuItem("Show current", true);
    itmilis.setActionCommand("toggleampere");
    itmilis.addActionListener(this);    
    itpower = new JCheckBoxMenuItem("Show power", true);
    itpower.setActionCommand("togglepower");
    itpower.addActionListener(this);
    JMenuItem itsave = new JMenuItem("Save");
    itsave.setActionCommand("save");
    itsave.addActionListener(this);
    JMenuItem itsaveas = new JMenuItem("Save as");
    itsaveas.setActionCommand("saveAs");
    itsaveas.addActionListener(this);
    JMenuItem itexport = new JMenuItem("Export");
    itexport.setActionCommand("export");
    itexport.addActionListener(this);
    JMenuItem itclose = new JMenuItem("Close");
    itclose.setActionCommand("close");
    itclose.addActionListener(this);
    menur.add(itsave);
    menur.add(itsaveas);
    menur.add(itexport);
    menur.add(itclose);
    itavg = new JMenu ("Averaging");
    itavauto = new JCheckBoxMenuItem("Auto select", true); 
    itavauto.setActionCommand("autoaverage");
    itavauto.addActionListener(this);
    itav1 = new JRadioButtonMenuItem("None");
    itav1.setSelected(true);
    itav1.setActionCommand("avRad1x");
    itav1.addActionListener(this);
    itav2 = new JRadioButtonMenuItem("2x");
    itav2.setActionCommand("avRad2x");
    itav2.addActionListener(this);
    itav4 = new JRadioButtonMenuItem("4x");
    itav4.setActionCommand("avRad4x");
    itav4.addActionListener(this);
    itav8 = new JRadioButtonMenuItem("8x");
    itav8.setActionCommand("avRad8x");
    itav8.addActionListener(this);
    itav16 = new JRadioButtonMenuItem("16x");
    itav16.setActionCommand("avRad16x");
    itav16.addActionListener(this);
    itav32 = new JRadioButtonMenuItem("32x");
    itav32.setActionCommand("avRad32x");
    itav32.addActionListener(this);
    itav50s = new JRadioButtonMenuItem("50 segments");
    itav50s.setActionCommand("avRad50s");
    itav50s.addActionListener(this);
    updateAverageButtons();
    menuview.add(itavg);
    menuview.add(itselectall);
    menuview.add(itvolts);
    menuview.add(itmilis);
    menuview.add(itpower);
    itavg.add(itavauto);
    itavg.add(itav1);
    itavg.add(itav2);
    itavg.add(itav4);
    itavg.add(itav8);
    itavg.add(itav16);
    itavg.add(itav32);
    itavg.add(itav50s);    
    bgav = new ButtonGroup();
    bgav.add(itav1);
    bgav.add(itav2);
    bgav.add(itav4);
    bgav.add(itav8);
    bgav.add(itav16);
    bgav.add(itav32);
    bgav.add(itav50s);
    JMenuItem stats = new JMenuItem("Statistics");
    stats.setActionCommand("stats");
    stats.addActionListener(this);
    menuadv.add(stats);
    menuadv.add(new JSeparator());
    if (Main.DEVELMODE) {
    	activateAdvancedItems();
    } else {
    	iadvadv = new JMenuItem("Enable unsupported functions");
    	iadvadv.setActionCommand("enableunsupp");
    	iadvadv.addActionListener(this);
    	menuadv.add(iadvadv);
    }
    //makes drawing object
    gd = new GraphDrawer();
    //Place scrollbars
    scrollzoom = new JScrollBar(JScrollBar.VERTICAL, 1, 10, 1, 100);
    scrollzoom.setToolTipText("Zoom");
    scrollzoom.setMinimum(0);
    scrollzoom.addAdjustmentListener(this);
    scrollpartvisible = new JScrollBar(JScrollBar.HORIZONTAL, 1, 10, 1, 100);
    scrollpartvisible.setToolTipText("Horizontal (Time) View Scroller");
    scrollpartvisible.addAdjustmentListener(this);
    panelmain.add(scrollzoom);
    panelmain.add(scrollpartvisible);    
    //add list
    listdata = new JList();
    listdata.setAutoscrolls(true);
    listdata.setToolTipText("Hold STRG or CTRL key for multiple selections");
    listdata.addListSelectionListener(this);    
    scrolllist = new JScrollPane(listdata);
    panelmain.add(scrolllist);
    //add labels
    lvolts = new JLabel("mV");
    lvolts.setToolTipText("red graph");
    lmilis = new JLabel("mA");
    lmilis.setToolTipText("green graph");
    lpower = new JLabel("mW");
    lpower.setToolTipText("blue graph");
    panelmain.add(lvolts);
    panelmain.add(lmilis);
    panelmain.add(lpower);
    //Display the window
    frame.setJMenuBar(jmainmenu);
    frame.add(panelmain);
    frame.pack();
    frame.setLocation(50, 195);
    frame.setSize(windowsizex, windowsizey);
    stopRedrawing = false;
    resize();
    frame.addComponentListener(this);
    frame.setVisible(true); 
  }
  
  /**
   * Please call redraw (if needed) that manually
   *
   */
  public void resize() {
  	boolean stopRedrawinglocal = stopRedrawing;
  	stopRedrawing = true;
    int sizex = frame.getWidth();
    int sizey = frame.getHeight();
    if (sizex < 500) sizex = 500;
    if (sizey < 400) sizey = 400-yoffset;
    sizey -= yoffset;
    jmainmenu.setBounds(0, 0, sizex, 20);
    scrolllist.setBounds(10, 10, 250, 80);
    listdata.setBounds(0, 0, 250, 80);
    imagesizex = sizex-50;
    imagesizey = sizey-160;
    gd.updateSize(imagesizex, imagesizey);
    updateScrollzoomRange();
    scrollzoom.setBounds(sizex-30,100,20,imagesizey);
    scrollpartvisible.setBounds(10, sizey-50, imagesizex, 20);
    stopRedrawing = stopRedrawinglocal;
    lvolts.setBounds(270, 10, sizex-280, 24);
    lmilis.setBounds(270, 35, sizex-280, 24);
    lpower.setBounds(270, 60, sizex-280, 24);
		//panelmain.updateUI();
  }
  
  public void redraw() {
  	if (stopRedrawing) return;
  	//System.out.println("Zoom: "+scrollzoom.getValue());
  	//Tools.debugWho();
  	//select the Data in the vertical selection
  	if (vde != null) {
  		Vector<DataEntry> selde = new Vector<DataEntry>();
  		int end = Math.min(scrollpartvisible.getValue()+datasetsShown(), vde.size());
  		for (int i = scrollpartvisible.getValue(); i < end; i++) {
  			selde.add(vde.get(i));
  		}
  		//look if line may be drawn
  		//if (scrollzoom.getValue() != 1)
  		//gd.setLine(-1); //make invisible
  		//generate Vectors for drawing
  		Vector<Double> graph1 = new Vector<Double>();
  		Vector<Double> graph2 = new Vector<Double>();
  		Vector<Double> graph3 = new Vector<Double>();
  		int graph1max = getMaximumVoltage(selde);
  		int graph1min = getMinimumVoltage(selde);
  		double graph1delta = graph1max-graph1min;
  		int graph2max = getMaximumAmps(selde);
  		int graph2min = getMinimumAmps(selde);
  		double graph2delta = graph2max-graph2min;
  		int graph3max = getMaximumPower(selde);
  		int graph3min = getMinimumPower(selde);
  		double graph3delta = graph3max-graph3min;
  		//update labels
  		lvolts.setText("Voltage: "+graph1min+"mV ... "+graph1max+"mV");
  		lmilis.setText("Current: "+graph2min+"mA ... "+graph2max+"mA");
  		lpower.setText("Power: "+graph3min+"mW ... "+graph3max+"mW");
  		//some calculations to determine the range for drawing
  		if (graph1delta < imagesizey) { 
  			graph1min -= (imagesizey-graph1delta)/2; //set to the middle 
  			graph1delta = imagesizey;
  		}
  		if (graph2delta < imagesizey) { 
  			graph2min -= (imagesizey-graph2delta)/2; //set to the middle 
  			graph2delta = imagesizey;
  		}
  		if (graph3delta < imagesizey) { 
  			graph3min -= (imagesizey-graph3delta)/2; //set to the middle 
  			graph3delta = imagesizey;
  		}
  		for (int i = 0; i < selde.size(); i++) {
          if (itvolts.isSelected())
  			graph1.add(new Double(((double)(selde.get(i).getVoltage()-graph1min))/graph1delta));
          if (itmilis.isSelected())     
  			graph2.add(new Double(((double)(selde.get(i).getCurrent()-graph2min))/graph2delta));
          if (itpower.isSelected())
  			graph3.add(new Double(((double)(selde.get(i).getPower()-graph3min))/graph3delta));
  		}
  		//System.out.println("Entries: "+graph1.size());
        if (itvolts.isSelected() == false) graph1 = null;
        if (itmilis.isSelected() == false) graph2 = null;
        if (itpower.isSelected() == false) graph3 = null;
  		gd.updateData(graph1, graph2, graph3);
  	}
  	if (gd.isHasUpdated()) {
      //add the image
      if (jlimage != null) {
        jlimage.removeAll();
        panelmain.remove(jlimage);
        jlimage = null;
      }    
      jlimage = new JLabel(gd.getGraphic());
      jlimage.setBounds(10, 100, imagesizex, imagesizey);
      panelmain.add(jlimage);
  	  jlimage.updateUI();
  	}
  	
  }
  
	private int getMinimumPower(Vector<DataEntry> graph) {
		int k = Integer.MAX_VALUE;
		for (int i = 0; i < graph.size(); i++) {
			int val = graph.get(i).getPower();
			k = Math.min(k, val);
		}
		return k;
	}
	
	private int getMinimumVoltage(Vector<DataEntry> graph) {
		int k = Integer.MAX_VALUE;
		for (int i = 0; i < graph.size(); i++) {
			int val = graph.get(i).getVoltage();
			k = Math.min(k, val);
		}
		return k;
	}
	
	private int getMinimumAmps(Vector<DataEntry> graph) {
		int k = Integer.MAX_VALUE;
		for (int i = 0; i < graph.size(); i++) {
			int val = graph.get(i).getCurrent();
			k = Math.min(k, val);
		}
		return k;
	}

	private int getMaximumPower(Vector<DataEntry> graph) {
		int k = Integer.MIN_VALUE;
		for (int i = 0; i < graph.size(); i++) {
			int val = graph.get(i).getPower();
			k = Math.max(k, val);
		}
		return k;
	}
	
	private int getMaximumVoltage(Vector<DataEntry> graph) {
		int k = Integer.MIN_VALUE;
		for (int i = 0; i < graph.size(); i++) {
			int val = graph.get(i).getVoltage();
			k = Math.max(k, val);
		}
		return k;
	}
	
	private int getMaximumAmps(Vector<DataEntry> graph) {
		int k = Integer.MIN_VALUE;
		for (int i = 0; i < graph.size(); i++) {
			int val = graph.get(i).getCurrent();
			k = Math.max(k, val);
		}
		return k;
	}
	
	/**
   * Sets all data which may be useful for drawing.
   * @param ud data selected in the list
   */
  public void setDisplayData(Vector<DataEntry> de) {
  	vde = de;
  	updateScrollzoomRange();
  	redraw();
  }
  
  private void updateScrollzoomRange() {
  	boolean stopRedrawinglocal = stopRedrawing;
  	stopRedrawing = true;
  	int scmax;  	
  	try {
  		double dscmax;
  		dscmax = Math.log(((double)imagesizex)/((double)vde.size()))/Math.log(0.9);
  		dscmax += 1.0;
  		scmax = (int)dscmax;
  	} catch (Exception e) {
  		scmax = 1;
  	}
  	scmax = Math.max(scmax, 0);
  	int scval = scrollzoom.getValue();
  	if (scval > scmax)
  		scrollzoom.setValue(scval);
  	scrollzoom.setMaximum(scmax);
  	scrollzoom.setValue(scval); //because it looks like set maximum resets this value
  	//System.out.println("Update scrollzoomrange: max:"+scrollzoom.getMaximum()+" val:"+scrollzoom.getValue());
  	//Tools.debugWho();
  	adjustscrollpart();
  	stopRedrawing = stopRedrawinglocal;
  }
  
  public void setRecordedData(Vector<String> sel) {
  	//System.out.println("setRecordedData: "+sel.size());
  	int[] selected = listdata.getSelectedIndices();
  	listdata.removeListSelectionListener(this);
  	//problem setListData fires listener without selected data -> useless redraw, cant use zoom while recording
  	listdata.removeAll();
  	listdata.setListData(sel);
  	listdataElems = sel.size();
  	if (Tools.max(selected) < listdataElems) //set selected only if all are within the range
    	listdata.setSelectedIndices(selected);
  	listdata.addListSelectionListener(this);
  	sess.messageTell("newRecordSelections"); //now force a reread of the values + redraw
  	//listdata.updateUI(); //Results in a failure if called directly on the program start (opening a file given on the command line)
  }

  private int datasetsShown() {
  	double d = ((double)vde.size())*Math.pow(0.9, scrollzoom.getValue());
  	int di = (int)(d+0.5);
  	if (di < imagesizex) {
  		di = imagesizex;
  	}
  	return di;
  }
  
  private void adjustscrollpart() {
    if (vde == null)
      return;
    scrollpartvisible.setMinimum(0);
    scrollpartvisible.setMaximum(vde.size());
    scrollpartvisible.setVisibleAmount(datasetsShown());
    if (scrollpartvisible.getValue() > (vde.size()-datasetsShown()))
      scrollpartvisible.setValue(vde.size()-datasetsShown());
  	scrollpartvisible.setMaximum(vde.size());
  }
  
  public void actionPerformed(ActionEvent e) {
    if (e.getActionCommand().equals("selectall")) {
      selectAll();
    } else if (e.getActionCommand().equals("togglevolts")) {
      redraw();
    } else if (e.getActionCommand().equals("toggleampere")) {
      redraw();
    } else if (e.getActionCommand().equals("togglepower")) {
      redraw();
    } else if (e.getActionCommand().equals("enableunsupp")) {
    	activateAdvancedItems();
    }
    if (e.getActionCommand().equals("autoaverage")) {
      updateAverageButtons();
      if (itavauto.isSelected()) {      //only update if now selected, because otherwise nothing changes in the graph
        sess.messageTell(e.getActionCommand()); 
      }
    } else if ((e.getActionCommand().startsWith("avRad") == false) || (itavauto.isSelected() == false))
      sess.messageTell(e.getActionCommand());   //important, get called on most events (but not on all to prevent endless loops or things like that
  }

  private void updateAverageButtons() {
    if (itavauto.isSelected()) {
      itav1.setEnabled(false);
      itav2.setEnabled(false);
      itav4.setEnabled(false);
      itav8.setEnabled(false);
      itav16.setEnabled(false);
      itav32.setEnabled(false);
      itav50s.setEnabled(false);
    } else {
      itav1.setEnabled(true);
      itav2.setEnabled(true);
      itav4.setEnabled(true);
      itav8.setEnabled(true);
      itav16.setEnabled(true);
      itav32.setEnabled(true);
      itav50s.setEnabled(true);
    }
  }
  
  public void componentHidden(ComponentEvent e) {
    //ignored    
  }

  public void componentMoved(ComponentEvent e) {
    //ignored    
  }

  public void componentResized(ComponentEvent e) {
  	int sizex = frame.getWidth();
    int sizey = frame.getHeight();
    if ((sizex != windowsizex) || (sizey != windowsizey)) {
      resize();
      redraw();
      windowsizex = sizex;
      windowsizey = sizey;
    }
  }

  public void componentShown(ComponentEvent e) {
    //ignored    
  }

  public void adjustmentValueChanged(AdjustmentEvent e) {
    if (e.getSource().equals(scrollzoom)) {
      adjustscrollpart();
    }
    if (e.getSource().equals(scrollpartvisible)) { //get called by adjustscrollpart() anyway
      redraw();
    }
  }

  public void valueChanged(ListSelectionEvent e) {
    //there is only the one list
    sess.messageTell("newRecordSelections");
  }

  public boolean isSelectedRecord(int usefulindex) {
    return listdata.isSelectedIndex(usefulindex);
  }
    
  public void selectAll() {
    int ind[] = new int[listdataElems];
    for (int i = 0; i < listdataElems; i++) {
      ind[i] = i;
    }
    listdata.removeListSelectionListener(this); //because for EVERY selected element all listeners get called -> meaning datafetching + redrawing
    listdata.setSelectedIndices(ind);
    listdata.addListSelectionListener(this);
    valueChanged(new ListSelectionEvent(listdata, 0, listdataElems-1, false));
  }
    
  public int getUpdatedAvgFactor(int dataentries) {
    if (itavauto.isSelected()) {
      //try to put 4 values to each pixel column
      int scaler = dataentries / (windowsizex*4);
      if (scaler < 2) {
        itav1.setSelected(true);
      } else if (scaler < 4) {
        itav2.setSelected(true);
      } else if (scaler < 8) {
        itav4.setSelected(true);
      } else if (scaler < 16) {
        itav8.setSelected(true);
      } else if (scaler < 32) {
        itav16.setSelected(true);
      } else {
        itav32.setSelected(true);
      }
    }
    int avf = 1; 
    if (itav1.isSelected()) avf = 1;
    if (itav2.isSelected()) avf = 2;
    if (itav4.isSelected()) avf = 4;
    if (itav8.isSelected()) avf = 8;
    if (itav16.isSelected()) avf = 16;
    if (itav32.isSelected()) avf = 32;
    if (itav50s.isSelected()) avf = dataentries/50;
    return avf;
  }
  
  public void setFilename(String name) {
    frame.setTitle("Record - "+name);
  }
  
  /**
   * Tries to free some memory by setting references to null pointers and then make the window invisible and runn GC. 
   *
   */
  public void destroy() {
  	vde = null;
  	sess = null;
  	gd.updateData(null, null, null);
  	gd = null;
  	frame.setVisible(false);
  	frame.dispose();
  	frame.removeAll();
  }

	public void windowActivated(WindowEvent e) {
		//ignore
	}

	public void windowClosed(WindowEvent e) {
		//ignore
	}

	public void windowClosing(WindowEvent e) {
		frame.removeWindowListener(this); //do not receive multiple events
		sess.messageTell("close");
	}

	public void windowDeactivated(WindowEvent e) {
		//ignore
	}

	public void windowDeiconified(WindowEvent e) {
		//ignore
	}

	public void windowIconified(WindowEvent e) {
		//ignore
	}

	public void windowOpened(WindowEvent e) {
		//ignore
	}

	public void setLine(int stopTime, int entriesSel) {
		gd.setLine(stopTime/getUpdatedAvgFactor(entriesSel));
	}
	
	private void activateAdvancedItems() {
		if (!Main.DEVELMODE) {
			new ShowMessage("Now some 'private' testing functions are selectable in the 'Advanced' menu.\n\r" +
					"They are entirely unsupported, may permanently alter the recorded data and/or " +
					"fire Exceptions under some conditions.\n\r" +
					"If you want to know what they do, read the sourcecode and don't ask me :-)");
			try {
				menuadv.remove(iadvadv);
			} catch (NullPointerException e) {
				//do not worry, an element which does not exists has no need to be removed
			}
		}
	  JMenuItem analyze1 = new JMenuItem("Full detect -DeltaU");
    analyze1.setActionCommand("fd1");
    analyze1.addActionListener(this);
    menuadv.add(analyze1);
    JMenuItem analyze2 = new JMenuItem("Full detect DeltaDeltaZero");
    analyze2.setActionCommand("fd2");
    analyze2.addActionListener(this);
    menuadv.add(analyze2);
    JMenuItem analyze3 = new JMenuItem("Full detect -DeltaU noise reduced");
    analyze3.setActionCommand("fd3");
    analyze3.addActionListener(this);
    menuadv.add(analyze3);
    JMenuItem analyze4 = new JMenuItem("Full detect final dynamic");
    analyze4.setActionCommand("fd6");
    analyze4.addActionListener(this);
    if (Main.DEVELMODE) {
    	KeyStroke f12key = KeyStroke.getKeyStroke("F12");
    	analyze4.setAccelerator(f12key);
    }
    menuadv.add(analyze4);
    JMenuItem analyze5 = new JMenuItem("Full detect final fixed");
    analyze5.setActionCommand("fd7");
    analyze5.addActionListener(this);
    menuadv.add(analyze5);
    JMenuItem bad1 = new JMenuItem("Bad detect 1");
    bad1.setActionCommand("fd4");
    bad1.addActionListener(this);
    menuadv.add(bad1);
    JMenuItem bad2 = new JMenuItem("Bad detect 2");
    bad2.setActionCommand("fd5");
    bad2.addActionListener(this);
    menuadv.add(bad2);
    menuadv.add(new JSeparator());
    JMenuItem smooth1 = new JMenuItem("Smooth 1");
    smooth1.setActionCommand("smooth1");
    smooth1.addActionListener(this);
    menuadv.add(smooth1);
	}
  
}
