package control;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Vector;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import control.detector.BadDetect;
import control.detector.BadDetect2;
import control.detector.NegDeltaU;
import control.detector.NegDeltaUFinal;
import control.detector.NegDeltaUFinal2;
import control.detector.NoDeltaDeltaU;
import control.detector.NegDeltaUNoNoise;

import model.FullDetect;
import model.ReceiveMessage;
import model.RecordedData;

import data.ChargingData;
import data.DataEntry;
import data.DischargingData;
import data.OtherData;
import data.SessionNotifier;
import data.UsefulData;

import view.ChooseFile;
import view.RecordInfo;
import view.RecordWindow;
import view.ShowMessage;
import view.ShowProgressbar;
import view.ShowQuestion;

public class Session implements ReceiveMessage {

	private ArrayList<RecordedData> superdata;
	private RecordWindow rw;
	private RecordInfo ri;
	private String filename = "New";
	private String dirname = "New"; 
	private String userComment;
	private boolean modified = false;
	
	
  /**
   * Opens the file and imports the data
   * @param importfile the filename to open
   */
  public Session (File file, boolean iscompressed) {
    filename = file.getName();
    dirname = file.getPath();
    rw = new RecordWindow(this);
    ri = new RecordInfo();
    rw.setFilename(filename);
    ri.setFilename(filename);
    superdata = new ArrayList<RecordedData>();
    SessionNotifier.getInstance().registerListener(this);
    FileInputStream srcreader;
    try {
      srcreader = new FileInputStream(file);
    } catch (FileNotFoundException e) { //the file was selected by the user, so this should not happen very often
      e.printStackTrace();
      return;
    }
    int guessSize = 0;
    InputStream is;
    if (iscompressed) {
      try {
        is = new GZIPInputStream(new BufferedInputStream(srcreader));
        guessSize = srcreader.available()*19; //The compression ratio is mostly somewhere around 20. (11...24)
      } catch (IOException e) {
        e.printStackTrace();
        return;
      }
    } else {
      is = new BufferedInputStream(srcreader);
      try {
				guessSize = is.available();
			} catch (IOException e) {
				//not important here
			}
    }
    if (understand(is, guessSize) == true) {
    	updateGFX();
      includeAll();
    } else
      new ShowMessage("Error: The the file "+file.toString()+" contains invalid data");
    try {
      is.close();
    } catch (IOException e) {
      e.printStackTrace();
    }
    markSaved();
  }
  
  public Session() {
    filename = new String("Unknown");
    dirname = ConfigFile.getInstance().getSettings("OpenPath");
    if (dirname == null)
    	dirname = new String();
    rw = new RecordWindow(this);
    ri = new RecordInfo();
    rw.setFilename(filename);
    ri.setFilename(filename);
    superdata = new ArrayList<RecordedData>();
    SessionNotifier.getInstance().registerListener(this);
    //add date of record created
    Date d = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd;HH:mm:ss");
		String recStart = sdf.format(d);
		//System.out.println("Session.init: Info: Date: " + recStart);
    addDataLine("D:"+recStart);
    includeAll();
  	updateGFX();
	}

	private boolean understand(InputStream is, int guessSize) {
    String line;
    ShowProgressbar progress = null;
    int readedbytes = 0;
    try {
      progress = new ShowProgressbar("Reading data", guessSize);
      while (is.available() > 0 ) {
        line = Tools.extractline(is);
        readedbytes += line.length();
        boolean success = addDataLine(line);
        if (!success) {
        	System.out.println("Session.understand: Error: Abort reading data");
        	progress.close();
        	return false;
        }
        progress.setProgress(readedbytes);
      }
      progress.close();
    } catch (IOException e) {
    	if (progress != null)
    		progress.close();
    	System.out.println("Session.understand: Error: Reading failed");
    	e.printStackTrace();
    	return false;
    }
    System.out.println("Session.understand: All data read");    
    return true;
  }
  
  public void printStats() {
  	System.out.println("Session.printStats: RecordedData sets "+superdata.size());
  	for (int i = 0; i < superdata.size(); i++) {
  		System.out.println("Session.printStats: "+superdata.get(i).toString()+" contains "+superdata.get(i).entries()+" entries");
  	}
  }

  public void messageTell(String str) {
    if (str.equals("newRecordSelections")) {
      includeSelectedAll();
    }
    if (str.startsWith("avRad")) {
      includeSelectedAll();
    }
    if (str.equals("autoaverage")) {
      includeSelectedAll();
    }
    if (str.equals("save")) {
      saveRecord(false, false);
    }
    if (str.equals("saveAs")) {
      saveRecord(false, true);
    }
    if (str.equals("export")) {
      saveRecord(true, true);
    }
    if (str.equals("close")) {
    	userWantsClose();
    }
    if (str.equals("stats")) {
    	ri.show();
    }
    if (str.startsWith("fd")) {
    	runDetector(str);
    }
    if (str.equals("smooth1")) {
    	smooth();
    }
  }
  
  private void runDetector(String str) {
  	//determine selected vector
  	int usefulindex = 0;
  	int entriesSel = 0;
    ChargingData cd = null;
    //calculate how many sets it will be
    for (int i = 0; i < superdata.size(); i++) {
      if (superdata.get(i).getClass().equals(OtherData.class) == false) {
      	if (rw.isSelectedRecord(usefulindex)) {
      		entriesSel += superdata.get(i).entries();
          if (superdata.get(i).getClass().equals(ChargingData.class)) {
          	cd = (ChargingData)superdata.get(i);
          }
      	}
        usefulindex++;
      } 
    }
    if (cd == null) {
    	System.out.println("Session.runDetector: Error: No ChargingData selected");
    	return;
    }
  	FullDetect fd = null;
  	if (str.equals("fd1")) {
  		fd = new NegDeltaU(cd);
  	}
  	if (str.equals("fd2")) {
  		fd = new NoDeltaDeltaU(cd);
  	}
  	if (str.equals("fd3")) {
  		fd = new NegDeltaUNoNoise(cd);
  	}
  	if (str.equals("fd4")) {
  		fd = new BadDetect(cd);
  	}
  	if (str.equals("fd5")) {
  		fd = new BadDetect2(cd);
  	}
  	if (str.equals("fd6")) {
  		fd = new NegDeltaUFinal(cd);
  	}
  	if (str.equals("fd7")) {
  		fd = new NegDeltaUFinal2(cd);
  	}
  	if (fd != null) {
  		System.out.println("Session.runDetector: Would stop after: "+Tools.getTimeFromSamples(fd.getStopTime()));
  		rw.setLine(fd.getStopTime(), entriesSel);
  		rw.redraw();
  	}
	}

  public void smooth() {
  	//run smooth on all selected
  	int usefulindex = 0;
    for (int i = 0; i < superdata.size(); i++) {
      if (superdata.get(i).getClass().equals(OtherData.class) == false) {
      	if (rw.isSelectedRecord(usefulindex)) {
          ((UsefulData)superdata.get(i)).smoothValues(5000);
      	}
        usefulindex++;
      } 
    }
    includeSelectedAll();
		rw.redraw();    
  }
  
	public void updateWindowRecodList() {
    Vector<String> v = new Vector<String>();
    for (int i = 0; i < superdata.size(); i++) {
      if (superdata.get(i).getClass().equals(OtherData.class) == false) {
        v.add(superdata.get(i).getDescription());
      }
      
    }
    rw.setRecordedData(v);
  }
  
  private void includeAll() {
    rw.selectAll(); //will fire an event to fetch the display data too
  }
  
  private void includeSelectedAll() {
  	int usefulindex = 0;
    int originDataSets = 0;
    //calculate how many sets it will be
    for (int i = 0; i < superdata.size(); i++) {
      if (superdata.get(i).getClass().equals(OtherData.class) == false) {
      	if (rw.isSelectedRecord(usefulindex)) {
      	  originDataSets += ((UsefulData)superdata.get(i)).size();
      	}
        usefulindex++;
      } 
    }
    //now add to vector
    Vector<DataEntry> all = new Vector<DataEntry>();
    int averagefactor = rw.getUpdatedAvgFactor(originDataSets);
    usefulindex = 0;
    for (int i = 0; i < superdata.size(); i++) {
      if (superdata.get(i).getClass().equals(OtherData.class) == false) {
        if (rw.isSelectedRecord(usefulindex)) {
            UsefulData ud = (UsefulData)superdata.get(i);
            ud.addToVector(all, averagefactor);
        }
        usefulindex++;
      } 
    }
    rw.setDisplayData(all);
  }
  
  private void saveRecord(boolean export, boolean newName) {
    //determine filename to use
    File file;
    boolean chname = false; //change name of window title
    if (export) {
      //always select a name and do not alter internal name
    	String suggested = Tools.replaceFileEnding(filename, ".txt");    	
      ChooseFile cf = new ChooseFile(".txt", ChooseFile.SAVE, dirname, suggested);
      if (cf.getFile() == null) { //user selected abort
      	return;
      }
      file = cf.getFile();
    } else { //a normal compresses save
      //if newname or current name is invalid, select a new name and use afterwards
      if ((newName) || (!filename.endsWith(".crf"))) {
      	String suggested = Tools.replaceFileEnding(filename, ".crf");
        ChooseFile cf = new ChooseFile(".crf", ChooseFile.SAVE, dirname, suggested);
        if (cf.getFile() == null) { //user selected abort
        	return;
        }
        //Add .crf if the name is without an ending
        file = new File(Tools.addIfWithoutFileEnding(cf.getFilename(), ".crf")); //may change the extension without the user knowing
        if (file.exists()) { //do not automatically overwrite existing files -> use the name the user explicitly wanted
        	file = cf.getFile(); //back to user selected file (where the user knows that he is overwriting if the file exists)
        }
        chname = true;
      } else
        file = new File(dirname);
    }
    System.out.println("Session.saveRecord: Using filename: "+file);
    //open the needed streams
    FileOutputStream srcwriter;
    try {
      srcwriter = new FileOutputStream(file);
    } catch (FileNotFoundException e) {  //we want to create the file anyway, so this will not happen
      e.printStackTrace();
      return;
    }
    OutputStream os;
    if (!export) { 
      try {
        os = new GZIPOutputStream(srcwriter);
      } catch (IOException e) {
        e.printStackTrace();
        return;
      }
    } else
      os = new BufferedOutputStream(srcwriter);
    //write to the stream
    for (int i = 0; i < superdata.size(); i++) {
      try {
        superdata.get(i).pushToStream(os);
      } catch (IOException e) {
        e.printStackTrace();
        System.out.println("Session.saveRecord: Error: Could not save or export Recorded Data: '"+superdata.get(i).getDescription()+"'. Continiuing with the rest");
      }
    }
    //save possible not handeled line from addDataLine
    if (bufline != null) {
			try {
				os.write((bufline+Tools.newline).getBytes());
			} catch (IOException e) {
				e.printStackTrace();
				System.out.println("Session.saveRecord: Error: Could not save or export Recorded Data: '"+bufline+"'. Continuing with the rest");
			}
    }
    //add commentary information
    try {
  		String str = ri.getUserComment();
  		str = str.replaceAll("\r", "");		
  		str = "C:"+str.replaceAll("\n", Tools.newline+"C:");
  		os.write(str.getBytes());
		} catch (IOException e) {
			e.printStackTrace();
			System.out.println("Session.saveRecord: Error: Could not save or export User Comment in Recorded Data. Continuing with the rest");
		} catch (NullPointerException e) {
			e.printStackTrace();
			System.out.println("Session.saveRecord: Error: Could not save or export User Comment (Nullpointer exception) in Recorded Data. Continuing with the rest");
		}
    //close the streams
    try {
      os.close();
      if (!export) { //a normal save
      	markSaved();
      	ri.markSaved();
      }
    } catch (IOException e) {
      e.printStackTrace();
    }
    //change the title
    if (chname) {
      filename = file.getName();
      dirname = file.getPath();
      rw.setFilename(filename);
      ri.setFilename(filename);
    } 
  }

	public void updateGFX() {
		updateWindowRecodList();
		updateStatText();
	}
	
  private int mode = -1; //-1 unknown. 0 = Other, 2 = Charging, 3 = Discharging
  private int newmode = 0; //start with 'Other' something
  private String bufline = null;
  private RecordedData de = null;
  private boolean newobject = false;  
  /**
   * This function is some sort of state machine, the states are the five variable above
   * @param data
   * @return true if all worked well, false on an error
   */
	public boolean addDataLine(String data) {
    String line = data;
		line = line.trim();
		//determine what todo
		if (line.length() > 0) {
			markModified();
			if (line.startsWith("C:")) { //this is a comment, extract from normal superdata and add to RecordInfo
				addUserCommentLine(line.substring(2, line.length()));
				return true;
			}
		  if (line.length() > 2) {
		    if (line.startsWith("B:")) {
		      newobject = true;
		      bufline = line;
		      return true;
		    } else if (line.startsWith("I:2")) {
		      newmode = 2;
		    } else if (line.startsWith("I:3")) {
		      newmode = 3;
		    } else if (line.startsWith("T:") == false) //Terminate strings should not change something
		      newmode = 0;
		  } else
		    newmode = 0;
		  //now do what to do
		  if (mode != newmode) {
		    newobject = true;
		  }
		  if (newobject) {
		    if (newmode == 2) {
		      de = new ChargingData(bufline);
		    } else if (newmode == 3) {
		      de = new DischargingData(bufline);
		    } else {
		      de = new OtherData(bufline);
		    }
		    superdata.add(de);
		    newobject = false;
		  }
		  bufline = null;
		  if (de != null) {
		    if (de.add(line) == false) { //add data line to data class
		      System.out.println("Session.addDataLine: Error: Adding data to RecordData failed.");
		      return false;							
		    }
		  } else {
		    System.out.println("Session.addDataLine: Error: No RecordedData type selected.");
		    return false;
		  }
		  mode = newmode;
		}  
    return true;
	}
	
	
	private void addUserCommentLine(String str) {
		if (userComment == null) {
			userComment = str;
		} else
			userComment += Tools.newline+str;
		ri.setUserComment(userComment);
	}
	
	
	private void updateStatText() {
		String stats = new String();
		int unclaimedData = 0;
		for (int i = 0; i < superdata.size(); i++) {
			if (superdata.get(i).getClass().equals(OtherData.class)) {
				OtherData od = (OtherData)superdata.get(i);
				unclaimedData += od.entries();
				//search for start date in superdata
				String temp = od.getByBeginning("D:");
				if (temp != null) {
					stats += "Record created on: "+temp+Tools.newline+Tools.newline;
					unclaimedData--; //one less because the 'D:' line is used 
				}		
				//search for errors 
				temp = od.getByBeginning("E:");
				if (temp != null) {
					stats += "Charger reset by: "+convertErrorToDescription(temp)+
					          Tools.newline+Tools.newline;
					unclaimedData--; //one less because the 'D:' line is used 
				}
			}
			//give a description for every non OtherData in superdata
			if (superdata.get(i) instanceof UsefulData) {
				UsefulData ud = (UsefulData)superdata.get(i);
				stats += ud.getAdvancedDescription()+Tools.newline+Tools.newline;
			}			
		}
		//show unclaimed data
		stats += "Unidentified data lines in Record: "+unclaimedData;
		//update
		ri.setStatisticText(stats);
	}
	
	
	private String convertErrorToDescription(String error) {
		if (error.startsWith(" B")) {
			return "Brown out detector (low voltage)";
		}
		String lasttask = "Unknown";
		if (error.length() >= 4) {
			lasttask = error.substring(3);
		}
		if (error.startsWith(" J")) {
			return "Illegal jump most likely caused by Task: "+lasttask; 
		}
		if (error.startsWith(" W")) {
			return "Watchdog hit. Last running Task was: "+lasttask; 
		}
		if (error.startsWith(" S")) {
			return "Stack corruption of task: "+lasttask+" meaning the task running BEFORE this task was most likely responsible"; 
		}		
		return "Unknown";
	}

 
	public void markModified() {
		modified = true;
	}
	
	private void markSaved() {
		modified = false;
	}
	
	public boolean isModified() {
		return (modified | ri.isModified());
	}
	
	private void userWantsClose() {
		ri.setInvisible();
		if (isModified()) { //ask the user if he wants to save
			ShowQuestion sq = new ShowQuestion("Record has been modified", "The last changes of the record\n\r"+
					filename+"\n\rwere not saved. Save before closing?");
			if (sq.agreed()) {
				saveRecord(false, false);
			}
		}
    SessionNotifier.getInstance().removeListener(this);
		superdata = null;
    rw.destroy();
    rw = null;
    if (SerialToSession.getInstance().getActiveSession() == this)
    	SerialToSession.getInstance().stop(); //so that the next time a new session is created
    System.gc();
	}
	
}
