package control;

/* Charge Control Version 1.1 (c) 2008 by Malte Marwedel
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

import java.awt.EventQueue;
import java.io.File;
import java.util.Timer;
import java.util.TimerTask;

import data.SessionNotifier;

import model.ReceiveMessage;
import view.ChooseFile;
import view.Prime;
import view.ShowMessage;

public class Main implements ReceiveMessage {

	public static final boolean DEBUGMODE = false; //tells some things which go wrong
	public static final boolean DEVELMODE = false; //allows some advanced functions
	
  private Prime primeview;
  private SerialIO sio;
  private String openPath;
  private boolean workingSerialLib = true;
  
  
  public static final String info = "Charge Control version 1.1 (c) 2008 by Malte Marwedel"+Tools.newline+Tools.newline+
                "This program is free software; you can redistribute it and/or modify"+Tools.newline+
                "it under the terms of the GNU General Public License as published by"+Tools.newline+
                "the Free Software Foundation; either version 2 of the License, or"+Tools.newline+
                "(at your option) any later version."+Tools.newline+Tools.newline+
                "This program is distributed in the hope that it will be useful,"+Tools.newline+
                "but WITHOUT ANY WARRANTY; without even the implied warranty of"+Tools.newline+
                "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"+Tools.newline+
                "GNU General Public License for more details."+Tools.newline+Tools.newline+
                "You should have received a copy of the GNU General Public License"+Tools.newline+
                "along with this program; if not, write to the Free Software"+Tools.newline+
                "Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA";
  
  /**
   * @param args Give a compresses record as first parameter and the record will be opened
   */
  public static void main(String[] args) {
    System.out.println(Main.info);
    System.out.println();
    System.out.println("DEBUGMODE is: "+DEBUGMODE);
    System.out.println("DEVELMODE is: "+DEVELMODE);
    
    final String[] argsf = args;
    /*The invokeLater is needed, because otherwise there are minor problems with the GUI
    if record files given as parameters are opened*/
    final Runnable startUp = new Runnable() {
      public void run() {
      	new Main(argsf);
      }
    };
    EventQueue.invokeLater(startUp);
  }
  


  private Main(String[] args) {
    primeview = new Prime(this);
    primeview.createAndShowGUI();
    sio = SerialIO.getInstance();
    try {
     
      primeview.setAvariablePorts(sio.getAvariableSerialPorts());
    } catch (java.lang.UnsatisfiedLinkError e) {
    	if (Main.DEBUGMODE)
    		e.printStackTrace();
      workingSerialLib = false;
      primeview.disablePortButtons();
      new ShowMessage("Error: I/O Library is not correctly installed. Continuing without support for a serial port. \n\r"+
          "For i386 Linux: Copy librxtxSerial.so to (JRE installation dir)/jre/lib/i386/\n\r"+
          "For other operating systems (Windows, Mac, Solaris) and architectures please see INSTALL file of RXTX lib.");
    } catch (java.lang.NoClassDefFoundError e) {
    	if (Main.DEBUGMODE)
    		e.printStackTrace();
    	workingSerialLib = false;
    	primeview.disablePortButtons();
    	new ShowMessage("Error: Missing classes (jar file) for the serial communication.\n\r"+
    			"Continuing without support for a serial port.");
    }
    //set up default path if possible
    openPath = ConfigFile.getInstance().getSettings("OpenPath");
    //open if parameter is a .crf file
    if (args.length > 0) {
    	int files;
    	if (Main.DEVELMODE) { //allow opening multiple files
    		files = args.length;
    	} else
    		files = 1;
    	for (int i = 0; i < files; i++) {
    		if (args[i].endsWith(".crf")) {
    			File f = new File(args[i]);
    			if (f.isFile() && f.canRead()) {
    				new Session(f, true);
    			} else {
    				new ShowMessage("Error: Could not open file: "+args[i]);
    			}
      	}
    	}
    	
    }
    //let start the 'look' thread
    Timer timer = new Timer();
    timer.schedule(updateGUIandInternalsTask, 500, 500);
  }
  
  public void messageTell(String str) {
    //System.out.println("got: "+str);
    if (str.equals("recordimport")) {
      ChooseFile cf = new ChooseFile(".txt", ChooseFile.OPEN, openPath, null);
      //System.out.println("Main.messageTell: Record import filename: "+cf.getFilename());
      if (cf.getFile() != null) {
      	new Session(cf.getFile(), false);
        openPath = cf.getDirectory();
        ConfigFile.getInstance().setSettings("OpenPath",openPath);
      }
    } else if (str.equals("recordopen")) {
      ChooseFile cf = new ChooseFile(".crf", ChooseFile.OPEN, openPath, null); //crf = charge record file
      if (cf.getFile() != null) {
      	new Session(cf.getFile(), true);
      	openPath = cf.getDirectory();
        ConfigFile.getInstance().setSettings("OpenPath",openPath);
      }
    } else if (str.equals("calibrate")) {
			new ShowMessage("Warning: Please note, that wrong calibration values may result in an overloading of the charger and/or the battery,\n\r" +
					"THIS MAY CAUSE PERMANENT DAMAGE TO THE CHARGER AND/OR THE BATTERY.");
      Calibration.getInstance().show();
    } else if (str.equals("portopenclose")) {
    	handlePortopenclose();
    } else if (str.equals("appexit")) { //close open serial port (if open) on application exit
    	if (sio.getPortState() == SerialIO.PORT_OPEN) {
    		sio.portClose(); //now close it
    	}
    	SessionNotifier.getInstance().tellExit();
    }
  }

  public boolean isWorkingSerialLib() {
    return workingSerialLib;
  }
  
  private void handlePortopenclose() {
  	if (sio.getPortState() == SerialIO.PORT_CLOSED) {
  		//now open it
  		sio.portOpen(primeview.getSelectedPort());
  	} else if (sio.getPortState() == SerialIO.PORT_OPEN) {
  		//now close it
  		Calibration.getInstance().disableSerialComm();
  		sio.portClose();
  	}
  	//re-check current state (since opening or closing may have failed) for update in button description
  	switch(sio.getPortState()) {
  		case SerialIO.PORT_CLOSED: primeview.portClosed(); break;
  		case SerialIO.PORT_OPEN: primeview.portOpened(); break;
  	}	
  }
  
  TimerTask updateGUIandInternalsTask = new TimerTask() {
    public void run() {
    	EventQueue.invokeLater(doLookforSerial);
    }
  };
  
  final Runnable doLookforSerial = new Runnable() {
    public void run() {
    	//update main window
    	primeview.updateChargerState(SerialThread.getInstance().getChargerstateDescription());
    	//update buttons in CalibrationWindow
    	int chargerstate = SerialThread.getInstance().getChargerstate();
    	if (((chargerstate == SerialThread.CHARGER_IDLE) || (chargerstate == SerialThread.CHARGER_UNKNOWN)) &&
    			(sio.getPortState() == SerialIO.PORT_OPEN)) {
    		Calibration.getInstance().enableSerialComm();
    	} else
    		Calibration.getInstance().disableSerialComm();
    	//let look serial to session into the fifo
    	SerialToSession.getInstance().lookIntoFIFO();
    }
  };
  

}