/*3D Scanner
  Version 1.0
  (c) 2005, 2008, 2009 by Malte Marwedel
  www.marwedels.de/malte

  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


  Pin connection:
  PINB.0 = Output, LCD RS
  PINB.1 = Output, LCD E
  PINB.2 = Output, LCD D7
  PINB.3 = Output, LCD D6
  PINB.4 = Output, LCD D5
  PINB.5 = Output, LCD D4
  PINB.6 = n/a
  PINB.7 = n/a
  PINC.0 = Input, Distance sensor
  PINC.1 = Input, measure current of servos
  PINC.2 = Input, Key
  PINC.3 = Input, not used
  PINC.4 = Input, External Input 2
  PINC.5 = Input, External Input 1
  PINC.6 = n/a
  PINC.7 = n/a
  PIND.0 = Input, RS232 RX
  PIND.1 = Output, RS232 TX
  PIND.2 = Input, not used
  PIND.3 = Input, not used
  PIND.4 = Input, not used
  PIND.5 = Output, Servo vertical (Y)
  PIND.6 = Output, Servo horizontal (X)
  PIND.7 = Input, not used

md5sum of the compiled .hex file (avr-gcc: 4.2.2):
(the current date string 2009-02-06 is compiled in):
e162843989a7ff497b61dbf9ea3b91cc
*/

#include "main.h"

//Global variables, within this file

//The three variables describe the mode of operation

u08 get_request;
#define GET_NONE 0
#define GET_DIRECT 1
#define GET_LINEAR 2
#define GET_PLANE 3
#define GET_INPUT1 4
#define GET_INPUT2 5


/*
2 = always send
1 = always send if servos are idle
0 = send a single request if servos are in its final position
*/
u08 send_continuus;

/*
minimum time for which the servos defined as moving, after their position
changed.
*/
u08 volatile reading_blocking; //gets decremented by timer1 every 8ms
//approx 80ms
#define START_BLOCKING 10


#define YES 1
#define NO 0

//State variables for the LCD to display content

typedef struct datapackage {
	u08 sensor_request;
	u16 sensor_value;
	u08 alpha;
	u08 beta;
	u08 servo_idle;
} datapackage_t;

/*
RS232 Commands, have to end with a carriage return (13):
Hxx : Set horizontal position (number in hex)
Vxx : Set vertical position(number in hex)
G00 : Abfrage aufheben, falls noch nicht bearbeitet oder wie bei
      send_continuus != 0
G01 : Send direct distance sensor value
G02 : Send linearized distance sensor value
G03 : Send distance as vector
G04 : Send value from input 1
G05 : Send value from input 2
C00 : Send whenever a g0X is recived
C01 : Send always as long as the servos are idle
C02 : Send always

RS232 Answer:
G:g;V:xxxxx; Sends the current Mode m and the Value: xxxxx
G:g;X:xxxxx;xxxxx;xxxxx; If g=3
*/


/*
converts the non-linear sensor value into centimeters with the help of a table
*/
u16 conv_in_cm (u16 sensorvalue) {
	u16 centimeters;
	//If within the allowed range
	if ((sensorvalue <= (500+83)) && (sensorvalue >= 83)) {
		sensorvalue = 500 - (sensorvalue -83);
		centimeters = pgm_read_byte_near(&convtable[sensorvalue]);
	} else {
		centimeters = 0;
	}
	return centimeters;
}


u16 conv_in_mm(u16 sensorvalue) {
	return  conv_in_cm(sensorvalue)*10;
}

const char unknown1[] PROGMEM = "E:CmdUnknown;";
const char fifoe1[] PROGMEM = "E:FIFOFree;\r";

/*
Prints an error message if the received command is not valid
*/
void cmd_error(u08 * rec, u08 size) {
	print_p(DEV_UART, unknown1);
	u08 j;
	for (j = 0; j < size; j++) {
		print_word(DEV_UART, rec[j], 3);
		if ((rec[j] >= '!') && (rec[j] <= '~')) {
			//printable ascii
			print_char(DEV_UART, ':');
			print_char(DEV_UART, rec[j]);
		}
		print_char(DEV_UART, (u08)';');
	}
	print_char(DEV_UART, (u08)'\r');
}

/*
  Decodes the commands, coming from the RS232 port
*/
void analyse_income_cmd(void) {
	u08 uart_emp_local[4];
	u08 i;
	u08 in_number = 0;
	//Look if there are at least 4 bytes in the RX FIFO
	if (uart_rx_available() < 4) {
		return;
	}
	for (i = 0; i < 4; i++) {
		if (uart_rx_available() == 0) { //may not happen
			print_p(DEV_UART, fifoe1);
			return;
		}
		uart_emp_local[i] = uart_rx_get();
		if ((uart_emp_local[i] == 13) && (i < 3)) {
			cmd_error(uart_emp_local, i+1);
			return;
		}
	}
	if (uart_emp_local[3] == 13) { //if carriage return, start evaluation
		//decode the hexadecimal parameter
		for (i = 1; i < 3; i++) {
			in_number *= 16;
			if ((uart_emp_local[i] >= '0') && (uart_emp_local[i] <= '9')) {
				in_number += uart_emp_local[i]-'0';
			} else
			if ((uart_emp_local[i] >= 'a') && (uart_emp_local[i] <= 'f')) {
				in_number += (uart_emp_local[i]-'a')+10;
			} else
			if ((uart_emp_local[i] >= 'A') && (uart_emp_local[i] <= 'F')) {
				in_number += (uart_emp_local[i]-'A')+10;
			} else {
				cmd_error(uart_emp_local, 4);
				return;
			}
		}
		//act, depending on the given command
		if (uart_emp_local[0] == 'H') {
			if (servo1 != in_number) {
				servo1 = in_number;
				reading_blocking = START_BLOCKING;
			}
		} else
		if (uart_emp_local[0] == 'V') {
			if (servo2 != in_number) {
				servo2 = in_number;
				reading_blocking = START_BLOCKING;
			}
		} else
		if ((uart_emp_local[0] == 'G') && (in_number < 6)) {
			get_request = in_number;
		} else
		if ((uart_emp_local[0] == 'C') && (in_number < 3)) {
			send_continuus = in_number;
		} else {
			cmd_error(uart_emp_local, 4);
		}
	} else
		cmd_error(uart_emp_local, 4);
}

/*
returns YES if the servos consume low power, meaning they are not moving
Otherwise NO is returned.
*/
u08 servo_power_watch(void) {
  static u16 servo_power = 0; //the power of the servos
	u16 currpower;
	//print("S\n");
	//a large number means lower power consumption
	currpower = sample_ad(SERVO_RESISTOR_PIN, 8);
	if (currpower > servo_power) {
		servo_power = currpower;
	}
	/* servo_power will always have a relative high value and wont overflow if
	  12 is substracted. And even if it does, then there is something wrong with
	  the circuit.
	*/
	if ((currpower > (servo_power - 12)) &&  (reading_blocking == 0)){
		//less than approx 45mA higher than the minimum consumption
		return YES; //not moving
	} else {
		return NO; //moving
	}
}

/* Experimentally measured, the servos have a moving range of 60° from the
middle position in each direction */

double calc_angle(double servopos) {
	double angle;
	angle = ((double)M_PI*120)/((double)180*(double)255)*servopos-(M_PI*60/180);
	return angle;
}

void calc_vec(u08 s1, u08 s2, u16 sensorvalue, double * vec) {
	double angle[2], k;
	/* Input: s1: servo1, s2: servo2, sensorvalue.
	   The result will be stored in * vec,
	   which must be a vector of at least three elements (x, y, z).

	Here we need a little bit math :-D
	We expect to be the servo position 128;128 to have a angle of 90°
	against the surface
	alpha:
	0x01 = -70°
	0x7f = 00°
	0xff = 70°

	alpha1 = 140°/254*servo1-70
	alpha2 = ((PI*140°)/(180°*254))*servo1-((PI*70°/180°))
	beta1 = 140°/254*servo2-70
	beta2 = ((PI*140°)/(180°*254))*servo2-((PI*70°/180°))
	we expect z= 1. Because (tan alpha)=(x/z) and (tan beta)=(y/z),
	we wrote this as vector v (x;y;z) (tan alpha;tan beta;1).
	From this we make the einheitsvektor (Length 1)
	k = sqrt ((tan alpha)^2+(tan beta)^2+1)
	v1 = 1/k*v = ((tan alpha)/k;(tan beta)/k;1/k)
	Test it: The sum of v1 has now to be 1
	Now we multiply v1 with l(sensor value). As result we get a vector va,
	in which the x, y und z coordinates give us the point in the space
	where the sensor reflected.
	In the calculation below is:
	x = vec[0]; y = vec[1]; z = vec[2]; alpha2 = angle[0]; beta2 = angle[1];
	This helps to make the code more compact by using loops.

	So, it took me two days to come to the conclusion to use vector calculations,
	In the two days I became nearly crazy because I did not found a soultion. ;-)
	Then, ending up with the calculation above, programming and testing it, took
	only two hours. Sometimes the solution is simpler than you could think of *g*

	In memorian to my math teacher Ms Günther, without her, I would have never
	learned vector calcualtions. She died three months after I finished school.
	:'-(
	*/
	angle[0] = s1; //Make a single loop for servo1 and servo2
	angle[1] = s2;
	u08 i;
	for (i = 0; i < 2; i++) {
		angle[i] = calc_angle(angle[i]);
		vec[i] = tan(angle[i]);
	}
	k = sqrt(square(vec[0])+square(vec[1])+1);
	vec[2] = 1;
	for (i = 0; i < 3; i++) {
		//the loop saves program memory, approx 150 bytes
		vec[i] /= k; //calculate normal vector
		vec[i] *= sensorvalue; //multiply with measured length
		vec[i] += 0.5; //for correct rounding
	}
}

const char answerhead[] PROGMEM = "G:$1;X:";
/*
Generates the answer depending on the mode setted by the rs232 commands
the data are stored in the lastdata struct. They are not, read in this
function.
Returns: 1 if the given request could not be handled (because the servos are
moving or simmilar reasons)
0: request was handled.
*/
u08 do_request(u08 servo_idle, datapackage_t * lastdata) {
	u16 sensorvalue;
	if (((servo_idle == YES) || (send_continuus == 2))
	    && (get_request != GET_NONE)) {
		printw_p(DEV_UART, answerhead, get_request, 0);
		if ((get_request != GET_INPUT1) && (get_request != GET_INPUT2)) {
			sensorvalue = sample_ad(DISTANCE_AD_PIN, 32);
		} else if (get_request == GET_INPUT1) {
			sensorvalue = sample_ad(EXTIN1_PIN, 32);
		} else
			sensorvalue = sample_ad(EXTIN2_PIN, 32);
		if ((get_request == GET_LINEAR) || (get_request == GET_PLANE)) {
			sensorvalue = conv_in_mm(sensorvalue);
			//sensorvalue = 123; //for testing only
		}
		lastdata->sensor_request = get_request;
		lastdata->sensor_value = sensorvalue;
		lastdata->alpha = servo1;
		lastdata->beta = servo2;
		lastdata->servo_idle = servo_idle;
		if (get_request == GET_PLANE) {
			double vec[3];
			calc_vec(servo1, servo2, sensorvalue, vec);
			u08 i;
			for (i = 0; i < 3; i++) {
				print_sword(DEV_UART, (s16)vec[i], 5);
				print_char(DEV_UART, (u08)';');
			}
		} else {
			print_word(DEV_UART, sensorvalue, 5);
		}
		print_char(DEV_UART, (u08)'\r'); //carriage return
		if (send_continuus == 0) {
			get_request = GET_NONE;
		}
	}
	if ((servo_idle == NO) && (get_request != GET_NONE) && (send_continuus == 0)) {
		return 1;
	}
	return 0;
}

const char lcd1[] PROGMEM = "\rG:$1 V:$4";
const char lcd2[] PROGMEM = "      ";
const char lcd3[] PROGMEM = "mm    ";
const char lcd4[] PROGMEM = "\nA:$3 B:$3     ";
const char lcd5[] PROGMEM = "\nA:`4"DEGREE_SYMB" B:`4"DEGREE_SYMB" ";
const char lcd6[] PROGMEM = "\rG:$1 Vxyz [cm]:  ";

void update_lcd(datapackage_t * lastdata) {
	static u08 dispmode = 0;
	/* Modes for dispmode:
	   0: Display mode and values from lastdata
	      G:g V:vvvvEE
	      H:aaa V:bbb
	   1: Display angles instead of A and b
	      G:g V:vvvvEE
	      H:aaaa° V:bbbb°
	   2: Displays vector
	      G:g Vxyz:
	*/
	if (lastdata->sensor_request == GET_NONE) {
		return;
	}
	if (keys_pressed() == 1) {
		dispmode++;
		if (dispmode > 2)
			dispmode = 0;
	}
	if ((dispmode == 2) && ((lastdata->sensor_request == GET_LINEAR) ||
	    (lastdata->sensor_request == GET_PLANE))) {
		printw_p(DEV_LCD, lcd6, lastdata->sensor_request, 0);
		double vec[3];
		calc_vec(lastdata->alpha, lastdata->beta, lastdata->sensor_value, vec);
		print_char(DEV_LCD, (u08)'\n');
		u08 i;
		for (i = 0; i < 3; i++) {
			print_sword(DEV_LCD, vec[i]/10, 4);
			print_char(DEV_LCD, (u08)';');
		}
	} else {
		printw_p(DEV_LCD, lcd1, lastdata->sensor_request, lastdata->sensor_value);
		if ((lastdata->sensor_request == GET_LINEAR) ||
		    (lastdata->sensor_request == GET_PLANE)) {
			print_p(DEV_LCD, lcd3);
		} else {
			print_p(DEV_LCD, lcd2);
		}
		if (dispmode == 1) {
			double alpha = calc_angle(lastdata->alpha)*180.0/M_PI;
			double beta = calc_angle(lastdata->beta)*180.0/M_PI;
			printw_p(DEV_LCD, lcd5, (s16)alpha, (s16)beta);
		} else {
			printw_p(DEV_LCD, lcd4, lastdata->alpha, lastdata->beta);
		}
	}
}

const char uarterrorstr[] PROGMEM = "E:Uart;$3\r";

void error_handling(void) {
	u08 uerr = uart_error_get();
	if (uerr != 0) {
		printw_p(DEV_UART, uarterrorstr, uerr, 0);
	}
}

#ifndef BUILD_DATE
	#define BUILD_DATE "Unknown date"
	#warning "BUILD_DATE not set in CFLAGS"
#endif

const char splash[] PROGMEM = "\rI:3D Scanner 1.0\r";
const char bdate[] PROGMEM = "\nBuild:"BUILD_DATE;

const char rwd[] PROGMEM = "\rR:Watchdog\n";
const char rbrown[] PROGMEM = "\rR:Brownout\n";
const char rext[] PROGMEM = "\rR:External\n";
const char rpower[] PROGMEM = "\rR:PowerOn\n";

int main(void) {
	datapackage_t lastdata = {0, 0, 0.0, 0.0, NO};
	u08 resetsource = MCUCSR;
	MCUCSR = 0;
	DDRB = 0x3F;  //LCD output
	DDRC = 0; //Analoge inputs
	PORTC = 0x04; //pull-up for key
	DDRD = 0x62; //Servo output and RS232 RX/TX
	wdt_reset();
	wdt_enable(WDTO_2S);
	lcd_init(); //as sideeffect: provides starup delay
	uart_init();
	ad_init(7);
	keys_init();
	sei(); //Enable interrupts
	//print interrupt source (needs sei() uart_init())
	if (resetsource & (1<<WDRF)) {
		print_p(DEV_UART, rwd);
	}
	if (resetsource & (1<<BORF)) {
		print_p(DEV_UART, rbrown);
	}
	if (resetsource & (1<<EXTRF)) {
		print_p(DEV_UART, rext);
	}
	if (resetsource & (1<<PORF)) {
		print_p(DEV_UART, rpower);
	}
	//show some messages
	print_p(DEV_LCD, splash);
	print_p(DEV_LCD, bdate);
	print_p(DEV_UART, splash);
	_delay_ms(400.0);
	//enable the servos (may be the source of a reset)
	wdt_reset();
	wdt_enable(WDTO_250MS);
	servo_init();
	u08 waiting = 0;
	for (;;) { //Main loop
		wdt_reset();
		if (!waiting) {
			analyse_income_cmd();
		}
		u08 servo_idle = servo_power_watch();
		waiting = do_request(servo_idle, &lastdata); //only writes lastdata
		update_lcd(&lastdata); //only reads lastdata
		error_handling();
	}
}
