package control;

/* 3D Scanner Control Version 1.0 (c) 2009 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import gnu.io.CommPort;

import java.awt.EventQueue;
import java.io.IOException;
import java.util.Vector;

/**
 * This is a thread for communicating with the serial cable.
 * It provides a status value for the charger and all incoming bytes are placed into a line based FIFO
 * Moreover commands for reading and writing the configuration are provided
 * 
 * This class implements the singleton pattern
 * 
 * @author Malte Marwedel
 *
 */

public class SerialThread implements Runnable {
	
	private static SerialThread st;
	
	private final int POLL_FREQ_MS = 10;

	private SerialIO sio;
	
	private volatile Vector<String> rxfifo;
	private volatile Vector<String> txfifo;

	private volatile boolean shouldRun = true;
	private volatile boolean iamRunning = false;

	private Thread sthr;
	
	private volatile CommPort commport;
	
	public static SerialThread getInstance(SerialIO s) {
		if (st == null) {
			st = new SerialThread(s);
		}
		return st;
	}
	
	private SerialThread(SerialIO s) {
		sio = s;
		rxfifo = new Vector<String>();
		txfifo = new Vector<String>();
	}
	
	public void run() {
		iamRunning = true;
		String accumu = new String();
		while(shouldRun) {
			int avar;
			try {
				avar = commport.getInputStream().available();
				//System.out.println("Available on input "+avar);
				if (avar > 0) {
					byte arr[] = new byte[avar];
					commport.getInputStream().read(arr, 0, avar);
					String buff = new String(arr);
					buff = buff.replaceAll("\n", "");
					buff = buff.replaceAll("\r", "\n");
					//System.out.println("Raw: '"+temp+"'"+"newline at: "+temp.indexOf('\n'));
					int k = 0;
					int l;
					while ((l = buff.indexOf('\n', k)) >= 0) { //for every newline do
						accumu += buff.substring(k, l);
						//System.out.println("add to fifo: '"+accumu+"'");
						putRxFifo(accumu);
						accumu = new String();
						k = l+1;
					}
					if (k != buff.length()) { //if read does not end with a \n 
						accumu += buff.substring(k, buff.length()); //safe the rest for next round
					}
				}
				//try to send if something in the fifo
				while (getTxFifoSize() > 0) {
					String cmd = getTxFifo();
					//System.out.println("Write: '"+cmd+"'");
					commport.getOutputStream().write(cmd.getBytes());
				}
				
			} catch (IOException e) {
				avar = 0;
				System.out.println("Serial.Thread.run: Error: "+e.getMessage());
			}
			try {
	      Thread.sleep(POLL_FREQ_MS);
      } catch (InterruptedException e) {
	      //who cares
      }
		}
		if (accumu.length() > 0) {
			putRxFifo(accumu); //add possible unfinished strings
		}
		iamRunning = false;
	}
	
	public void start(CommPort cp) {
		commport = cp;
		shouldRun = true;
		sthr = new Thread(this);
		sthr.start();
		while(!iamRunning) {			
			try {
	      Thread.sleep(1);
      } catch (InterruptedException e) {
	      //who cares
      }
		}
	}
	
	public void stop() {
		shouldRun = false;
		while(iamRunning) {
			try {
	      Thread.sleep(1);
      } catch (InterruptedException e) {
	      //who cares
      }
		}
	}
  
  final Runnable doLookforSerial = new Runnable() {
    public void run() {
    	sio.rxDataWaiting();
    }
  };
	
	public synchronized int getRxFifoSize() {
		return rxfifo.size();
	}
	
	private synchronized void putRxFifo(String str) {
		rxfifo.add(str);
		EventQueue.invokeLater(doLookforSerial);
	}
	
	public synchronized String getRxFifo() {
		if (rxfifo.size() > 0) {
			String str = rxfifo.get(0);
			rxfifo.remove(0);
			return str;
		}
		System.out.println("SerialThread.getRxFifo: Warning: Tried to read from FIFO, but FIFO is empty");
		return null;
	}

	public synchronized int getTxFifoSize() {
		return txfifo.size();
	}
	
	public synchronized void putTxFifo(String str) {
		txfifo.add(str);
	}
	
	public synchronized String getTxFifo() {
		if (txfifo.size() > 0) {
			String str = txfifo.get(0);
			txfifo.remove(0);
			return str;
		}
		System.out.println("SerialThread.getTxFifo: Warning: Tried to read from FIFO, but FIFO is empty");
		return null;
	}
	
	
}
