package control;

/* 3D Scanner Control Version 1.0 (c) 2009 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import java.awt.EventQueue;
import java.util.Timer;
import java.util.TimerTask;

import view.ShowMessage;
import model.ReceiveMessage;
import model.ScannerData;

public class RangeScan {

	private int sx, sy, ex, ey, resolution, mode;
	private int cx, cy; //currently wished servo position
	private int ox, oy; //old servor position
	private ScannerData sd;
	private SerialIO sio;
	
	private boolean abortWished = false;
	private boolean isTerminated;
	
	public final int TIMEOUT_MILLIS = 5000;
	private Timer cancelRangeScanTimer;
	
	ReceiveMessage timeoutListener;
	
	public RangeScan(int sx, int sy, int ex, int ey, int resolution, int mode, ScannerData sd, SerialIO sio, ReceiveMessage tl) {
		this.sx = sx;
		this.sy = sy;
		this.ex = ex;
		this.ey = ey;
		this.resolution = resolution;
		this.mode = mode;
		this.sd = sd;
		this.sio = sio;
		this.timeoutListener = tl;
		cx = sx;
		cy = sy;
		ox = -1;
		oy = -1;
		//clear reciving buffer
		while (sio.getRxFifoSize() > 0) {
			sio.getRxFifo();
		}
		cancelRangeScanTimer = new Timer();
		timerRenew();
		//start first data gathering
		updateServoPosition();
		sendThisDataRequest();	
	}

	private boolean lightGot = false;
	private boolean requestGot = false;
	
	public void gotData(String answer) {
		if (isTerminated) {//ignore data after we have finished
			System.out.println("RangeScan.gotData: Warning: Got data after being terminated.");
			return;
		}
		if (answer.startsWith("G:"+Main.SCAN_MODE_LIGHT)) {
			lightGot = true;
		}
		if (answer.startsWith("G:"+mode)) {
			requestGot = true;
		}
		if (answer.startsWith("E:")) {
			new ShowMessage("The 3D Scanner send an error as answer'"+answer+"'. Aborting range scan.");
			timerCancel();
			isTerminated = true;
			return;
		}
		sd.setData(cx, cy, resolution, answer);
		//System.out.println("gotData: answer: "+ answer);
		if ((lightGot) && (requestGot)) {
			if (!abortWished) {
				isTerminated = sendNextDataRequest();
				if (isTerminated) {
					timerCancel();
				} else
					timerRenew();
			} else {
				timerCancel();
				isTerminated = true;
			}
			lightGot = requestGot = false;
		}
	}
	
	/**
	 * Sets the new servo position and requests the values on this one.
	 * @return true: the range scan has finished. no new data are expected to come in.
	 */
	private boolean sendNextDataRequest() {
		boolean terminate = moveNextPosition();
		if (!terminate) {
			sendThisDataRequest();	
		}
		return terminate;
	}

	private void sendThisDataRequest() {
		sio.sendCommand("G04");
		if (mode == 1)
			sio.sendCommand("G01");
		if (mode == 2)
			sio.sendCommand("G02");
		if (mode == 3)
			sio.sendCommand("G03");
	}
	
	/**
	 * Moves the servos to the next valid position
	 * @return true: If the next calculated position is invalid. otherwise false.
	 */
	private boolean moveNextPosition() {
		int xdirection, ydirection;
		int maxx, minx, maxy, miny;
		//set up some variables
		if (ex < sx) {
			xdirection = -1;
		} else
			xdirection = 1;
		if (ey < sy) {
			ydirection = -1;
		} else
			ydirection = 1;
		maxx = Math.max(sx, ex);
		maxy = Math.max(sy, ey);
		minx = Math.min(sx, ex);
		miny = Math.min(sy, ey);
		//update y position
		cy += resolution*ydirection;
		if ((cy > maxy) || (cy < miny)) {
			cy = sy;
			cx += resolution*xdirection;
			if ((cx > maxx) || (cx < minx)) {
				return true;
			}
		}
		updateServoPosition();
		return false;
	}
		
	private void updateServoPosition() {
		if (ox != cx) {
			sio.sendCommand("H"+Tools.intToHex(cx, 2));
			//System.out.println("H:"+cx);
			ox = cx;
		}
		if (oy != cy) {
			sio.sendCommand("V"+Tools.intToHex(cy, 2));
			oy = cy;
			//System.out.println("V:"+cy);
		}
	}

	public boolean isTerminated() {
		return isTerminated;
	}
	
	public void requestAbort() {
		abortWished = true;
	}
	
	
	//for aborting scans if no data are coming in
	
	private void timerCancel() {
		try {
			cancelRangeScanTimer.cancel();
			cancelRangeScanTimer.purge();
		} catch (IllegalStateException e) {
			System.out.println("RangeScan.timerCancel: Warning: Timer was already canceled");
			//not expected to happen, but can be ignored
		}
	}
	
	private void timerRenew() {
		cancelRangeScanTimer.cancel();
		cancelRangeScanTimer.purge();
	  TimerTask abortRangeScanTimer = new TimerTask() {
	    public void run() {
	    	EventQueue.invokeLater(abortRangeScanRunnable);
	    	//System.out.println("timer called");
	    }
	  };
	  cancelRangeScanTimer = new Timer();
    cancelRangeScanTimer.schedule(abortRangeScanTimer, TIMEOUT_MILLIS);
  	//System.out.println("timer setup");	
	}
  
  final Runnable abortRangeScanRunnable = new Runnable() {
    public void run() {
    	isTerminated = true;
    	timeoutListener.messageTell("rangeScanTimeout");
			new ShowMessage("Error: Aborting scan because the scanner did returned a valid answer within "+TIMEOUT_MILLIS+"ms.");
    }
  };

}
