package control;

/* 3D Scanner Control Version 1.0 (c) 2009 by Malte Marwedel

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

import java.io.IOException;

import view.ChooseFile;
import view.ExportWindow;
import view.RootWindow;
import view.ShowMessage;
import model.ReceiveMessage;
import model.RxListener;
import model.ScannerData;

public class Main implements ReceiveMessage, RxListener {

	public static final int SCAN_MODE_OFF = 0;
	public static final int SCAN_MODE_DIRECT = 1;
	public static final int SCAN_MODE_LINEAR = 2;
	public static final int SCAN_MODE_VECTOR = 3;
	public static final int SCAN_MODE_LIGHT = 4;
	
	
	private static Main m;
	
	public static final String info = "3D Scanner Control version 1.0 (c) 2009 by Malte Marwedel"+Tools.newline+Tools.newline+
  "This program is free software; you can redistribute it and/or modify"+Tools.newline+
  "it under the terms of the GNU General Public License as published by"+Tools.newline+
  "the Free Software Foundation; either version 2 of the License, or"+Tools.newline+
  "(at your option) any later version."+Tools.newline+Tools.newline+
  "This program is distributed in the hope that it will be useful,"+Tools.newline+
  "but WITHOUT ANY WARRANTY; without even the implied warranty of"+Tools.newline+
  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"+Tools.newline+
  "GNU General Public License for more details."+Tools.newline+Tools.newline+
  "You should have received a copy of the GNU General Public License"+Tools.newline+
  "along with this program; if not, write to the Free Software"+Tools.newline+
  "Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA";
	
	
	private ScannerData sd;
	private RootWindow rw;
	private ExportWindow ew;
	private SerialIO sio;
	
	private RangeScan rs;
	
	/**
	 * @param args filename for opening a binary data object
	 */
	public static void main(String[] args) {
		System.out.println(info);
		m = new Main();
		if (args.length > 0) {
			m.loadFile(args[0]);
		}
	}
	
	private Main() {
		sd = new ScannerData();
		sio = SerialIO.getInstance();
		sio.setRxListener(this);
		rw = new RootWindow(this);
    try {
    	rw.setAvariablePorts(sio.getAvariableSerialPorts());
    } catch (java.lang.UnsatisfiedLinkError e) {
    	new ShowMessage("Error: I/O Library is not correctly installed. Continuing without support for a serial port. \n\r"+
          "For i386 Linux: Copy librxtxSerial.so to (JRE installation dir)/jre/lib/i386/\n\r"+
          "For other operating systems (Windows, Mac, Solaris) and architectures please see INSTALL file of RXTX lib.");    	
    } catch (java.lang.NoClassDefFoundError e) {
    	new ShowMessage("Error: Missing classes (jar file) for the serial communication.\n\r"+
			"Continuing without support for a serial port.");
    }
	}

	public ScannerData getScannerData() {
		return sd;
	}

  private void handlePortopenclose() {
  	if (sio.getPortState() == SerialIO.PORT_CLOSED) {
  		//now open it
  		sio.portOpen(rw.getSelectedPort());
  	} else if (sio.getPortState() == SerialIO.PORT_OPEN) {
  		sio.portClose();
  	}
  	//re-check current state (since opening or closing may have failed) for update in button description
  	switch(sio.getPortState()) {
  		case SerialIO.PORT_CLOSED: rw.portClosed(); break;
  		case SerialIO.PORT_OPEN: rw.portOpened(); break;
  	}	
  }
	
	public void messageTell(String str) {
		if (rw == null) {
			return; //may happen during the program start
		}
		if (str.equals("appexit")) {
			sd.appExit();
    	if (sio.getPortState() == SerialIO.PORT_OPEN) {
    		sio.portClose(); //now close it
    	}
		} else if (str.equals("portopenclose")) {
    	handlePortopenclose();
		} else if (str.equals("linear")) {
			sio.sendCommand("G02");
		} else if (str.equals("direct")) {
			sio.sendCommand("G01");
		} else if (str.equals("vector")) {
			sio.sendCommand("G03");
		} else if (str.equals("light")) {
			sio.sendCommand("G04");
		} else if (str.equals("horizontal")) {
			int value = rw.getHorizontal();
			String cmd = "H"+Tools.intToHex(value, 2);
			sio.sendCommand(cmd);
		} else if (str.equals("vertical")) {
			int value = rw.getVertical();
			String cmd = "V"+Tools.intToHex(value, 2);
			sio.sendCommand(cmd);
		} else if (str.equals("linearscan")) {
			setupRangeScan(2);
		} else if (str.equals("vectorscan")) {
			setupRangeScan(3);
		} else if (str.equals("save")) {
			saveFile();
		} else if (str.equals("load")) {
			loadFile();
		} else if (str.equals("abortrangescan")) {
			abortRangeScan();
		} else if (str.equals("savePNG")) {
			savePngFile();
		} else if (str.equals("rangeScanTimeout")) {
			abortRangeScan();
		} else if (str.equals("showExport")) {
			if (ew == null) {
				ew = new ExportWindow(this);
			} else
				ew.showWindow();
		} else if (str.equals("povPreview")) {
			povPreview();
		} else if (str.equals("povExport")) {
			povExport();
		}
	}
	
	private void povExport() {
    ChooseFile cf = new ChooseFile(".pov", ChooseFile.SAVE, ConfigFile.getInstance().getSettings("exportDir") , "");
    if (cf.getFile() != null) {
    	ConfigFile.getInstance().setSettings("exportDir", cf.getDirectory());
    	PovRayExport.exportFile(cf.getFilename(), sd, ew.getMinDist(), ew.getMaxDist(), ew.getObjectType(), ew.getTextureType());
    }
	}
	
	private void povPreview() {
		if (PovRayExport.exportFile("3d-preview.pov", sd, ew.getMinDist(), ew.getMaxDist(), ew.getObjectType(), ew.getTextureType())) {
			String cmdarray[] = new String[4];
			if (System.getProperty("os.name").startsWith("Windows")) { //Windows
				cmdarray[0] = "povray.exe";
			} else //Unix
				cmdarray[0] = "povray";
			cmdarray[1] = "-W800";
			cmdarray[2] = "-H600";
			cmdarray[3] = "3d-preview.pov";
			try {
				Process p = Runtime.getRuntime().exec(cmdarray);
				new OutputForwarder(p);
			} catch (IOException e) {
				new ShowMessage("Error: calling '"+cmdarray[0]+"' failed. Is the program installed and in your PATH?");
			e.printStackTrace();
			}
		}
	}
	
	private void abortRangeScan() {
		if (rs != null) {
			rs.requestAbort();
			if (rs.isTerminated()) {
				rw.changeAreaScan(Main.SCAN_MODE_OFF);
			}
		}
	}
	
	private void setupRangeScan(int mode) {
		int sx = Integer.parseInt(rw.getSxText());
		int sy = Integer.parseInt(rw.getSyText());
		int ex = Integer.parseInt(rw.getExText());
		int ey = Integer.parseInt(rw.getEyText());
		int resolution = rw.getResolution();
		if ((sx >= 0) && (sx < 256) && (sy >= 0) && (sy < 256) && (ex >= 0) && (ex < 256) && 
				(ey >= 0) && (ey < 256)) {
			rw.changeAreaScan(mode);
			rs = new RangeScan(sx, sy, ex, ey, resolution, mode, sd, sio, this);
		} else
			new ShowMessage("Error: Beginning and ending points for scanning an area are out of range. They must be -1 < X < 256.");
	}
	
	private void savePngFile() {
		String fname = Tools.replaceFileEnding(sd.getFilename(), ".png");
    ChooseFile cf = new ChooseFile(".png", ChooseFile.SAVE, ConfigFile.getInstance().getSettings("fileDir"), fname);
    if (cf.getFile() != null) {
    	sd.savePNG(cf.getFile());
    }
	}
	
	private void loadFile() {
    ChooseFile cf = new ChooseFile(ScannerData.FILE_ENDING, ChooseFile.OPEN, ConfigFile.getInstance().getSettings("fileDir"), null);
		if (cf.getFilename() != null) {
			loadFile(cf.getFilename());
			ConfigFile.getInstance().setSettings("fileDir", cf.getDirectory());
		}
	}
	
	private void loadFile(String filename) {
		sd = new ScannerData(filename);
		rw.updateImage();
	}

	private void saveFile() {
    ChooseFile cf = new ChooseFile(ScannerData.FILE_ENDING, ChooseFile.SAVE, ConfigFile.getInstance().getSettings("fileDir"), sd.getFilename());
		if (cf.getFilename() != null) {
			sd.saveFile(cf.getFilename());
			ConfigFile.getInstance().setSettings("fileDir", cf.getDirectory());
		}
	}

	public void rxDataWaiting() {
		while (sio.getRxFifoSize() > 0) {
			String answer = sio.getRxFifo();
			//System.out.println("Main.rxDataWaiting: Got from the serial port: '"+answer+"'");
			if (rs == null) {
				rw.setAnswerText(answer);
			} else {
				rs.gotData(answer);
				if (rs.isTerminated()) {
					System.out.println("Main: Range Scan finished");
					rw.changeAreaScan(SCAN_MODE_OFF);
					rs = null;
				}
				rw.updateImage();
			}
		}
		
	}
	
}
